/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "ftiles.h"

#include "prop.h"
#include "creator.h"

#include "debug.h"

#include "config.inc"
#include <algorithm>


/*
 * FullTiles
 */
static void destroy_imp_(FullTiles::Imp source, int w, int h)
{
  if (source) {
    DBG_ASSERT(w>0 && h>0);
    for (int j=0; j<h; ++j) {
      if (source[j]) Free(source[j]);
    }
    Free(source);
  }
}

static void cleanup_imp_(FullTiles::Imp source, int w, int h)
{
  if (source) {
    for (int j=0; j<h; ++j) {
      for (int i=0; i<w; ++i) {
        if (source[j][i]) {
          delete source[j][i];
          source[j][i] = 0;
        }
      }
    }
  }
}

static FullTiles::Imp create_imp_(int w, int h)
{
  if (w>0 && h>0) {
    FullTiles::Imp ret;
    Calloc(ret, h);
    if (ret) {
      for (int j=0; j<h; ++j) {
        Calloc(ret[j], w);
        if (!ret[j]) {
          destroy_imp_(ret, w, h);
          ret=0;
          break;
        }
      }
    }
    return ret;
  }
  return 0;
}

static void copy_imp_(FullTiles::Imp src, FullTiles::Imp dst, int srcx, int srcy, int dstx, int dsty, int w, int h)
{
  DBG_ASSERT(dst);
  if (src) {
    for (int j=0; j<h; ++j) {
      for (int i=0; i<w; ++i) {
        dst[j-dsty][i-dstx] = src[j-srcy][i-srcx];
      }
    }
  }
}

static void dup_imp_(FullTiles::Imp src, FullTiles::Imp dst, int srcx, int srcy, int dstx, int dsty, int w, int h)
{
  DBG_ASSERT(dst);
  if (src) {
    DBG_ASSERT(w>0 && h>0);
    for (int j=0; j<h; ++j) {
      for (int i=0; i<w; ++i) {
        if (src[j-srcy][i-srcx]) {
          dst[j-dsty][i-dstx] = src[j-srcy][i-srcx]->Clone();
        }
      }
    }
  }
}

static FullTiles::Imp clone_imp_(FullTiles::Imp source, int w, int h)
{
  DBG_ASSERT(source);
  DBG_ASSERT(w>0 && h>0);
  FullTiles::Imp ret = create_imp_(w, h);
  dup_imp_(source, ret, 0, 0, 0, 0, w, h);
  return ret;
}


FullTiles::FullTiles(BaseCreator *c, int x, int y, int w, int h):
  Tiles(c, x, y, w, h), tiles_(create_imp_(w, h)), default_value_(c->Create())
{
}


FullTiles::FullTiles(BaseCreator *c):
  Tiles(c), tiles_(0), default_value_(c->Create())
{
}


FullTiles::FullTiles(const FullTiles &other):
  Tiles(other),
  tiles_(clone_imp_(other.tiles_, other.width(), other.height())),
  default_value_(other.default_value_->Clone())
{
}


FullTiles::~FullTiles()
{
  cleanup_imp_(tiles_, width(), height());
  destroy_imp_(tiles_, width(), height());
  delete default_value_;
}


FullTiles *FullTiles::clone() const
{
  return new FullTiles(*this);
}


FullTiles &FullTiles::operator=(const FullTiles &other)
{
  int old_w = width();
  int old_h = height();
  Tiles::operator=(other);
  Imp tmp = clone_imp_(other.tiles_, other.width(), other.height());
  std::swap(tiles_, tmp);
  cleanup_imp_(tiles_, width(), height());
  destroy_imp_(tmp, old_w, old_h);
  default_value_->CopyFrom(other.default_value_);
  return *this;
}


BaseProperty *FullTiles::get_(int i, int j) const
{
  return is_in(i, j) ? tiles_[j-beginj()][i-begini()] : 0;
}


void FullTiles::set_(int i, int j, BaseProperty *p)
{
  if (is_in(i, j)) {
    BaseProperty *tmp = get(i, j);
    tiles_[j-beginj()][i-begini()] = p;
    delete tmp;
  }
  else {
    /* Trying to set a tile outside the layer */
    int new_firsti = std::min(begini(), i);
    int new_firstj = std::min(beginj(), j);
    int new_lasti = std::max(endi(), i+1);
    int new_lastj = std::max(endj(), j+1);
    int old_width = width();
    int old_height = height();
    { /* Resize it */
      Imp tmp = create_imp_(new_lasti-new_firsti, new_lastj-new_firstj);
      copy_imp_(tiles_, tmp, begini(), beginj(), new_firsti, new_firstj, old_width, old_height);
      std::swap(tiles_, tmp);
      destroy_imp_(tmp, old_width, old_height);
    }
    set_(i, j, p);
  }
}


void FullTiles::copy_(int i, int j, const BaseProperty *p)
{
  if (get_(i, j)) {
    tiles_[j-beginj()][i-begini()]->CopyFrom(p);
  }
  else {
    set_(i, j, p->Clone());
  }
}


void FullTiles::clear(int i, int j)
{
  if (is_in(i, j)) {
    copy(i, j, default_value_);
  }
}


void FullTiles::move_(int dx, int dy)
{
}


void FullTiles::insert_col_(int x, int count)
{
  Imp tmp = create_imp_(width()+count, height());
  for (int j=beginj(); j<endj(); ++j) {
    int i=begini();
    for ( ; i<x; ++i) tmp[j-beginj()][i-begini()]=get(i, j);
    for ( ; i<x+count; ++i) tmp[j-beginj()][i-begini()]=default_value_->Clone();
    for ( ; i<endi()+count; ++i) tmp[j-beginj()][i-begini()]=get(i-count, j);
  }
  std::swap(tiles_, tmp);
  destroy_imp_(tmp, width(), height());
}


void FullTiles::remove_col_(int x, int count)
{
  Imp tmp = create_imp_(width()-count, height());
  for (int j=beginj(); j<endj(); ++j) {
    int i=begini();
    for ( ; i<x; ++i) tmp[j-beginj()][i-begini()]=get(i, j);
    for ( ; i<x+count; ++i) delete get(i, j);
    for ( ; i<endi(); ++i) tmp[j-beginj()][i-count-begini()]=get(i, j);
  }
  std::swap(tiles_, tmp);
  destroy_imp_(tmp, width(), height());
}


void FullTiles::insert_row_(int y, int count)
{
  Imp tmp = create_imp_(width(), height()+count);
  int j=beginj();
  for ( ; j<y; ++j)
    for (int i=begini(); i<endi(); ++i) tmp[j-beginj()][i-begini()]=get(i, j);
  for ( ; j<y+count; ++j)
    for (int i=begini(); i<endi(); ++i) tmp[j-beginj()][i-begini()]=default_value_->Clone();
  for ( ; j<endj()+count; ++j)
    for (int i=begini(); i<endi(); ++i) tmp[j-beginj()][i-begini()]=get(i, j-count);
  std::swap(tiles_, tmp);
  destroy_imp_(tmp, width(), height());
}


void FullTiles::remove_row_(int y, int count)
{
  
  Imp tmp = create_imp_(width(), height()-count);
  int j=beginj();
  for ( ; j<y; ++j)
    for (int i=begini(); i<endi(); ++i) tmp[j-beginj()][i-begini()]=get(i, j);
  for ( ; j<y+count; ++j)
    for (int i=begini(); i<endi(); ++i) delete get(i, j);
  for ( ; j<endj(); ++j)
    for (int i=begini(); i<endi(); ++i) tmp[j-count-beginj()][i-begini()]=get(i, j);
  std::swap(tiles_, tmp);
  destroy_imp_(tmp, width(), height());
}

