/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "entry.h"
/*------------------------------------------------------------------------------
  The entry dialog uses 2 DIALOG structures
  The first one (NAME_IDX):
    proc: d_text_proc
    d1: Unused
    d2: Unused
    dp: String (name)
    dp2:
    dp3: Entry*
  The second one (ENTRY_IDX):
    proc: static Entry::edit_proc (static member, because it is called as a C
          function), calls Entry::Proc, which calls Entry::proc, which is set
          by the caller of SetState
    d1: Set by the caller of SetState
    d2: Set by the caller of SetState
    dp: Set by the caller of SetState
    dp2: Set by the caller of SetState
    dp3: Set by the caller of SetState
------------------------------------------------------------------------------*/


#include "debug.h"
#include "utils.h"

#include "help.h"

#include <altheme.h>
#include <altheme/internal/theme.h>

#include <allegro.h>


#define NAME_IDX 0
#define ENTRY_IDX 1


static int do_nothing(int /*msg*/, DIALOG * /*d*/, int /*c*/)
{
  return D_O_K;
}


/* Call the member function Proc, because we need the object data */
int Entry::edit_proc(int msg, DIALOG *d, int c)
{
  if(msg==MSG_IDLE) return D_O_K;
  Entry *e = CAST_FROM_VOID_PTR(Entry*, d[NAME_IDX-ENTRY_IDX].dp3);
  DBG_ASSERT(e);
  return e->Proc(msg, d, c);
}


Entry::Entry(DIALOG *d, const char *help_id):
  dialogs_(d), proc_(0), help_id_(help_id), show_name_(true)
{
  /* The name: displays the title */
  dialogs_[NAME_IDX].proc = altheme_text_proc;
  dialogs_[NAME_IDX].dp3 = this;
  /* The data: calls the proc given to SetState */
  dialogs_[ENTRY_IDX].proc = edit_proc;
  SetDefaultState();
}


Entry::~Entry(void)
{
}


void Entry::ShowName(bool yesno)
{
  show_name_ = yesno;
}


/* Set the DIALOG fields of the ENTRY_IDX field */
void Entry::SetState(
  const char *name, int (*proc)(int, DIALOG*, int),
  int d1, int d2, void *dp, void *dp2, void *dp3, int flags
)
{
  DBG_ASSERT(dialogs_[NAME_IDX].dp3 == this);
  if (show_name_) {
    dialogs_[NAME_IDX].dp = CAST_TO_VOID_PTR(name);
  }
  else {
    dialogs_[NAME_IDX].dp = CAST_TO_VOID_PTR(EmptyString().string());
  }
  proc_ = proc;
  dialogs_[ENTRY_IDX].d1 = d1;
  dialogs_[ENTRY_IDX].d2 = d2;
  dialogs_[ENTRY_IDX].dp = dp;
  dialogs_[ENTRY_IDX].dp2 = dp2;
  dialogs_[ENTRY_IDX].flags = flags;
  dialogs_[ENTRY_IDX].flags |= D_DIRTY;
  dialogs_[ENTRY_IDX].dp3 = dp3;
}


void Entry::SetFlagsState(int flags)
{
  dialogs_[ENTRY_IDX].flags = flags;
}


void Entry::SetDefaultState(void)
{
  /* The global EmptyString may not be already initialized */
  SetState(EMPTY_STRING, do_nothing, 0, 0, 0, 0, 0, 0);
}


int Entry::GetD1State(void) const
{
  return dialogs_[ENTRY_IDX].d1;
}


int Entry::GetD2State(void) const
{
  return dialogs_[ENTRY_IDX].d2;
}


void *Entry::GetDPState(void) const
{
  return dialogs_[ENTRY_IDX].dp;
}


void *Entry::GetDP2State(void) const
{
  return dialogs_[ENTRY_IDX].dp2;
}


void *Entry::GetDP3State(void) const
{
  return dialogs_[ENTRY_IDX].dp3;
}


int Entry::GetFlagsState(void) const
{
  return dialogs_[ENTRY_IDX].flags;
}


void Entry::Clear(void)
{
  altheme_fill_frame(
    screen, dialogs_[NAME_IDX].x, dialogs_[NAME_IDX].y,
    dialogs_[NAME_IDX].x+dialogs_[NAME_IDX].w,
    dialogs_[NAME_IDX].y+dialogs_[NAME_IDX].h, dialogs_[NAME_IDX].bg
  );
  altheme_fill_frame(
    screen, dialogs_[ENTRY_IDX].x, dialogs_[ENTRY_IDX].y,
    dialogs_[ENTRY_IDX].x+dialogs_[ENTRY_IDX].w,
    dialogs_[ENTRY_IDX].y+dialogs_[ENTRY_IDX].h, dialogs_[ENTRY_IDX].bg
  );
}


void Entry::Redraw(void)
{
  DBG_ASSERT(dialogs_[NAME_IDX].dp3 == this);
  dialogs_[NAME_IDX].flags |= D_DIRTY;
  dialogs_[ENTRY_IDX].flags |= D_DIRTY;
}


/* Called by edit_proc, so we can access the Entry data
   Calls the actual Allegro proc */
int Entry::Proc(int msg, DIALOG *d, int c)
{
  int ret;

  if (help_id_) {
    msg = do_help(msg, help_id_);
  }

  if(msg==MSG_CHAR && ((c>>8)==KEY_ENTER || (c>>8)==KEY_ENTER_PAD)) {
    broadcast_dialog_message(MSG_APPLY, 0);
    return D_USED_CHAR;
  }
  /* Make sure the Entry DIALOGS are ok */
  DBG_ASSERT(dialogs_[NAME_IDX].dp3 == this);
  /* Make sure the given DIALOG is ok (note: access the DIALOG *before* the
     given one) */
  DBG_ASSERT(d[NAME_IDX-ENTRY_IDX].dp3 == this);
  DBG_ASSERT(d == &(dialogs_[ENTRY_IDX]));

  d->proc = proc_;
  ret = proc_(msg, d, c);
  d->proc = edit_proc;

  return ret;
}


void Entry::SetSize(int x, int y, int w, int h)
{
  if(w > h) {
    /* Put the name on the left */
    dialogs_[NAME_IDX].x = x;
    dialogs_[NAME_IDX].y = y;
    dialogs_[NAME_IDX].w = w/2;
    dialogs_[NAME_IDX].h = h;
    dialogs_[ENTRY_IDX].x = x+w/2;
    dialogs_[ENTRY_IDX].y = y;
    dialogs_[ENTRY_IDX].w = w/2;
    dialogs_[ENTRY_IDX].h = h;
  }
  else {
    /* Put the name above */
    dialogs_[NAME_IDX].x = x;
    dialogs_[NAME_IDX].y = y;
    dialogs_[NAME_IDX].w = w;
    dialogs_[NAME_IDX].h = text_height(font)+8;
    if (show_name_) {
      dialogs_[NAME_IDX].flags &= ~D_HIDDEN;
    }
    else {
      dialogs_[NAME_IDX].flags |= D_HIDDEN;
    }
    dialogs_[ENTRY_IDX].x = x;
    dialogs_[ENTRY_IDX].y = y+dialogs_[NAME_IDX].h;
    dialogs_[ENTRY_IDX].w = w;
    dialogs_[ENTRY_IDX].h = h - dialogs_[NAME_IDX].h;
  }
}

