/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__STRING_TYPE__
#define EME__STRING_TYPE__
/*----------------------------------------------------------------------------
  String Property

  - Types
  - Casts
  - Draw
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

class Entry;


struct String {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;
  typedef const char* Base;
  typedef int Param;


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Draw
     ---------------------------------*/
  static void Draw(const char *str, BITMAP *bmp, int x, int y, int w, int h);


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(const char *str): string_(0)
      {DBG_SET_PROP_TYPE("String"); Set(str);}
    ~Property(void);
    Property *Clone(void) const;

    /* Accessors
       --------------------*/
    void Set(const char *str);
    const char *Get(void) const
      {return string_;}

    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);
    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    char *string_;
  }; /* class String::Property */


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(StaticString *name, int max_len, const char *str);
    Creator *Clone(void) const;
    ~Creator(void);

    /* Properties
       --------------------*/
    Property *Create(const char *value) const
      { return new Property(value); }

    virtual void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
      float scale, int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;


    /* Entry handlers
       --------------------*/
    void UpdateEntry(Entry *e, const BaseProperty *p) const;
    void UpdateProperty(BaseProperty *p, const Entry *e) const;

  protected:
    Creator(StaticString *name, int max_len, const char *str, BaseProperty *p):
      BaseCreator(name, p), max_len_(max_len)
      {DBG_SET_CREATOR_TYPE("String"); Init(str);}
    Creator(const Creator *other):
      BaseCreator(other), max_len_(other->max_len_)
      {DBG_SET_CREATOR_TYPE("String"); Init(other->str_);}
    void Init(const char *str);

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<String>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<String>(GetVariableReference()); }

    int max_len_; /* In characters (not bytes) */
    char *str_; /* String for the entry */
  }; /* class String::Creator */


  /* Private
     ---------------------------------*/
private:
}; /* struct String */

#ifdef DEBUG
typedef String::Creator StringCreator;
typedef String::Property StringProperty;
#endif

#endif /* EME__STRING_TYPE__ */

