/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "commands.h"

#include "debug.h"
#include "utils.h"
#include "globals.h"
#include "gui.h"

#include "command.h"

#include <stdlib.h>


#define DEFAULT_MAX_UNDO_LEVELS 32

int Commands::global_max_undo_levels_ = DEFAULT_MAX_UNDO_LEVELS;

void Commands::ChangeMaxUndoLevels(int new_max)
{
  global_max_undo_levels_ = new_max;
}


static int far_past = -2;


Commands::Commands():
  max_undo_levels_(GetMaxUndoLevels()),
  last_executed_(-1), save_point_(-1)
{
  Calloc(cmds_, max_undo_levels_);
}


Commands::~Commands(void)
{
  Clear();
  Free(cmds_);
  LOGEXEC();
}


void Commands::ChangeMaxUndoLevels()
{
  if(GetMaxUndoLevels() == max_undo_levels_) {
    return;
  }
  Command **new_cmds;
  Calloc(new_cmds, GetMaxUndoLevels());
  if(!new_cmds) {
    return;
  }
  /* Throw away all undone commands (they can no more be redone) */
  for(int i=0; i<max_undo_levels_; i++) {
    delete cmds_[i];
  }
  Free(cmds_);
  cmds_ = new_cmds;
  max_undo_levels_ = GetMaxUndoLevels();
  if (!HasChanged()) {
    last_executed_ = -1;
    save_point_ = -1;
  }
  else {
    last_executed_ = -1;
    save_point_ = far_past;
  }
}


void Commands::Execute(Command *cmd)
{
  /* If the command does nothing (by both Execute and UnExecute) there is
     no need to execute it and keep it */
  if(cmd->IsEmpty()) {
    delete cmd;
    return;
  }

  /* If the command can be merged with the last one, merge it and execute it */
  if(last_executed_>=0 && cmds_[last_executed_]->Merge(cmd)) {
    cmd->Execute();
    GUI.RedrawMap();
    delete cmd;
    return;
  }

  /* If the command wasn't merged, add it to the undoable list and execute it */

  /* If there is still some rooms to save the command */
  if(last_executed_ < max_undo_levels_-1) {
    /* Throw away all undone commands (they can no more be redone) */
    for(int i=last_executed_+1; i<max_undo_levels_; i++) {
      delete cmds_[i];
      cmds_[i] = 0;
    }

    if (save_point_>last_executed_) save_point_ = far_past;

    /* Save the new command */
    cmds_[last_executed_+1] = cmd;
    last_executed_++;
  }

  /* If there is no more room to save the command */
  else {
    /* Throw away the oldest command and move back the other ones */
    delete cmds_[0];
    for(int i=0; i<max_undo_levels_-1; i++) {
      cmds_[i] = cmds_[i+1];
    }

    save_point_ --;
    if (save_point_<far_past) save_point_ = far_past;

    /* Save the new command */
    cmds_[max_undo_levels_-1] = cmd;
  }

  /* Execute the command */
  cmd->Execute();
  GUI.RedrawMap();
}


void Commands::Undo(void)
{
  if(last_executed_>=0 && cmds_[last_executed_]) {
    cmds_[last_executed_]->UnExecute();
    GUI.RedrawMap();
    last_executed_--;
  }
}


void Commands::Redo(void)
{
  if(last_executed_<max_undo_levels_-1 && cmds_[last_executed_+1]) {
    last_executed_++;
    cmds_[last_executed_]->Execute();
    GUI.RedrawMap();
  }
}


void Commands::Clear(void)
{
  for(int i=0; i<max_undo_levels_; i++) {
    delete cmds_[i];
    cmds_[i] = 0;
  }
  last_executed_ = -1;
  save_point_ = -1;
}

