/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cmdapply.h"

#include "debug.h"
#include "utils.h"

#include "selected.h"
#include "prop.h"
#include "creator.h"
#include "lutils.h"
#include "tiles.h"
#include "stiles.h"
#include "map.h"

#include <stdlib.h>

#include <typeinfo>


CommandApply::CommandApply(
  Map *map, int l, const SelectedTiles *sel_tiles,
  const BaseProperty *value, int id, int probability
):
  Command(), id_(id), probability_(probability)
{
  PROFILE_BEGIN(fill_create)
  DBG_ASSERT(sel_tiles);
  layer_ = map->GetLayer(l);
  DBG_ASSERT(layer_);
  PROFILE_BEGIN(fill1)
  if (!sel_tiles->Empty()) {
    /* Fill the selected tiles */
    tiles_ = sel_tiles->Clone();
    CHECK_POINTER(tiles_);
  }
  else {
    /* Fill all the tiles */
    tiles_ = LayerUtils::SelectAll(layer_);
    CHECK_POINTER(tiles_);
  }
  PROFILE_END(fill1)

  PROFILE_BEGIN(fill2)
  saved_tiles_ = LayerUtils::Copy(layer_, tiles_);
  CHECK_POINTER(saved_tiles_);
  PROFILE_END(fill2)

  PROFILE_BEGIN(fill3)
  saved_empty_tiles_ = LayerUtils::SelectEmpty(layer_, tiles_);
  CHECK_POINTER(saved_empty_tiles_);
  PROFILE_END(fill3)

  PROFILE_BEGIN(fill4)
  tile_value_ = value->Clone();
  PROFILE_END(fill4)
  PROFILE_END(fill_create)
}


CommandApply::CommandApply(
  Map *map, int l, int x, int y, const BaseProperty *value, int id
):
  Command(), id_(id), probability_(FILL_PROBA_MAX)
{
  PROFILE_BEGIN(apply_begin)
  tiles_ = new SelectedTiles();
  CHECK_POINTER(tiles_);
  tiles_->Add(x, y);

  layer_ = map->GetLayer(l);
  DBG_ASSERT(layer_);
  saved_tiles_ = LayerUtils::Copy(layer_, tiles_);
  CHECK_POINTER(saved_tiles_);

  saved_empty_tiles_ = LayerUtils::SelectEmpty(layer_, tiles_);
  CHECK_POINTER(saved_empty_tiles_);

  tile_value_ = value->Clone();
  PROFILE_END(apply_begin)
}


CommandApply::~CommandApply(void)
{
  delete tiles_;
  delete tile_value_;
  delete saved_tiles_;
  delete saved_empty_tiles_;
}


void CommandApply::Execute(void)
{
  if(probability_ == FILL_PROBA_MAX) {
    PROFILE_BEGIN(apply_execute)
    LayerUtils::Fill(layer_, tiles_, tile_value_);
    PROFILE_END(apply_execute)
  }
  else if(probability_ != 0) {
    PROFILE_BEGIN(fill_execute)
    for (SelectedTiles::iterator i=tiles_->Begin(); i!=tiles_->End(); ++i) {
      if(rand()%(FILL_PROBA_MAX) < probability_) {
        LayerUtils::Fill(layer_, tiles_->GetX(i), tiles_->GetY(i), tile_value_);
      }
    }
    PROFILE_END(fill_execute)
  }
}


void CommandApply::UnExecute(void)
{
  LayerUtils::Paste(layer_, saved_tiles_->begini(), saved_tiles_->beginj(), saved_tiles_);
  if (!saved_empty_tiles_->Empty()) {
    LayerUtils::Cut(layer_, saved_empty_tiles_);
  }
}


int CommandApply::Merge(Command *cmd)
{
  if(typeid(*this) != typeid(*cmd)) {
    /* Can't merge */
    return 0;
  }
  CommandApply *cmd_apply = DYNAMIC_CAST(CommandApply*, cmd);
  if(id_!=cmd_apply->id_ || id_==MAX_MERGE_ID || cmd_apply->id_==MAX_MERGE_ID) {
    /* Don't merge when it is not a draw, but a true apply or when there has
       been a mouse release */
    return 0;
  }
  if(
    probability_ == cmd_apply->probability_ &&
    layer_ == cmd_apply->layer_ &&
    tile_value_->IsEqualTo(cmd_apply->tile_value_)
  ) {
    tiles_->Add(cmd_apply->tiles_);
    /* Should not merge other things than draw point */
    DBG_ASSERT(
      cmd_apply->saved_tiles_->count()==1 ||
      cmd_apply->saved_empty_tiles_->Num()==1
    );
    if (cmd_apply->saved_tiles_->count()>0) {
      SparseTiles::const_iterator first = cmd_apply->saved_tiles_->begin();
      BaseProperty *p = cmd_apply->saved_tiles_->get_tile(first)->Clone();
      CHECK_POINTER(p);
      int i = cmd_apply->saved_tiles_->get_i(first);
      int j = cmd_apply->saved_tiles_->get_j(first);
      DBG_ASSERT(!saved_tiles_->get(i, j));
      saved_tiles_->set(i, j, p);
    }
    saved_empty_tiles_->Add(cmd_apply->saved_empty_tiles_);
    /* Merged */
    return 1;
  }
  else {
    return 0;
  }
}


int CommandApply::IsEmpty(void) const
{
  if(probability_==0) return 1;

  if (tiles_->Empty()) return 1;

  if (!saved_empty_tiles_->Empty()) return 0;

  SparseTiles::const_iterator i;
  for (i=saved_tiles_->begin(); i!=saved_tiles_->end(); ++i) {
    if(! tile_value_->IsEqualTo(saved_tiles_->get_tile(i))) {
      return 0;
    }
  }
  return 1;
}

