/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__LIST_TYPE__
#define EME__LIST_TYPE__
/*----------------------------------------------------------------------------
  List Property

  - Types
  - Casts
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

class Entry;
struct DIALOG;


struct List {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;
  typedef int Base;
  typedef char const*const* Param;


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(char const*const* items, int value);
    ~Property(void);
    Property *Clone(void) const;

    /* Accessors
       --------------------*/
    void Set(int idx)
      {value_ = idx;}
    int Get(void) const
      {return value_;}

    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);
    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    int value_;

    /* To be able to type the list properties */
    char const*const* items_; /* Does not own it */
    char const*const* GetItems(void) const
      { return items_; }
    void SetItems(char const*const* items)
      { items_ = items; }
    friend class List::Creator;
  }; /* class List::Property */


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(
      StaticString *name,
      const char *const*strings, /* NULL terminated array of strings */
      int default_value
    );
    Creator *Clone(void) const;
    ~Creator(void);

    /* Properties
       --------------------*/
    Property *Create(int idx) const
      { return new Property(items_, idx); }

    int IsSameType(const BaseProperty *p) const;
    virtual void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
      float scale, int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;

    /* Entry handlers
       --------------------*/
    void UpdateEntry(Entry *e, const BaseProperty *p) const;
    void UpdateProperty(BaseProperty *p, const Entry *e) const;

    /* Misc
       --------------------*/
    int NumItems(void) const
      {return num_items_;}
    const char *Item(int i) const
      {return items_[i];}

  protected:
    Creator(StaticString *name, const char *const* items, BaseProperty *p):
      BaseCreator(name, p)
      {DBG_SET_CREATOR_TYPE("List"); Init(items);}
    Creator(const Creator *other):
      BaseCreator(other)
      {DBG_SET_CREATOR_TYPE("List"); Init(other->items_);}
    void Init(const char *const*string);

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<List>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<List>(GetVariableReference()); }

    int num_items_;
    char **items_;  /* Null terminated array */
  }; /* class List::Creator */


  /* Private
     ---------------------------------*/
private:
  static int display_list_callback(
    BITMAP *dest, int index, int x, int y, int w, DIALOG *d
  );

}; /* struct List */

#ifdef DEBUG
typedef List::Creator ListCreator;
typedef List::Property ListProperty;
#endif

#endif /* EME__LIST_TYPE__ */

