/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__BOOLEAN_TYPE__
#define EME__BOOLEAN_TYPE__
/*----------------------------------------------------------------------------
  Boolean Property

  - Types
  - Casts
  - Draw
  - Property
  - Creator
  - Private
----------------------------------------------------------------------------*/

#include "debug.h"
#include "utils.h"

#include "creator.h"
#include "prop.h"

#include "template.h"

class Entry;


struct Boolean {
  /* Types
     ---------------------------------*/
  class Creator;
  class Property;
  typedef int Base;
  typedef void Param;


#ifdef EME__COMPATIBILITY
  /* Casts
     ---------------------------------*/
  static inline Property *Cast(BaseProperty *p)
    { return DYNAMIC_CAST(Property*, p); }
  static inline const Property *Cast(const BaseProperty *p)
    { return DYNAMIC_CAST(const Property*, p); }

  static inline Creator *Cast(BaseCreator *c)
    { return DYNAMIC_CAST(Creator*, c); }
  static inline const Creator *Cast(const BaseCreator *c)
    { return DYNAMIC_CAST(const Creator*, c); }
#endif


  /* Draw
     ---------------------------------*/
  static void Draw(bool yesno, BITMAP *bmp, int x, int y, int w, int h);


  /* Property
     ---------------------------------*/
  friend class Property: public BaseProperty {

  public:
    /* Constructor/destructor
       --------------------*/
    Property(int value): BaseProperty(), is_set_(value)
      {DBG_SET_PROP_TYPE("Boolean");}
    ~Property(void);
    Property *Clone(void) const;

    /* Accessors
       --------------------*/
    inline void Set(int value)
      {if(value) Set(); else Reset();}
    void Set(void)
      {is_set_=true;}
    void Reset(void)
      {is_set_=false;}

    inline int Get(void) const
      {return is_set_;}

    /* Using the property value
       --------------------*/
    void CopyFrom(const BaseProperty *p);

    int IsEqualTo(const BaseProperty *p) const;

  private:
    Property(const Property&);
    Property &operator=(const Property&);

    int is_set_;
  }; /* class Boolean::Property */


  /* Creator
     ---------------------------------*/
  friend class Creator: public BaseCreator {

  public:
    /* Constructor/destructor
       --------------------*/
    Creator(StaticString *name, int yesno);
    ~Creator(void);
    Creator *Clone(void) const;

    /* Properties
       --------------------*/
    Property *Create(int value) const
      { return new Property(value); }

    virtual void Draw(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
      float scale, int l, int i, int j
    ) const;
    virtual int PrintValue(
      const BaseProperty *p, BITMAP *bmp, int x, int y, int color
    ) const;

    /* Entry handlers
       --------------------*/
    void UpdateEntry(Entry *e, const BaseProperty *p) const;
    void UpdateProperty(BaseProperty *p, const Entry *e) const;

  protected:
    Creator(StaticString *name, BaseProperty *p):
      BaseCreator(name, p)
      {DBG_SET_CREATOR_TYPE("Boolean");}
    Creator(const Creator *other): BaseCreator(other)
      {DBG_SET_CREATOR_TYPE("Boolean");}

  private:
    Creator(const Creator&);
    Creator &operator=(const Creator&);

    inline const Property *ReferenceProperty(void) const
      { return ::Cast<Boolean>(GetReference()); }
    inline Property *ReferenceProperty(void)
      { return ::Cast<Boolean>(GetVariableReference()); }
  }; /* class Boolean::Creator */
  

  /* Private
     ---------------------------------*/
private:
}; /* struct Boolean */

#ifdef DEBUG
typedef Boolean::Creator BooleanCreator;
typedef Boolean::Property BooleanProperty;
#endif

#endif /* EME__BOOLEAN_TYPE__ */

