/* seme - a simple map editor based on eme
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef SEME__MAP__
#define SEME__MAP__
/*------------------------------------------------------------------------------
  SemeMap

- Constructor/destructor
- Loading/saving format file
- Creating/loading/saving map
- Saving C code
- Editing map properties
- Editing layers
- Map and tiles shape and alii
- Compression
- Internal functions and data
------------------------------------------------------------------------------*/

#include "debug.h"


#include <stdio.h> /* For FILE */

#include "mapdlg.h" /* For MAP_SHAPE, TILE_SHAPE, ODD */

class SemeProperty;
class Wrapper;
class Brush;


#define LOSE_MAP() \
  (alert(Translation("Warning !"), 0, \
         Translation("Current map will be lost"), \
         Translation("Proceed anyway"), Translation("Cancel"), 0, 0) == 1)



class SemeMap {

public:
  /* Constructor/destructor
     ---------------------------------*/
  SemeMap(void);
  SemeMap(const SemeMap *map);
  ~SemeMap(void);

  int IsDirty(void)
    {return dirty;}


  /* Loading/saving format file
     ---------------------------------*/
  /* 'fname' encoding is U_CURRENT */
  int LoadFormat(const char *fname);
  int SaveFormat(const char *fname);

  void SetFormatFname(const char *fname);
  const char *GetFormatFname(void) const;
  /* Must not keep a pointer to the returned string */
  const char *GetAsciiFormatFname(void) const;


  /* Creating/loading/saving map
     ---------------------------------*/
  int NewMap(void);
  int LoadMap(const char *fname);
  int SaveMap(const char *fname, const Wrapper &wrapper);


  /* Loading/saving brush
     ---------------------------------*/
  Brush *LoadBrush(const char *fname);
  int SaveBrush(const char *fname, const Brush *brush);


  /* Saving C code
     ---------------------------------*/
  int SaveCCode(const char *fname);
  int SaveCppCode(const char *fname);


  /* Editing map properties
     ---------------------------------*/
  int GetPropertyCount(void)
    {return props_count;}

  void AddProperty(SemeProperty *p);

  SemeProperty *RemoveProperty(int index);

  SemeProperty *ReplaceProperty(int index, SemeProperty *p);

  SemeProperty *GetProperty(int index);


  /* Editing layers
     ---------------------------------*/
  int GetLayerCount(void)
    {return flayers_count;}

  void AddLayer(SemeProperty *l, int is_full);

  SemeProperty *RemoveLayer(int index);

  SemeProperty *ReplaceLayer(int index, SemeProperty *l, int is_full);

  SemeProperty *GetLayer(int index);

  int IsLayerFull(int index);


  /* Compression
     ---------------------------------*/
  void Compress(int yesno)
    {compress_map = yesno;}

  int IsCompressed(void) const
    {return compress_map;}


  /* Map and tiles shape and alii
     ---------------------------------*/
  void SetMapShape(MAP_SHAPE shape)
    {map_shape = shape;}

  void SetTileShape(TILE_SHAPE shape, int w, int h)
    {tile_shape = shape; tile_w = w; tile_h = h;}

  void SetTileOffset(int dx, int dy, ODD odd)
    {tile_odd = odd; tile_dx = dx; tile_dy = dy;}


  MAP_SHAPE GetMapShape(void) const
    {return map_shape;}

  TILE_SHAPE GetTileShape(void) const
    {return tile_shape;}

  int GetTileW(void) const
    {return tile_w;}

  int GetTileH(void) const
    {return tile_h;}

  int GetTileDX(void) const
    {return tile_dx;}

  int GetTileDY(void) const
    {return tile_dy;}

  ODD GetTileOdd(void) const
    {return tile_odd;}


protected:
  /* Internal functions and data
     ---------------------------------*/
  int Proceed(void);

  int SaveCHeader(
    const char *hfile, const char *base_upper, const char *base_lower
  );
  int SaveCCleanupFunc(
    FILE *file, const char *base_upper, const char *base_lower
  );
  int SaveCTestFunc(
    FILE *file, const char *base_upper, const char *base_lower,
    const char *cleanup
  );
  int SaveCLoadFunc(
    FILE *file, const char *base_upper, const char *base_lower,
    const char *cleanup
  );
  int SaveCDestroyFunc(
    FILE *file, const char *base_upper, const char *base_lower
  );
  int SaveCFile(
    const char *cfile,
    const char *base, const char *base_upper, const char *base_lower
  );


private:
  SemeMap(const SemeMap&);
  SemeMap &operator=(const SemeMap&);

  MAP_SHAPE map_shape;
  TILE_SHAPE tile_shape;
  int tile_w, tile_h;
  ODD tile_odd;
  int tile_dx, tile_dy;
  int props_count;
  SemeProperty **props;
  int flayers_count;
  SemeProperty **flayers;

  char *format_fname;

  int use_const;
  int compress_map;

  int dirty;
};

#endif /* SEME__MAP__ */

