


# Version string needs a specific handling because its quotes are eaten
# somewhere between make and bash.
# Note that user code should use EME_VERSION_TEXT, *not* EME_VERSION_STR
cflags="$cflags -DEME_VERSION_TEXT=\"$eme_version\""

# In some pathological cases (the plugin has no reference to libeme), the
# linker doesn't link with libeme (because the link order is plugin, libeme,
# allegro).
# Force a libeme symbol to be handled as an undefined symbol.
# Doing this should always trigger linking of libeme.a.
ldflags="-Wl,--undefined=al_clean_get_executable_name"

usage()
{
  exit_code=$1

  cat <<EOF
emetool is a tool to compile eme plugins.

Usage:
	emetool --help | -h
This help.

	emetool --version
Print eme version.

	emetool --options
Print the options used when eme was compiled.

	emetool --remake
Remake eme with the same options. Must be called in the eme directory
($eme_dir).

	emetool --cflags | --cppflags
Print compilation flags for plugins on standard output.

	emetool --ldflags
Print link flags for plugins on standard output.

	emetool --libs
Print libraries for plugins on standard output.

	emetool --compile OBJECT SOURCE
Compile SOURCE file into OBJECT file.
SOURCE file must be a C or C++ file.

	emetool --link EXECUTABLE OBJECTS...
Link OBJECTS files to create EXECUTABLE.

	emetool --build EXECUTABLE SOURCES...
Compile SOURCES files to create EXECUTABLE.
SOURCES files must be C or C++ files.

	emetool --makefile EXECUTABLE SOURCES...
Prints a makefile for compiling SOURCES into EXECUTABLE on the standard output.
SOURCES files must be C files ending with .c, or C++ files ending with .cc, .C or .cpp.
The makefile has two targets: all (compiles SOURCES into EXECUTABLE) and clean
(removes EXECUTABLE and object files)

See also the documentation: $eme_dir/doc/emetool.htm
EOF
  exit $exit_code
}


print_and_execute()
{
  echo $1
  return `$1`
}


handle_error()
{
  echo "$1  Try emetool --help"
  exit $2
}


the_options=""
get_options()
{
  the_options=""
  test "$system" == "" || the_options="$the_options SYSTEM=$system"
  test "$compatibility" == "$default_compatibility" || the_options="$the_options COMPATIBILITY=$compatibility"
  test "$debug" == "$default_debug" || the_options="$the_options DEBUG=$debug"
  test "$profile" == "$default_profile" || the_options="$the_options PROFILE=$profile"
  test "$unicode" == "$default_unicode" || the_options="$the_options UNICODE=$unicode"
  test "$memory_pool" == "$default_memory_pool" || the_options="$the_options MEMORY_POOL=$memory_pool"
  test "$override_libs" == "" || the_options="$the_options OVERRIDE_LIBS=$override_libs"
  test "$override_includes" == "" || the_options="$the_options OVERRIDE_INCLUDES=$override_includes"
  test "$test_flags" == "" || the_options="$the_options TESTFLAGS=$test_flags"
}


compile()
{
  object=$1
  source=$2

  test "$object" == "" && handle_error "No object file specified." 2
  test "$source" == "" && handle_error "No source file specified." 2
  test "$3" != "" && handle_error "Too many source files specified." 2

  print_and_execute "$cc $cflags -o $object -c $source"
}


link()
{
  executable=$1
  shift # Other parameters: object files

  test "$executable" == "" && handle_error "No executable file specified." 2
  test "$1" == "" && handle_error "No object files specified." 2

  print_and_execute "$cc $cflags -o $executable $@ $libs $ldflags"
}


build()
{
  executable=$1
  shift # Other parameters: source files

  test "$executable" == "" && handle_error "No executable file specified." 2
  test "$1" == "" && handle_error "No source files specified." 2

  print_and_execute "$cc $cflags -o $executable $@ $libs $ldflags"
}


makefile()
{
  executable=$1
  shift # Other parameters: source files

  test "$executable" == "" && handle_error "No executable file specified." 2
  test "$1" == "" && handle_error "No source files specified." 2

  cat <<EOF
#
# Autogenerated makefile, do not hand edit
#
# make
# make all
#    Compile and build the executable
#
# make clean
#    Remove object files and executable
#

RM=$remove

CC=$cc

CFLAGS+=$cflags

LDFLAGS+=$ldflags

LIBS+=$libs

EXECUTABLE=$executable

# Replace .c .cc .C and .cpp by .o
OBJS=\$(patsubst %.c,%.o, \$(patsubst %.cc,%.o, \$(patsubst %.C,%.o, \$(patsubst %.cpp,%.o, $*))))


all: \$(EXECUTABLE)


# Link
\$(EXECUTABLE): \$(OBJS)
	\$(CC) \$(CPPFLAGS) \$(LDFLAGS) \$(CFLAGS) -o \$@ \$^ \$(LIBS)


# Object files are created by implicit rules
%.o: %.c
	\$(CC) \$(CPPFLAGS) \$(CFLAGS) -o \$@ -c \$<

%.o: %.cc
	\$(CC) \$(CPPFLAGS) \$(CFLAGS) -o \$@ -c \$<

%.o: %.C
	\$(CC) \$(CPPFLAGS) \$(CFLAGS) -o \$@ -c \$<

%.o: %.cpp
	\$(CC) \$(CPPFLAGS) \$(CFLAGS) -o \$@ -c \$<


# Remove object files and executables
mostlyclean:
	-\$(RM) \$(OBJS)

clean: mostlyclean
	-\$(RM) \$(EXECUTABLE)

.PHONY: all clean mostlyclean

EOF
  return 0
}


if test $# -eq 0; then
  usage 1
fi


case $1 in
  --help | -h)
    usage 0
  ;;

  --version)
    echo $eme_version
  ;;

  --options)
    get_options
    echo "$the_options"
  ;;

  --remake)
    # FIXME: check the current directory is actually eme directory
    get_options
    # It seems that because the_options contains ``='',
    # print_and_execute can't work
    echo "$make $the_options"
    $make $the_options
  ;;

  --cflags | --cppflags)
    echo "$cflags"
  ;;

  --ldflags)
    echo "$ldflags"
  ;;

  --libs)
    echo "$libs"
  ;;

  --compile)
    shift
    compile $@
  ;;

  --link)
    shift
    link $@
  ;;

  --build)
    shift
    build $@
  ;;

  --makefile)
    shift
    makefile $@
  ;;

  *)
    usage 1
  ;;
esac

# vim: noet syntax=sh
