/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "xmlparser.h"

#include "structures.h"
#include "sparklet_utils.h"

#include "xmlserver.h"
#include "exceptions.h"

#define EMPTY_STR std::string("")


using namespace std;

extern XMLServer *gXMLSrv;


XMLParser::XMLParser(const string &XMLFile) {
	XMLFilePath = FixFilenameSlashes(XMLFile);

	try {
		Document = gXMLSrv->GetDocument(XMLFilePath);
	}
	catch (const ExSparkletException& ese) {
		_ERROR_;
		_SAY(ese.GetMsg());
		throw;
	}
}


string XMLParser::GetName(const TiXmlElement *Element) {
	return Element->Value();
}


const TiXmlElementList* XMLParser::GetAllChildren(const TiXmlElement *Element, const std::string &Name) {
	_SPARKLET_ASSERT(Element);

	TiXmlElementList* list = new TiXmlElementList();

	if (Name != "*") {
		for (const TiXmlElement *child = Element->FirstChildElement(Name.c_str()); child; child = child->NextSiblingElement(Name.c_str())) {
			list->push_back(child);
		}
	}
	else {
		for (const TiXmlElement *child = Element->FirstChildElement(); child; child = child->NextSiblingElement()) {
			list->push_back(child);
		}
	}

	return list;
}


const TiXmlElementList* XMLParser::GetChildren(const TiXmlElement *Element, const std::string &Name) {
	_SPARKLET_ASSERT(Element);

	TiXmlElementList* list = new TiXmlElementList();

	if (Name != "*") {
		for (const TiXmlElement *child = Element->FirstChildElement(Name.c_str()); child; child = child->NextSiblingElement(Name.c_str())) {
			list->push_back(child);
		}
	}
	else {
		for (const TiXmlElement *child = Element->FirstChildElement(); child; child = child->NextSiblingElement()) {
			list->push_back(child);
		}
	}

	if (list->size() == 0) {
		_ERROR_;
		_SAY("There should be at least one '" + Name + "' element in '" + Element->Value() + "'!");
	}

	return list;
}


const TiXmlElement* XMLParser::GetChild(const TiXmlElement *Element, const std::string &Name) {
	_SPARKLET_ASSERT(Element);

	const TiXmlElement* Child;

	if (Name != "*")
		Child = Element->FirstChildElement(Name.c_str());
	else
		Child = Element->FirstChildElement();

	if (!Child) {
		_ERROR_;
		_SAY("There should be exactly one '" + Name + "' element in '" + Element->Value() + "'!");
	}

	return Child;
}

string XMLParser::GetValueS(const TiXmlElement *Att, const std::string &What, const string Default) {
	const char* Val = Att->Attribute(What.c_str());

	if (!Val)
		return Default;

	return Val;
}


int XMLParser::GetValueL(const TiXmlElement *Att, const std::string &What, int Default) {
	const char* Val = Att->Attribute(What.c_str());

	if (Val)
		return strtol(Val, NULL, 0);
	else
		return Default;
}


unsigned long XMLParser::GetValueUL(const TiXmlElement *Att, const std::string &What, unsigned long Default) {
	const char* Val = Att->Attribute(What.c_str());

	if (Val)
		return strtoul(Val, NULL, 0);
	else
		return Default;
}


float XMLParser::GetValueF(const TiXmlElement *Att, const string &What, const float Default) {
	const char* Val = Att->Attribute(What.c_str());

	if (Val)
#ifdef WIN32
		return (float)strtod(Val, NULL);
#else
		return strtof(Val, NULL);
#endif
	else
		return Default;
}
