/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "sprite.h"
#include "weapon.h"

#include "statusbar.h"

#include "bullet.h"
#include "bomb.h"
#include "missile.h"
#include "weapon_pack.h"
#include "sound.h"
#include "network_server.h"
#include "sparklet_utils.h"


#define FIRE_MODE_STR std::string("fire_mode")

#define SOUND_STR std::string("sound")

#define ARM_STR std::string("arm")

#define PERIOD_STR std::string("period")
#define IMPULSE_STR std::string("impulse")

#define FIRE_POINT_RADIUS_STR std::string("fire_point_radius")
#define FIRE_POINT_ANGLE_STR std::string("fire_point_angle")

#define FUZZYNES_STR std::string("fuzzynes")

#define AMMO_STR std::string("ammo")
#define AMMO_LIMIT_STR std::string("ammo_limit")

#define WEAPON_PACKET_STR std::string("weapon_packet")

#define WEAPON_STATUS_STR std::string("weapon_status")
#define INDEX_STR std::string("index")


using namespace std;

extern NetworkServer *gNetworkServer;


Weapon::Weapon(const TiXmlElement *Element, const Rect<float> &lOwnerGeo, const IDENTITY &lOwnerIdentity, Statusbar *S, bool SrvMode) :
	ComplexObjectExtension(Element, lOwnerGeo, SrvMode),
	OwnerGeo(lOwnerGeo),
	OwnerIdentity(lOwnerIdentity),
	Parser(DATA_PREFIX + XMLParser::GetValueS(Element, SOURCE_STR, "no weapon source given")),
	Offset(XMLParser::GetValueL(Element, X_STR, 0),
	XMLParser::GetValueL(Element, Y_STR, 0)) {

	InitialAmmo = XMLParser::GetValueL(Element, AMMO_STR, 0);
	Ammo = 0;

	Stb = S;
	WS = NULL;

	const TiXmlElement *Root = Parser.GetRoot();

	if (SrvMode)
		ReadEndpoints(Root, lOwnerGeo);

	Index = XMLParser::GetValueL(Root, INDEX_STR, 0);
	AmmoLimit = XMLParser::GetValueL(Root, AMMO_LIMIT_STR, 0);

	WeaponPacketElement = XMLParser::GetChild(Root, WEAPON_PACKET_STR);

	const  TiXmlElementList* Child = Parser.GetChildren(Root, FIRE_MODE_STR);
	FireMode.reserve(Child->size());
	for (USHORT x = 0; x < Child->size(); ++x) {
		const TiXmlElement *One = Child->at(x);
		FM Mode;

		Mode.Att = One;
		Mode.Impulse = XMLParser::GetValueF(One, IMPULSE_STR, 10.f);
		Mode.Fuzzynes = XMLParser::GetValueF(One, FUZZYNES_STR, 0.f);
		Mode.Period = XMLParser::GetValueF(One, PERIOD_STR, 0.1f);

		if (ServerMode)
			Mode.Sample = new Sound(XMLParser::GetChild(One, SOUND_STR), lOwnerGeo);

		FireMode.push_back(Mode);
	}
	delete Child;

	Child = XMLParser::GetChildren(Root, ARM_STR);
	ArmList.reserve(Child->size());
	for (USHORT x = 0; x < Child->size(); ++x) {
		const TiXmlElement *One = Child->at(x);
		Arm New;

		New.Firearm = new Sprite(One, OwnerGeo, ServerMode);
		New.Firearm->SetLoc(New.Firearm->GetLoc() + Offset);
		SpriteLayer.push_back(New.Firearm);

		New.FirePointRadius = XMLParser::GetValueF(One, FIRE_POINT_RADIUS_STR, 0.f);
		New.FirePointAngle = XMLParser::GetValueF(One, FIRE_POINT_ANGLE_STR, 0.f);

		ArmList.push_back(New);
	}
	delete Child;

#ifndef SERVER_ONLY
	if (Stb) {
			WS = Stb->GetWeaponStatus(Root);
			WS->SetAmmoLimit(AmmoLimit);
	}
#endif //SERVER_ONLY

	for (USHORT x = 0; x < ArmList.size(); ++x)
		ArmList[x].Firearm->Stop();

	WStatus = HIDDEN;
}


Weapon::~Weapon() {
	if (ServerMode) {
		for (USHORT y = 0; y < FireMode.size(); ++y)
			delete FireMode[y].Sample;
	}
}


void Weapon::Reset() {
	for (USHORT y = 0; y < FireMode.size(); ++y)
		FireMode[y].TimeRemaning = FireMode[y].Period;

	Hide();

	Ammo = 0;
	BarStatus = WEAPON_INACTIVE;
}


void Weapon::DecraseRemaningTime(float TimeElapsed) {
	for (USHORT y = 0; y < FireMode.size(); ++y) {
		if (FireMode[y].TimeRemaning > 0) {
			FireMode[y].TimeRemaning -= TimeElapsed;
		}
		if (FireMode[y].TimeRemaning < 0)
			FireMode[y].TimeRemaning = 0;
	}
}


void Weapon::StopWeaponAnim() {
	if (WStatus == HIDING) {
		bool Inactive = true;
		for (USHORT x = 0; x < ArmList.size(); ++x)
			if (ArmList[x].Firearm->IsOver()) {
				ArmList[x].Firearm->Stop();
				Inactive &= true;
			}
			else
				Inactive &= false;

		if (Inactive) {
			WStatus = HIDDEN;
			if (BarStatus != WEAPON_INACTIVE) {
				BarStatus = WEAPON_GATHERED;
			}
		}
	}

	if (WStatus == SHOWING) {
		bool Active = true;
		for (USHORT x = 0; x < ArmList.size(); ++x)
			if (ArmList[x].Firearm->IsOver()) {
				ArmList[x].Firearm->Pause();
				Active &= true;
			}
			else
				Active &= false;

		if (Active)
			WStatus = SHOWN;
	}
}


bool Weapon::NeedsUpdate(float TimeElapsed) {
#ifndef SERVER_ONLY
	if (WS) {
		WS->SetStatus(BarStatus);
		WS->SetAmmoCount(Ammo);
	}
#endif //SERVER_ONLY

	DecraseRemaningTime(TimeElapsed);
	StopWeaponAnim();

	return ComplexObjectExtension::NeedsUpdate(TimeElapsed);
}


Vector<float> Weapon::KeyDown(FIRE_TYPE FT, float Direction, Vector<float> Speed) {
	//return if there is no such fire mode
	if ((USHORT)FT >= FireMode.size())
		return Vector<float>(0, 0);

	//return if the weapon is not ready yet (still in the 'SHOWING' stage or not even that)
	if (WStatus != SHOWN)
		return Vector<float>(0, 0);

	Vector<float> LoopBack(0, 0);

	//check if it's time to shoot
	if (FireMode[FT].TimeRemaning == 0) {
		//fire every firearm
		for (USHORT x = 0; x < ArmList.size(); ++x) {
			//break if there is no ammo left
			if (Ammo <= 0)
				break;
			LoopBack += Fire(ArmList[x], FireMode[FT], Direction, Speed);
			Ammo--;
			//reset arm timer
			FireMode[FT].TimeRemaning = FireMode[FT].Period;
		}
	}

	return LoopBack;
}


Vector<float> Weapon::Fire(const Arm &Gun, const FM &Mode, float Direction, Vector<float> Velocity) {
	if (WStatus != SHOWN)
		return Vector<float>(0, 0);

	Projectile *NewOne = NULL;
	if (XMLParser::GetValueS(Mode.Att, "projectile", "boo") == BULLET_STR)
		NewOne = new Bullet(Mode.Att);
	else if (XMLParser::GetValueS(Mode.Att, "projectile", "boo") == BOMB_STR)
		NewOne = new Bomb(Mode.Att);
	else if (XMLParser::GetValueS(Mode.Att, "projectile", "boo") == MISSILE_STR)
		NewOne = new Missile(Mode.Att, OwnerIdentity.ClientID);
	else {
		_ERROR_;
		_DISPLAY(XMLParser::GetName(Mode.Att));
	}

	float RndFuzzynes = 0;
	if (Mode.Fuzzynes)
		RndFuzzynes = (rand() % (int)(Mode.Fuzzynes * 100.f)) / 100.f - 0.25f;

	NewOne->SetLocC(polar(Gun.FirePointRadius, Direction + Gun.FirePointAngle) + GetCenter(OwnerGeo) + Offset);
	NewOne->SetImpulse(Vector<float>(polar(Mode.Impulse + RndFuzzynes * 1000.f, Direction + RndFuzzynes)));
	NewOne->AddVelocity(Velocity);
	NewOne->Identity = OwnerIdentity;

	gNetworkServer->AddSound(Mode.Sample, true);

	return Vector<float>(polar(Mode.Impulse, Direction));
}


Vector<float> Weapon::KeyUp(FIRE_TYPE FT, float, Vector<float>) {
	return Vector<float>(0, 0);
}


bool Weapon::Load(USHORT HowMany) {
	if (Ammo < AmmoLimit) {
		Ammo += HowMany;
		if (Ammo > AmmoLimit)
			Ammo = AmmoLimit;

		return true;
	}

	return false;
}


void Weapon::MakeGathered(int Init) {
	BarStatus = WEAPON_GATHERED;

	if (Init != -1)
		Load(Init);
	else
		Load(InitialAmmo);
}


void Weapon::Show() {
	if (WStatus == HIDDEN) {
		for (USHORT x = 0; x < ArmList.size(); ++x)
			ArmList[x].Firearm->PlayOnce();

		WStatus = SHOWING;
	}

	BarStatus = WEAPON_SELECTED;
}


void Weapon::Hide() {
	if (WStatus == SHOWN) {
		for (USHORT x = 0; x < ArmList.size(); ++x)
			ArmList[x].Firearm->PlayOnceReversed();

		WStatus = HIDING;
	}
}


void Weapon::LeavePacket(const Point<float>& Loc) const {
	if (WeaponPacketElement) {
		WeaponPack *WP = new WeaponPack(WeaponPacketElement, 1);
		WP->SetLoc(Loc);
		WP->SetUseOnceAndDestroy(true);
		WP->SetAmmoCount(Ammo);
	}
}


WeaponInfo* Weapon::GetWeaponInfo() const {
	WeaponInfo *RetVal = new WeaponInfo;

	RetVal->Ammo = Ammo;
	RetVal->BarStatus = BarStatus;
	RetVal->WStatus = WStatus;
	RetVal->Index = Index;

	return RetVal;
}


void Weapon::SetWeaponInfo(const WeaponInfo* WI) {
	_SPARKLET_ASSERT(WI->Index == Index);

	STATUS lStatus = static_cast<STATUS>(WI->WStatus);
	Ammo = WI->Ammo;
	BarStatus = static_cast<WEAPON_STATUS>(WI->BarStatus);

	if (lStatus == SHOWN) {
		for (USHORT x = 0; x < ArmList.size(); ++x) {
			ArmList[x].Firearm->PlayOnce();
			ArmList[x].Firearm->JumpToEnd();
			ArmList[x].Firearm->Pause();
		}
		WStatus = SHOWN;
	}
	else if (lStatus == HIDDEN) {
		for (USHORT x = 0; x < ArmList.size(); ++x) {
			ArmList[x].Firearm->JumpToStart();
			ArmList[x].Firearm->Stop();
		}
		WStatus = HIDDEN;
	}

	if (lStatus == SHOWING)
		Show();
	else if (lStatus == HIDING)
		Hide();
}
