/***************************************************************************
                          structures.h  -  description
                             -------------------
    begin                : Fri Feb 28 2003
    copyright            : (C) 2003 by Milan Mimica
    email                : milan.mimica@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef STRUCTURES_H
#define STRUCTURES_H


#include <cmath>


/*Define sparklet version string.*/
#ifdef SERVER_ONLY
	#define SPARKLET_VERSION_STR "0.9.9 (server-only version)"
#else
	#define SPARKLET_VERSION_STR "0.9.9"
#endif


/*Set directory where game data is kept.*/
#ifdef WIN32
	#define DATA_PREFIX "data\\"
#else
	#define DATA_PREFIX "data/"
#endif


/*Detecting appropriate slashes*/
#ifdef WIN32
	#define FS_SLASH '\\'
	/*This is only used by FixFileNameSlashes() to replace it with the right one.*/
	#define	WRONG_FS_SLASH '/'
#else
	#define FS_SLASH '/'
	/*This is only used by FixFileNameSlashes() to replace it with the right one.*/
	#define WRONG_FS_SLASH '\\'
#endif


/* Define a filename for a list of game servers that is downloaded from the internet. */
#define SERVER_LIST_FILE_NAME "data/srvlst.txt"


/* Returns a unique 'long' generated from 4 characters.*/
#define GEN_ID(a, b, c, d) (((a)<<24) | ((b)<<16) | ((c)<<8) | (d))


#include "logging_console.h"
extern LoggingConsole *gLC;


/*Don't remove! Won't compile otherwise.*/
#undef writeString
#undef readString


/*Whether to define debug marcos...*/
#ifdef DEBUG

	/*Debug to global log file*/

	#include <sstream>
	/*These marcos print message to a global logging console declared above.*/

	#define _CHECKPOINT_ do {                                             \
		std::ostringstream oss;                                           \
		oss << "Checkpoint " << __FILE__ << ":" << __LINE__ << std::endl; \
		gLC->write_string(oss.str().c_str());                             \
	} while (false)

	#define _ERROR_ do {                                               \
		std::ostringstream oss;                                        \
		oss << "Error " << __FILE__ << ":" << __LINE__ << std::endl;   \
		gLC->write_string(oss.str().c_str());                          \
	} while (false)

	#define _DISPLAY(val) do {                                                             \
		std::ostringstream oss;                                                            \
		oss << __FILE__ << ":" << __LINE__ << " " << #val << " = " << (val) << std::endl;  \
		gLC->write_string(oss.str().c_str());                                              \
	} while (false)

	#define _SAY(val) do {                                              \
		std::ostringstream oss;                                         \
		oss << __FILE__ << ":" << __LINE__ << " " << val << std::endl;  \
		gLC->write_string(oss.str().c_str());                           \
	} while (false)

	#define _SPARKLET_ASSERT(val) do {              \
		if (!(val)) {                               \
			_ERROR_;                                \
			_SAY(#val + std::string(" is false!")); \
		}                                           \
	} while (false)

#else /*DEBUG*/

	#define _CHECKPOINT_			do {} while (false)
	#define _ERROR_					do {} while (false)
	#define _DISPLAY(val)			do {} while (false)
	#define _SAY(val)				do {} while (false)
	#define _SPARKLET_ASSERT(val)	do {} while (false)

#endif /*DEBUG*/


#ifndef M_PI
	#define M_PI 3.14159265389f
	#define M_PI_2 1.57079632679f
#endif


/*Tag names often used in XML files*/
#define W_STR std::string("w")
#define H_STR std::string("h")
#define X_STR std::string("x")
#define Y_STR std::string("y")
#define SOURCE_STR std::string("source")


/*Returns true if given rectangles collide.*/
#define CheckRectCollision(x1,y1,w1,h1,x2,y2,w2,h2) (!( ((x1)>=(x2)+(w2)) || ((x2)>=(x1)+(w1)) || ((y1)>=(y2)+(h2)) || ((y2)>=(y1)+(h1)) ))


/*NetworkServer will not generate object's IDs less than this value.*/
#define RESERVED_OBJECT_UID_COUNT 100

/*NetowrkServer will not generate path codes less than this value.*/
#define RESERVED_PATH_CODE_COUNT 10

/*Codes for random object codes*/
#define RANDOM_YELLOW_PARTICLE 1
#define RANDOM_RED_PARTICLE 2
#define RANDOM_BLUE_PARTICLE 3

/*Used to make the code work on any(?) resolution. They represent the number of "game units" that
fit into screen width/height, notwithstanding current resolution.*/
#define UNITS_PER_SCREEN_W 1024
#define UNITS_PER_SCREEN_H 768


/*Maximum ratio between camera width/height and UNITS_PER_SCREEN_* when zooming in and zooming out.*/
#define ZOOM_IN_MAX_RATIO 0.6f
#define ZOOM_OUT_MAX_RATIO 1.6f


/*Minimum zoom percent change.*/
#define ZOOM_GRANULARITY_PERCENT 4.f


/*The range where the sound can be heard by someone else.*/
#define ACOUSTICAL_RANGE sqrtf(UNITS_PER_SCREEN_W * UNITS_PER_SCREEN_W + UNITS_PER_SCREEN_H * UNITS_PER_SCREEN_H) * 1.2f;

/*The distance at which the sound sample is fully panned to one speaker.*/
#define ACOUSTICAL_SWEEP_EXTREME UNITS_PER_SCREEN_W * 0.7f;


/*Factories use these to auto-register objects. They just work, ignore them.*/
#define DECLARE_FACTORY(X, Y)                                    \
struct X ## Factory : public Y ## Factory {                      \
        Y* operator()(const TiXmlElement* Element, float Scale) {\
                return new X(Element, Scale);                    \
        }                                                        \
};


#define DECLARE_CLI_FACTORY(X, Y)                                                        \
struct X ## Factory : public Y ## Factory {                                              \
        Y* operator()(const TiXmlElement* Element, const Size &Dim, float Scale) {       \
                return new X(Element, Dim, Scale);                                       \
        }                                                                                \
};


/*Some often used types.*/
typedef unsigned short int USHORT;
typedef unsigned long int ULONG;
typedef unsigned char UCHAR;


typedef unsigned char TEAM;
#define TEAM_BLUE 0
#define TEAM_RED 1
#define TEAM_YELLOW 2
#define TEAM_GREEN 3
#define LAST_TEAM TEAM_GREEN
#define TEAM_NO_TEAM 255


struct Size {
	USHORT Width;
	USHORT Height;

	Size() {};
	Size(USHORT W, USHORT H) {
		Width = W;
		Height = H;
	}
};


inline bool operator==(const Size &rhs, const Size &lhs) {
	return (lhs.Width == rhs.Width && lhs.Height == rhs.Height);
}


inline bool operator!=(const Size &rhs, const Size &lhs) {
	return (lhs.Width != rhs.Width || lhs.Height != rhs.Height);
}


template <typename Type>
struct Vector {
	Type x;
	Type y;

	Vector() {};
	Vector(Type X, Type Y) :
		x(X),
		y(Y) {}

	Vector operator+=(const Vector &rhs) {
		x += rhs.x;
		y += rhs.y;
		return *this;
	}

	Vector operator-=(const Vector &rhs) {
		x -= rhs.x;
		y -= rhs.y;
		return *this;
	}

	Vector operator*=(const Type &rhs) {
		x *= rhs;
		y *= rhs;
		return *this;
	}

	Vector operator/=(const Type &rhs) {
		x /= rhs;
		y /= rhs;
		return *this;
	}
};

template <typename Type>
inline Vector<Type> operator+(const Vector<Type> &lhs, const Vector<Type> &rhs) {
	return Vector<Type>(lhs.x + rhs.x, lhs.y + rhs.y);
}

template <typename Type>
inline Vector<Type> operator-(const Vector<Type> &lhs, const Vector<Type> &rhs) {
	return Vector<Type>(lhs.x - rhs.x, lhs.y - rhs.y);
}

template <typename Type>
inline Vector<Type> operator*(const Type &lhs, const Vector<Type> &rhs) {
    return Vector<Type>(rhs.x * lhs, rhs.y * lhs);
}

template <typename Type>
inline Vector<Type> operator*(const Vector<Type> &lhs, const Type &rhs) {
    return Vector<Type>(rhs * lhs.x, rhs * lhs.y);
}

template <typename Type>
inline Vector<Type> operator/(const Vector<Type> &lhs, const Type &rhs) {
	return Vector<Type>(lhs.x / rhs, lhs.y / rhs);
}

template <typename Type>
inline bool operator==(const Vector<Type> &rhs, const Vector<Type> &lhs) {
	return (lhs.x == rhs.x && lhs.y == rhs.y);
}

template <typename Type>
inline bool operator!=(const Vector<Type> &rhs, const Vector<Type> &lhs) {
	return (lhs.x != rhs.x || lhs.y != rhs.y);
}

template <typename Type>
inline float abs(const Vector<Type> &P) {
	return sqrt(float(P.x * P.x + P.y * P.y));
}

template <typename Type>
inline Type arg(const Vector<Type> &P) {
	return atan2(P.y, P.x);
}

template <typename Type>
inline Vector<Type> polar(const Type length, const Type angle) {
	return Vector<Type>(length * cos(angle), length * sin(angle));
}

template <typename Type>
inline Vector<Type> normalize(const Vector<Type> &vec) {
	return Vector<Type>(vec / abs(vec));
}

template <typename Type>
inline Type dot_product(const Vector<Type> &lhs, const Vector<Type> &rhs) {
	return abs(lhs) * abs(rhs) * cos(arg(lhs) - (arg(rhs)));
}

template <typename Type>
std::ostream& operator<<(std::ostream &os, const Vector<Type> &v) {
	os << "("<< v.x <<", " << v.y << ")";
	return os;
}


/*Vector and points can often be represented with the same structure.*/
#define Point Vector


template <typename Type>
struct Rect : public Point<Type>, public Size {
	Rect() {};
	Rect(Type NewX, Type NewY, USHORT NewW, USHORT NewH) {
		this->x = NewX;
		this->y = NewY;
		this->Width = NewW;
		this->Height = NewH;
	}

	Rect(const Point<Type> &P, const Size &S) {
		this->x = P.x;
		this->y = P.y;
		this->Width = S.Width;
		this->Height = S.Height;
	}

	Rect(const Rect<Type> &R) : Point<Type>(R), Size(R) {
	}

	Rect& operator=(const Size &Value) {
		this->Width = Value.Width;
		this->Height = Value.Height;
		return *this;
	}

	Rect& operator=(const Point<Type> &Value) {
		this->x = Value.x;
		this->y = Value.y;
		return *this;
	}
};


#endif
