/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "sparklet_utils.h"

#include <cstdio>
#include <algorithm>


float BoundAngle(float Angle) {
	Angle = fmod(Angle,  static_cast<float>(2.f * M_PI));

	if (Angle < -M_PI)
		Angle += 2.f * M_PI;

	if (Angle > M_PI)
		Angle -= 2.f * M_PI;

	return Angle;
}


Point<float> GetCenter(const Rect<float> &Geo) {
	return Point<float>(Geo.x + Geo.Width / 2.f, Geo.y + Geo.Height / 2.f);
}


#ifdef WIN32
//round float to the nearest integer
int lroundf(const float v) {
	const float f = floor(v);
	const float c = ceil(v);
	if (v >= 0) {
		if ((c - v) < (v - f)) // distance to ceil smaller than distance to floor
			return (int)c;
		else
			return (int)f;
	}
	else {
		if ((c - v) < (v - f)) // distance to ceil smaller than distance to floor
			return (int)f; // inverted
		else
			return (int)c; // inverted
	}
}
#endif


Point<float> GetCameraTargetSmart(const Point<float> &TargetObjectLoc, const Size &CameraDim, const Size &ArenaSize) {
	const USHORT HalfCameraW = CameraDim.Width / 2;
	const USHORT HalfCameraH = CameraDim.Height / 2;

	if (TargetObjectLoc.x > HalfCameraW) {
		if (TargetObjectLoc.x < ArenaSize.Width - HalfCameraW) {
			if (TargetObjectLoc.y > HalfCameraH) {
				if (TargetObjectLoc.y < ArenaSize.Height - HalfCameraH) {
					return TargetObjectLoc;
				}
				else {
					return Point<float>(TargetObjectLoc.x, (float)(ArenaSize.Height - HalfCameraH));
				}
			}
			else {
				return Point<float>(TargetObjectLoc.x, HalfCameraH);
			}
		}
		else {
			if (TargetObjectLoc.y > HalfCameraH) {
				if (TargetObjectLoc.y < ArenaSize.Height - HalfCameraH) {
					return Point<float>((float)(ArenaSize.Width - HalfCameraW), TargetObjectLoc.y);
				}
				else {
					return Point<float>((float)(ArenaSize.Width - HalfCameraW),
						(float)(ArenaSize.Height - HalfCameraH));
				}
			}
			else {
				return Point<float>((float)(ArenaSize.Width - HalfCameraW), HalfCameraH);
			}
		}
	}
	else {
		if (TargetObjectLoc.y > HalfCameraH) {
			if (TargetObjectLoc.y < ArenaSize.Height - HalfCameraH) {
				return Point<float>((float)HalfCameraW, TargetObjectLoc.y);
			}
			else {
				return Point<float>((float)HalfCameraW, (float)(ArenaSize.Height - HalfCameraH));
			}
		}
		else {
			return Point<float>((float)HalfCameraW, (float)HalfCameraH);
		}
	}
}


Point<float> CalcCameraLoc(Point<float> OldCamera, const Point<float> &TargetLoc) {
	//spacial case, initial value
	if (OldCamera.x == -1.f)
		return TargetLoc;

	/*Perform smoothing when changing old camera values to the target one.*/

	const Point<float> Distance = TargetLoc - OldCamera;

	//don't smooth if it's too far, just jump
	if (abs(Distance) < 400)
		OldCamera += Distance / 1.15f; //1.15 is a good value
	else
		OldCamera += Distance;

	return OldCamera;
}


Size CalcCameraDim(Size OldCamera, const Size &TargetDim) {
	OldCamera.Width += (USHORT)((TargetDim.Width - OldCamera.Width) / 10.f);
	OldCamera.Height += (USHORT)((TargetDim.Height - OldCamera.Height) / 10.f);

	return OldCamera;
}


std::string FixFilenameSlashes(const std::string &S) {
	std::string RetVal(S);

	for (USHORT x = 0; x < RetVal.length(); ++x) {
		if (RetVal[x] == WRONG_FS_SLASH)
			RetVal[x] = FS_SLASH;
	}

	return RetVal;
}


int GetTeamTintColor(const TEAM &Team) {
	switch (Team) {
		case TEAM_BLUE:
			return 0xffff6666;
		case TEAM_RED:
			return 0xff6666ff;
		case TEAM_GREEN:
			return 0xff66ff66;
		case TEAM_YELLOW:
			return 0xff00ffff;
	}

	_ERROR_;
	return 0;
}


int GetTeamColor(const TEAM &Team) {
	switch (Team) {
		case TEAM_BLUE:
			return 0xaaff0000;
		case TEAM_RED:
			return 0xaa0000ff;
		case TEAM_GREEN:
			return 0xaa00ff00;
		case TEAM_YELLOW:
			return 0xaa00ffff;
	}

	_ERROR_;
	return 0;
}


std::string GetTeamName(const TEAM &Team) {
	switch (Team) {
		case TEAM_BLUE:
			return "BLUE";
		case TEAM_RED:
			return "RED";
		case TEAM_GREEN:
			return "GREEN";
		case TEAM_YELLOW:
			return "YELLOW";
	}

	_ERROR_;
	return 0;
}


TEAM GetTeamFromName(const std::string &Team) {
	if (CaseInsensitiveMatch(Team, "blue"))
		return TEAM_BLUE;
	if (CaseInsensitiveMatch(Team, "red"))
		return TEAM_RED;
	if (CaseInsensitiveMatch(Team, "green"))
		return TEAM_GREEN;
	if (CaseInsensitiveMatch(Team, "yellow"))
		return TEAM_YELLOW;

	return TEAM_NO_TEAM;
}


Size ParseDimensionString(const std::string &Dim) {
	Size RetVal;
	sscanf(Dim.c_str(), "%hux%hu", &RetVal.Width, &RetVal.Height);
	return RetVal;
}


std::string FormatDimensionString(const Size &Dim) {
	char buff[32];
	sprintf(buff, "%hux%hu", Dim.Width, Dim.Height);
	return std::string(buff);
}


bool CaseInsensitiveMatch(std::string str1, std::string str2) {
	std::transform(str1.begin(), str1.end(), str1.begin(), (int(*)(int))tolower);
	std::transform(str2.begin(), str2.end(), str2.begin(), (int(*)(int))tolower);

	return str1 == str2;
}

