/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "bitmap_server.h"
#include "object_extension.h"
#include "rotated_object.h"

#include "sparklet_utils.h"


#define SHAPE_RES_ROT_STR std::string("shape_res_rot")
#define MOMENT_OF_INERTIA_STR std::string("moment_of_inertia")

#define ROTATION_STR std::string("rotation")
#define DIRECTION_STR std::string("direction")


//255 / 2PI
#define ANGLE_FACTOR 40.5845105f

/*Don't even rotate bitmap less that MIN_RELEVANT_ANGLE radians, no use if it. This is used only by
shape renderer routine.*/
#define MIN_RELEVANT_ANGLE 0.019f


using namespace std;

extern BitmapServer *gBmpSrv;


RotatedObject::RotatedObject(const TiXmlElement *Att, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	ShapeRendered(false),
	OldShapeDirection(0) {

	const TiXmlElement *Root = Parser->GetRoot();

	Direction = XMLParser::GetValueF(Root, DIRECTION_STR, 0.f);
	Rotation = XMLParser::GetValueF(Root, ROTATION_STR, 0.f);

	ShapeResRot = XMLParser::GetValueF(Root, SHAPE_RES_ROT_STR, 1.f) * Scale;
	MomentOfInertia = XMLParser::GetValueF(Root, MOMENT_OF_INERTIA_STR, 1.f) * Scale;

	Direction = XMLParser::GetValueF(Att, DIRECTION_STR, Direction);
	Rotation = XMLParser::GetValueF(Att, ROTATION_STR, Rotation);

	ShapeCache = create_bitmap_ex(16, Shape->w, Shape->h);
}


RotatedObject::~RotatedObject() {
	destroy_bitmap(ShapeCache);
}


void RotatedObject::OnUpdate() {
	Direction = BoundAngle(Direction);
	PhysicalObject::OnUpdate();
}


void RotatedObject::GetCompleteData(MessageCoder &MC) const {
	RotationInfo *RI = new RotationInfo();
	RI->Direction = Direction;
	RI->Rotation = Rotation;
	MC.Put(RI);

	PhysicalObject::GetCompleteData(MC);
}


void RotatedObject::CheckData() {
	if (OldRotation != Rotation) {
		DataChanged = true;
		OldRotation = Rotation;
	}
	else
		DataChanged = false;

	PhysicalObject::CheckData();
}


void RotatedObject::GetChangedData(MessageCoder &MC) const {
	if (DataChanged) {
		RotationInfo *RI = new RotationInfo();
		RI->Direction = Direction;
		RI->Rotation = Rotation;
		MC.Put(RI);
	}

	PhysicalObject::GetChangedData(MC);
}


bool RotatedObject::NextFrame(const float TimeElapsed) {
	if (PhysicalObject::NextFrame(TimeElapsed))
		return true;

	if (abs(OldShapeDirection - Direction) >= MIN_RELEVANT_ANGLE) {
		OldShapeDirection = Direction;
		ShapeDirectionChanged = true;

		return true;
	}

	return false;
}


bool RotatedObject::RenderShape() {
	if ((!ShapeChanged && !ShapeDirectionChanged) || (IsFixedShape && ShapeRendered))
		return false;

	const int Magenta = makecol16(255, 0, 255);

	if (ShapeChanged) {
		clear_to_color(tmp, Magenta);
		clear_to_color(ShapeCache, Magenta);

		for (USHORT x = 0; x < Extension.size(); ++x)
			Extension[x]->RenderShape(ShapeCache, tmp);
	}

	clear_to_color(Shape, Magenta);

	if (Direction)
		rotate_sprite(Shape, ShapeCache, 0, 0, ftofix(Direction * ANGLE_FACTOR));
	else
		draw_sprite(Shape, ShapeCache, 0, 0);

	ShapeChanged = false;
	ShapeDirectionChanged = false;
	ShapeRendered = true;

	return true;
}
