/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "sprite.h"
#include "weapon.h"
#include "engine1.h"
#include "engine2.h"
#include "spawn_place.h"
#include "wall.h"
#include "sound.h"

#include "game_logic.h"
#include "sparklet_utils.h"

#include "player_object.h"
#include "network_server.h"


#define SMOKE_SPRING_STR std::string("smoke_spring")
#define SMOKE_PARTICLE_STR std::string("smoke_particle")

#define EXPLOSION_STR std::string("explosion")
#define EXPLOSION_SOUND_STR std::string("explosion_sound")
#define ENGINE_CHASSIS_STR std::string("engine_chassis")
#define IS_GATHERED_STR std::string("is_gathered")

#define STATUSBAR_STR std::string("statusbar")


using namespace std;

extern float gTimeElapsed;
extern NetworkServer *gNetworkServer;
extern GameLogic *gGameLogic;
extern GameOptions gGameOpt;


PlayerObject::PlayerObject(const TiXmlElement* Att) :
	Object(Att, 1),
	MobileObject(Att, 1),
	PhysicalObject(Att, 1),
	RotatedObject(Att, 1),
	WeaponLoopBackImpulse(0, 0),
	FULL_ENERGY(800000),
	EXTRAFULL_ENERGY(FULL_ENERGY * 2),
	ZERO_ENERGY(0),
	Energy(FULL_ENERGY),
	EnergyLast(0),
	State(SPAWNING),
	EnergyPercent(100) {

	const TiXmlElement *Root = Parser->GetRoot();

	const TiXmlElement* Element = XMLParser::GetChild(Root, ENGINE_STR);
	const UCHAR EngineType = XMLParser::GetValueL(Element, ENGINE_TYPE_STR, 0);
	switch (EngineType) {
		case 1: Machinery = new Engine1(Element, Geo, true); break;
		case 2: Machinery = new Engine2(Element, Geo, true); break;
		default: _ERROR_;
	}
	Extension.push_back(Machinery);

	Element = Parser->GetChild(Root, EXPLOSION_STR);
	Explosion = new Sprite(Element, Geo, true);
	Explosion->Stop();
	Extension.push_back(Explosion);

	ExplosionSound = new Sound(XMLParser::GetChild(Root, EXPLOSION_SOUND_STR), Geo);

	Element = Parser->GetChild(Root, SMOKE_SPRING_STR);
	const TiXmlElementList* Spr = Parser->GetAllChildren(Element, SMOKE_PARTICLE_STR);
	for (USHORT x = 0; x < Spr->size(); x++) {
		SpringParticle *Particle = new SpringParticle(Spr->at(x), &Geo, &Direction);
		SmokeParticlesList.push_back(Particle);
	}
	delete Spr;

	Element = XMLParser::GetChild(Root, ENGINE_CHASSIS_STR);
	EngineChassis = new Sprite(Element, Geo, true);
	EngineChassis->Pause();
	Extension.push_back(EngineChassis);

	ArrangeWeapons(Root);

	Chassis = GetBaseSprite(Root);
	Chassis->Pause();

	OldEngInf = new EngineInfo;
	EngineInfo *TmpEI = Machinery->GetEngineInfo();
	OldEngInf->Count = TmpEI->Count;
	delete TmpEI;
	OldEngInf->Engine = new bool[OldEngInf->Count];
	fill(OldEngInf->Engine, OldEngInf->Engine + OldEngInf->Count, false);

	OldShipState.Energy = 0;

	ImpulseChanged = false;
	OldRotation = 0;
	OldAdditionalImpulse = Vector<float>(0, 0);
	JustTranslocated = false;

	gNetworkServer->Add(this, Object::ObjectUID);

	PreventMultipleCollide = true;
}


PlayerObject::~PlayerObject() {
	while(!SmokeParticlesList.empty()) {
		delete SmokeParticlesList.back();
		SmokeParticlesList.pop_back();
	}

	delete OldEngInf;
	delete ExplosionSound;

	for (std::map<const USHORT, WeaponInfo*>::iterator x = OldWeaponInfoList.begin(); x != OldWeaponInfoList.end(); ++x)
		delete x->second;
}


void PlayerObject::ArrangeWeapons(const TiXmlElement *Root) {
	const TiXmlElementList* Child = Parser->GetChildren(Root, WEAPON_STR);

	for (USHORT x = 0; x < Child->size(); ++x) {
		const TiXmlElement *Element = Child->at(x);

		Weapon *NewOne = new Weapon(Element, Geo, Identity, NULL, true);

		NewOne->Reset();
		Extension.push_back(NewOne);
		WeaponList.insert(make_pair(NewOne->GetIndex(), NewOne));

		WeaponInfo *WI = new WeaponInfo;
		WI->Ammo = 0;
		WI->WStatus = -1;
		WI->BarStatus = -1;
		WI->Index = 999;
		OldWeaponInfoList.insert(make_pair(NewOne->GetIndex(), WI));

		const bool IsGathered = (bool)XMLParser::GetValueL(Element, IS_GATHERED_STR, 0);
		if (IsGathered) {
			GatheredWeaponList.insert(make_pair(NewOne->GetIndex(), NewOne));
			InitialGatheredWeapons.insert(make_pair(NewOne->GetIndex(), NewOne));
		}
	}
	delete Child;

	if (InitialGatheredWeapons.size() == 0) {
		_ERROR_;
		_SAY("There should be at least one gathered weapon, sorry.");
	}
}


USHORT PlayerObject::GetBestWeaponIndex() {
	for (int x = 10; x >= 0; --x) {
		if (GatheredWeaponList.find(x) != GatheredWeaponList.end())
			if (!GatheredWeaponList[x]->IsEmpty())
				return x;
	}

	//all weapons out of ammo
	//need to think of something here
	//_ERROR_;
	return 0;
}


USHORT PlayerObject::GetNextGatheredWeaponIndex() {
	for (USHORT x = ActiveWeapon + 1; x < 10; ++x) {
		if (GatheredWeaponList.find(x) != GatheredWeaponList.end())
			if (!GatheredWeaponList[x]->IsEmpty())
				return x;
	}

	for (USHORT x = 0; x <= ActiveWeapon; ++x) {
		if (GatheredWeaponList.find(x) != GatheredWeaponList.end())
			if (!GatheredWeaponList[x]->IsEmpty())
				return x;
	}

	_ERROR_;
	return 0;
}


void PlayerObject::WeaponSwap() {
	if (ActiveWeapon != ActiveWeaponWaiting)
		GetActiveWeapon()->Hide();

	if (GetActiveWeapon()->IsEmpty()) {
		GetActiveWeapon()->Hide();
		SwitchToWeapon(GetBestWeaponIndex());
	}

	if (GetActiveWeapon()->IsHidden()) {
		ActiveWeapon = ActiveWeaponWaiting;
		GetActiveWeapon()->Show();
	}
}


bool PlayerObject::AddWeapon(const USHORT Index, const int InitAmmoCount) {
	if (WeaponList[Index]->IsGathered() == true || State != NORMAL)
		return false;

	GatheredWeaponList.insert(make_pair(Index, WeaponList[Index]));
	WeaponList[Index]->MakeGathered(InitAmmoCount);
	if (GetBestWeaponIndex() == Index)
		SwitchToWeapon(Index);

	return true;
}


bool PlayerObject::AddAmmo(const USHORT Index, USHORT AmmoCount) {
	if (State != NORMAL)
		return false;

	return WeaponList[Index]->Load(AmmoCount);
}


bool PlayerObject::AddEnergy(USHORT E) {
	if (Energy < FULL_ENERGY || State != NORMAL) {
		 Energy += E * FULL_ENERGY / 100;

		if (Energy > FULL_ENERGY)
			Energy = FULL_ENERGY;

		return true;
	}

	return false;
}


void PlayerObject::OnUpdate() {
	switch (State) {
		case NORMAL:
			UpdateEnergy();
			WeaponSwap();
		break;

		case EXPLODING:
			if (Explosion->IsOver()) {
				Explosion->Stop();
				Explosion->JumpToStart();

				SecondsToWait = 3;
				AfterWaitState = SPAWNING;
				State = WAITING;
			}
		break;

		case WAITING:
			if (SecondsToWait > 0)
				SecondsToWait -= gTimeElapsed;
			else
				State = AfterWaitState;
		break;

		case SPAWNING:
			SpawnPlace* SP = gGameLogic->GetFreeSpawnPlace();
			if (SP) {
				SP->Spawn(this);
				Reborn();
				State = NORMAL;
				JustSpawned = true;
			}
		break;
	}

	NextFrame(gTimeElapsed);
	UpdatePosition();
	SmokeSpring.Update(gTimeElapsed);

	RotatedObject::OnUpdate();
}


void PlayerObject::UpdatePosition() {
	Vector<float> MachineryForce = polar(Machinery->GetForward(), Direction);
	MachineryForce += polar(Machinery->GetStrafe(), Direction + (float)M_PI_2);

	AdditionalImpulse = WeaponLoopBackImpulse + CollidingImpulse + (MachineryForce * gTimeElapsed);
	if (OldAdditionalImpulse != AdditionalImpulse) {
		OldAdditionalImpulse = AdditionalImpulse;
		ImpulseChanged = true;
	}

	Impulse += AdditionalImpulse;
	Velocity += Impulse / Mass;
	Velocity -= CalcResistance(Velocity, ShapeRes * gNetworkServer->GetViscosity() / Mass * gTimeElapsed);

	WeaponLoopBackImpulse = Vector<float>(0, 0);

	float AngularMomentum = Machinery->GetTorque() * gTimeElapsed;
	Rotation += AngularMomentum / MomentOfInertia;
	Rotation -= CalcResistance(Rotation, ShapeResRot * gNetworkServer->GetViscosity() / MomentOfInertia * gTimeElapsed);

	//to achieve better gameplay do not rotate while there is some impulse
	if (abs(Impulse) && !AngularMomentum)
		Rotation = 0;

	if (abs(Impulse) && Rotation < 0 && !AngularMomentum)
		Rotation = 0;

	if (abs(Impulse) && Rotation > 0 && !AngularMomentum)
		Rotation = 0;

	Geo += Velocity * gTimeElapsed;
	Direction += Rotation * gTimeElapsed;
}


void PlayerObject::UpdateEnergy() {
	Energy -= static_cast<ULONG>(abs(CollidingImpulse / 25.f) + CollidingEnergy);

	if (Energy == ZERO_ENERGY || Energy > EXTRAFULL_ENERGY)
		Energy = 0;

	if (Energy != EnergyLast) {
		SetSmoke();
		SetChassis();
		EnergyPercent = (UCHAR)lroundf(Energy * 100.f / FULL_ENERGY);

		EnergyLast = Energy;
	}

	if (EnergyPercent == 0)
		Explode();
}


void PlayerObject::SetSmoke() {
	USHORT Count = 0;

	for (USHORT x = 0; x < SmokeParticlesList.size(); x++) {
		if (Energy < FULL_ENERGY / (x+4))
			++Count;
	}

	for (USHORT x = SmokeSpring.Size(); x < Count; ++x) {
		SmokeSpring.Add(SmokeParticlesList[x]);
	}

	if (Count < SmokeSpring.Size()) {
		const USHORT SmokeCount = SmokeSpring.Size();
		for (USHORT x = Count; x < SmokeCount; ++x)
			SmokeSpring.RemoveOne();
	}
}


void PlayerObject::SetChassis() {
	if (Energy == 0)
		Chassis->JumpToFrame(Chassis->GetFrameCount() - 1);
	else {
		const ULONG START_ENERGY = FULL_ENERGY / 3;
		const USHORT Stages = Chassis->GetFrameCount() - 1;
		const ULONG FRAME_WIDTH = START_ENERGY / Stages;

		USHORT Frame = 1;

		if (Energy <= START_ENERGY)
			Frame = Stages - (Energy / FRAME_WIDTH);

		if (Chassis->GetCurrentFrame() != Frame -1)
			Chassis->JumpToFrame(Frame - 1);
	}
}


/*
Finds the endpoint closest to the given cordinate component (UseX define which one) and returns the
distance component (UseX again) from center of the object to that point.
*/
float PlayerObject::GetClosestSpotDistance(float off, bool UseX) const {
	float Smallest = 9999.f;

	const Point<float> Center = GetCenter(Geo);
	for (USHORT x = 0; x < Extension.size(); ++x) {
		const std::vector<Vector<float> > *p = Extension[x]->GetEndpoints();
		if (!p)
			continue;
		for (USHORT y = 0; y < p->size(); ++y) {
			const Vector<float> RelToCenter(Geo + (*p)[y] - Center);
			const Vector<float> RotatedAbs(Center + polar(abs(RelToCenter), BoundAngle(Direction + arg(RelToCenter))));
			if (UseX) {
				if (Smallest > abs(RotatedAbs.x - off))
					Smallest = abs(RotatedAbs.x - off);
			}
			else {
				if (Smallest > abs(RotatedAbs.y - off))
					Smallest = abs(RotatedAbs.y - off);
			}
		}
	}

	if (Smallest == 9999.f) {
		_ERROR_;
		if (UseX)
			return Geo.Width / 2.f;
		else
			return Geo.Height / 2.f;
	}

	//return the nearest one
	if (UseX) {
		if (Geo.x + Geo.Width / 2.f < off)
			return abs(-Smallest + off - Center.x);
		else
			return abs(Smallest + off - Center.x);
	}
	else {
		if (Geo.y + Geo.Height / 2.f < off)
			return abs(-Smallest + off - Center.y);
		else
			return abs(Smallest + off - Center.y);
	}
}


float PlayerObject::GetSurfaceNormal(const Point<float> &Loc) const {
	const Point<float> TriangleCenter = Vector<float>(GetCenter(Geo)); //Is this true? It is not but it's close.
	const float LocDir = BoundAngle(arg(Loc - TriangleCenter) - Direction);

	float RetVal = 0;

	if (LocDir >= -M_PI / 16.f && LocDir < M_PI / 16.f)
		RetVal = 0;
	else if (LocDir >= M_PI / 16.f && LocDir < M_PI / 2.f + M_PI / 4.f - M_PI / 16.f)
		RetVal = M_PI / 2.f - M_PI / 8.f;
	else if (LocDir >= M_PI / 2.f + M_PI / 4.f - M_PI / 16.f && LocDir < M_PI / 2.f + M_PI / 4.f + M_PI / 16.f)
		RetVal = M_PI / 2.f + M_PI / 4.f;
	else if (LocDir >= M_PI / 2.f + M_PI / 4.f + M_PI / 16.f || LocDir < -M_PI + M_PI / 4.f - M_PI / 16.f)
		RetVal = M_PI;
	else if (LocDir >= -M_PI + M_PI / 4.f - M_PI / 16.f && LocDir < -M_PI + M_PI / 4.f + M_PI / 16.f)
		RetVal = -M_PI + M_PI / 4.f;
	else if (LocDir >= -M_PI + M_PI / 4.f + M_PI / 16.f && LocDir < -M_PI / 16.f)
		RetVal = -M_PI / 2.f + M_PI / 8.f;
	else {
		_ERROR_;
	}

	return BoundAngle(RetVal + Direction);
}


void PlayerObject::Damage(const PhysicalObject *Obj) {
	if (gGameOpt.GT == GT_TEAM_DEATHMATCH &&
		Identity.Team == Obj->Identity.Team &&
		Identity.OwnerObjUID != Obj->Identity.OwnerObjUID)
			PhysicalObject::DamageFrendly(Obj, gGameOpt.FrendlyFire);
	else
		PhysicalObject::Damage(Obj);

	LastObjCollided = Obj->Identity;
}


bool PlayerObject::OnCollide(const PhysicalObject *Obj) {
	if (!PhysicalObject::OnCollide(Obj))
		return false;

	//keep the ship and the wall at a decent distance
	//... by moving the ship, not the wall :-P
	if (Obj->GetTypeID() == Wall::ID) {
		const Rect<float> &Geo1 = Geo;
		const Rect<float> &Geo2 = Obj->GetGeo();
		const Vector<float> Dist = GetCenter(Geo1) - GetCenter(Geo2);
		const float xoff = Geo2.Width / 2.f + GetClosestSpotDistance(Geo2.x + Geo2.Width / 2.f, true) - abs(Dist.x);
		const float yoff = Geo2.Height / 2.f + GetClosestSpotDistance(Geo2.y + Geo2.Height / 2.f, false) - abs(Dist.y);

		if (abs(xoff - yoff) < 10) {
			if (GetCenter(Geo1).x < GetCenter(Geo2).x)
				Geo.x -= xoff;
			else
				Geo.x += xoff;

			if (GetCenter(Geo1).y < GetCenter(Geo2).y)
				Geo.y -= yoff;
			else
				Geo.y += yoff;
		}
		else if (xoff < yoff) {
			if (GetCenter(Geo1).x < GetCenter(Geo2).x)
				Geo.x -= xoff;
			else
				Geo.x += xoff;
		}
		else {
			if (GetCenter(Geo1).y < GetCenter(Geo2).y)
				Geo.y -= yoff;
			else
				Geo.y += yoff;
		}

		JustTranslocated = true;
	}

	return true;
}


void PlayerObject::Explode() {
	if (State == NORMAL) {
		Explosion->PlayOnce();
		Machinery->Reset();

		gNetworkServer->AddSound(ExplosionSound, true);

		if (GatheredWeaponList.find(ActiveWeapon) != GatheredWeaponList.end())
			GatheredWeaponList.find(ActiveWeapon)->second->LeavePacket(GetCenter(Geo));

		State = EXPLODING;

		gNetworkServer->SomeoneGotKilled(&LastObjCollided, &Identity);
	}
}


void PlayerObject::Reborn() {
	Energy = FULL_ENERGY;

	Object::ResetExtensions();

	GatheredWeaponList = InitialGatheredWeapons;
	for (std::map<const USHORT, Weapon*>::iterator x = GatheredWeaponList.begin(); x != GatheredWeaponList.end(); ++x)
		x->second->MakeGathered(-1);

	ActiveWeapon = GetBestWeaponIndex();
	GetActiveWeapon()->Show();
	ActiveWeaponWaiting = ActiveWeapon;

	Velocity = Vector<float>(0, 0);
	Rotation = 0;
}


void PlayerObject::Unlink() {
	gNetworkServer->RemovePlayerObj(Object::ObjectUID);
	PhysicalObject::Unlink();
}


void PlayerObject::OnArenaOut() {
	Energy = 0;
	UpdateEnergy();
}


bool PlayerObject::IsOver(FloatingObject* FloObj) {
	const USHORT Percent = 20;
	const Rect<float> &ObjGeo = FloObj->GetGeo();

	return State == NORMAL &&
		CheckRectCollision(Geo.x + Geo.Width * Percent / 100.f, Geo.y + Geo.Height * Percent / 100.f,
			Geo.Width - Geo.Width * Percent * 2 / 100.f, Geo.Height - Geo.Height * Percent * 2 / 100.f,
			ObjGeo.x + ObjGeo.Width / 2, ObjGeo.y + ObjGeo.Height / 2, 1, 1);
}


void PlayerObject::GetCompleteData(MessageCoder &MC) const {
	FullRotationInfo *FRI = new FullRotationInfo();
	FRI->Torque = Machinery->GetTorque();
	FRI->Direction = Direction;
	FRI->Rotation = Rotation;
	MC.Put(FRI);

	FullVelocityInfo *FMI = new FullVelocityInfo();
	FMI->Velocity = Velocity;
	FMI->Impulse = AdditionalImpulse;
	FMI->Loc = Geo;
	MC.Put(FMI);

	EngineInfo *EI = Machinery->GetEngineInfo();
	MC.Put(EI);

	ShipState *SS = new ShipState();
	SS->Energy = EnergyPercent;
	SS->ChassisFrame = Chassis->GetCurrentFrame();
	MC.Put(SS);

	ShipInfo *SI = new ShipInfo();
	SI->Name = Name;
	SI->Team = Identity.Team;
	MC.Put(SI);

	for (map<const USHORT, Weapon*>::const_iterator x = WeaponList.begin(); x != WeaponList.end(); ++x) {
		WeaponInfo *WI = x->second->GetWeaponInfo();
		MC.Put(WI);
	}

	Object::GetCompleteData(MC);
}


void PlayerObject::CheckData() {
	if (OldRotation != Rotation) {
		OldRotation = Rotation;
		RotationDataChanged = true;
	}
	else
		RotationDataChanged = false;

	fill(WeaponDataChanged, WeaponDataChanged + 10, false);
	for (map<const USHORT, Weapon*>::iterator x = WeaponList.begin(); x != WeaponList.end(); ++x) {
		WeaponInfo *WI = x->second->GetWeaponInfo();

		if ((*WI) != (*OldWeaponInfoList[x->first])) {
			delete OldWeaponInfoList[x->first];
			OldWeaponInfoList[x->first] = WI;
			WeaponDataChanged[x->first] = true;
		}
		else
			delete WI;
	}

	if (ImpulseChanged) {
		ImpulseChanged = false;
		ImpulseDataChanged = true;
	}
	else
		ImpulseDataChanged = false;

	//the ship can change location in abnormal ways, eg. but not limited to: when spawning
	if (JustSpawned || JustTranslocated) {
		JustSpawned = false;
		JustTranslocated = false;
		LocDataChanged = true;
		ImpulseDataChanged = true;
	}
	else
		LocDataChanged = false;

	ShipState SS;
	SS.Energy = EnergyPercent;
	SS.ChassisFrame = Chassis->GetCurrentFrame();
	if (OldShipState != SS) {
		OldShipState = SS;
		ShipDataChanged = true;
	}
	else
		ShipDataChanged = false;

	ShipInfo SI;
	SI.Name = Name;
	SI.Team = Identity.Team;
	if (OldShipInfo != SI) {
		OldShipInfo = SI;
		ShipInfoChanged = true;
	}
	else
		ShipInfoChanged = false;

	EngineInfo *EI = Machinery->GetEngineInfo();
	if ((*OldEngInf) != (*EI)) {
		delete OldEngInf;
		OldEngInf = EI;
		EngineDataChanged = true;
	}
	else {
		delete EI;
		EngineDataChanged = false;
	}

	Object::CheckData();
}


void PlayerObject::GetChangedData(MessageCoder &MC) const {
	if (RotationDataChanged) {
		FullRotationInfo *FRI = new FullRotationInfo();
		FRI->Torque = Machinery->GetTorque();
		FRI->Direction = Direction;
		FRI->Rotation = Rotation;
		MC.Put(FRI);
	}

	for (map<const USHORT, Weapon*>::const_iterator x = WeaponList.begin(); x != WeaponList.end(); ++x) {
		if (WeaponDataChanged[x->first]) {
			WeaponInfo *WI = x->second->GetWeaponInfo();
			MC.Put(WI);
		}
	}

	if (ImpulseDataChanged || LocDataChanged) {
		FullVelocityInfo *FMI = new FullVelocityInfo();
		FMI->Velocity = Velocity;
		FMI->Impulse = AdditionalImpulse;
		FMI->Loc = Geo;
		MC.Put(FMI);
	}

	if (EngineDataChanged) {
		EngineInfo *EI = Machinery->GetEngineInfo();
		MC.Put(EI);
	}

	if (ShipDataChanged) {
		ShipState *SS = new ShipState();
		SS->Energy = EnergyPercent;
		SS->ChassisFrame = Chassis->GetCurrentFrame();
		MC.Put(SS);
	}

	if (ShipInfoChanged) {
		ShipInfo *SI = new ShipInfo();
		SI->Name = Name;
		SI->Team = Identity.Team;
		MC.Put(SI);
	}

	Object::GetChangedData(MC);
}
