/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef NETWORK_SERVER_H
#define NETWORK_SERVER_H

#include "network_messages.h"


#include <string>
#include <map>
#include <queue>
#include <stack>
#include <set>
#include <vector>
#include <list>

#include <zig/zigserver.h>
#include <zig/buffer.h>

#undef writeString
#undef readString


#include "structures.h"

#include "message_coder.h"

#include "poll.h"

#define MAP_OBJ_ITER std::map<OUIDInfo, Object*>::iterator
#define MAP_FLO_OBJ_ITER std::map<OUIDInfo, FloatingObject*>::iterator
#define MAP_PHY_OBJ_ITER std::map<OUIDInfo, PhysicalObject*>::iterator
#define MAP_PLA_OBJ_ITER std::map<OUIDInfo, PlayerObject*>::iterator
#define MAP_CTL_OBJ_ITER std::map<OUIDInfo, ControlledObject*>::iterator


class Object;
class PhysicalObject;
class ControlledObject;
class PlayerObject;
class FloatingObject;
class Sound;
struct IDENTITY;


struct ServerOptions {
	USHORT Port;
	USHORT MaxClients;
	std::string Name;
	std::string Password;
	std::string WelcomeMessage;
	float Tick; //in seconds
	int NetRatio;
	int MinPacketCompressSize;
	int ClientTimeout;
	bool ReliableDeltaPackets;
};


class NetworkServer : private zigserver_c {
public:
	NetworkServer();

	bool Start(ServerOptions SO, GameOptions GO);
	//This method is called constantly.
	void Process() {zigserver_c::process_nonblocking();}
	void Update();
	void Stop();

	bool HighLoad() {return ClientList.size();}

	OUIDInfo Add(Object*);
	void Add(PhysicalObject*, OUIDInfo ObjectUID);
	void Add(FloatingObject*, OUIDInfo ObjectUID);
	void Add(PlayerObject*, OUIDInfo ObjectUID);

	void RemoveObject(OUIDInfo ObjectID);
	void RemovePhysicalObj(OUIDInfo ObjectID);
	void RemovePlayerObj(OUIDInfo ObjectID);
	void RemoveFloatingObj(OUIDInfo ObjectID);

	void AddSound(const Sound *S, bool Untunable);
	void RemoveSound(const OUIDInfo &SoundUID);

	float GetViscosity();

	void SetCamera(int ClientID, OUIDInfo CameraOwner);
	void ReleaseCamera(int ClientID, OUIDInfo CameraOwner);

	void SetInputObject(int ClientID, ControlledObject *InputObject);
	void ReleaseInputObject(int ClientID, ControlledObject *InputObject);

	void SomeoneGotKilled(const IDENTITY *Who, const IDENTITY *Whose);

	USHORT GetCodeFromPath(const std::string &Path);

private:
	/* Object lists are implemented as double buffer. That way you cannot make changes to the list
	   while iterating it.
	*/
	std::map<OUIDInfo, Object*> ObjectList;
	std::map<OUIDInfo, PlayerObject*> PlayerObjList;
	std::map<OUIDInfo, PhysicalObject*> PhysicalObjList;
	std::map<OUIDInfo, FloatingObject*> FloatingObjList;

	//object lists locally used (copies)
	std::map<OUIDInfo, Object*> lObjectList;
	std::map<OUIDInfo, PlayerObject*> lPlayerObjList;
	std::map<OUIDInfo, PhysicalObject*> lPhysicalObjList;
	std::map<OUIDInfo, FloatingObject*> lFloatingObjList;

	std::stack<OUIDInfo> ObjectToKill;

	std::map<const OUIDInfo, const Sound *const> SoundListPermanent;
	std::vector<const Sound*> SoundList;

	struct CloseObjects {
		CloseObjects(const Rect<float>* const r) : Quad(r) {}
		const Rect<float>* const Quad;
		std::vector<PhysicalObject*> List;
	};
	std::vector<CloseObjects*> CollisionRectList;
	const USHORT QuadSize;
	USHORT QuadsPerW;

	struct Client {
		std::set<OUIDInfo> RelevantObjects;
		std::set<OUIDInfo> SoundsListening;

		std::list<OUIDInfo> CameraOwner;
		std::list<OUIDInfo> lCameraOwner;
		std::list<ControlledObject*> ControlledObjList;

		OUIDInfo MainObjectUID;
		std::queue<set<INPUT_COMMAND> > KeysUp;

		Rect<float> CameraGeo;
		Size TargetCameraDim;
		std::string Name;
		OUIDInfo CameraOwnerSent;
		int LastPacketID;

		UCHAR SkippedPingResultDisplay;
		USHORT LastPingValue;

		TEAM Team;
		char Points;

		int DataMsgStream;
		int InfoMsgStream;
	};
	std::map<int, Client> ClientList;

	std::map<TEAM, char> TeamPointsList;
	bool ScoreChanged;

	bool MatchOver;
	bool CheckForWinner();
	void HonorTheWinner(const int &CliID);

	void OnTeamSuicide(const IDENTITY *Who);
	void OnTeamHomocide(const IDENTITY *Who, const IDENTITY *Whose);
	void OnSuicide(const IDENTITY *Who);
	void OnHomocide(const IDENTITY *Who, const IDENTITY *Whose);

	void PutScore(buffer_c &Data);
	void PutCameraInfo(Client &Cli, buffer_c &Data);

	void PutSoundInfo(Client &ActualCli, buffer_c &Data);
	inline bool IsInsideAcousticalRange(const Point<float> &CameraGeo, const Point<float> &Source);

	std::map<USHORT, std::string> PathCoder;
	std::pair<std::string, USHORT> PathCoderCache[5];

	Size ArenaSize;

	OUIDInfo CurrentUID;

	ServerOptions SrvOpt;

	Poll CurrentPoll;

	USHORT BroadcastFrameSkippedCount;

	int SendingPingResultsSkippedCount;

	/* We do our own timing so just ignore this callbacks. */
	virtual bool update_frame() {return true;}
	virtual bool broadcast_frame() {return true;}

	virtual bool accept_client(address_c &client_addr, buffer_c &custom, buffer_c &answer);
	virtual void client_connected(int client_id, buffer_c &custom);
	virtual void client_disconnected(int client_id, int zig_reason);
	virtual void client_incoming_data(int client_id, buffer_c &in, int packet_id);
	virtual void client_ping_result(int client_id, int ping_value);
	virtual bool get_server_info(address_c &remote_addr, buffer_c &custom, buffer_c &answer);
	virtual void finish();

	int CurrentMap;
	float TimeToWaitBeforeStartNewMatch;
	bool StartNewMatch();
	void CheckCollisions();
	void DoObjectRemoval();
	void DestroyWorld();
	bool CreateWorld(const std::string &MapFilename);
	OUIDInfo CreatePlayerObject(ClientInfo *CI, int ClientID);

	void SetCameraZoom(Size &CameraDim, float Offset);

	void ProcessOutgoingGameDataMsg();
	void ProcessOutgoingGameStateMsg();

	void ProcessPollResults();
	void ProcessRequestFromClient(const ServerRequestInfo *SRI, int CliID);
	void SendServerMessage(int ToWho, const std::string &Text);
	void ChangeTeam(int CliID, TEAM Team);

	void BroadcastMessage(buffer_c &data);

	bool Running;

	//possible return values of GetRequedMsgDataType
	enum {
		MDT_NONE,
		MDT_COMPLETE,
		MDT_CHANGED,
		MDT_JUST_UID};
	int GetRequedMsgDataType(Client &Cli, const Object *Obj, OUIDInfo ObjID);
	void ManageRelevantObjects(Client &Cli, MessageCoder &MC);

	void ProcessIncomingUnreliableMsg(int client_id, buffer_c &in);
	void ProcessIncomingReliableMsg(int client_id);

	std::map<std::string, std::string> MapListCache;

	struct SameTeam : unary_function<std::pair<int, Client>, bool> {
		SameTeam (const TEAM &T) : Team(T) {};
		bool operator() (const std::pair<int, Client> &cli) const {
			return cli.second.Team == Team;
		}
	private:
		const TEAM &Team;
	};

	int FindClientByName(const std::string &Name);
	const std::string UnambigouizePlayerName(const std::string &Name);
};


#endif
