/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <allegro.h>
#ifdef WIN32
	#include <winalleg.h>
#endif

#include <sstream>

#include "network_server.h"

#include "game_logic.h"
#include "initializer.h"
#include "network_messages.h"
#include "message_coder.h"
#include "sparklet_utils.h"

#include "player.h"

#include "sound.h"

#include "exceptions.h"

#include <zig/utils.h>

#define VISCOSITY_STR std::string("viscosity")


//define the interval in which the server will be sending ping results of all clients to all
//clients (in seconds)
#define SENDING_PING_RESULTS_INTERVAL 4


//number of seconds to wait before starting a new match
#define TIME_BETWEEN_MATCHES 8


GameLogic *gGameLogic = NULL;
NetworkServer *gNetworkServer = NULL;
GameOptions gGameOpt;

extern Initializer *gInitializer;
extern float gTimeElapsed;


OUIDInfo Sound::CurrentSoundUID(0);



NetworkServer::NetworkServer() :
	zigserver_c(gLC),
	QuadSize(400),
	Running(false) {

	gNetworkServer = this;

	const string SharedPath = std::string(DATA_PREFIX) + "*";
	for_each_file_ex(SharedPath.c_str(), 0, 0, FillPaths, &PathCoder);
}


int FillPaths(const char *filename, int attrib, void *PathCoder) {
	if (attrib & FA_DIREC) {
		if (strcmp(get_filename(filename), ".") && strcmp(get_filename(filename), "..")) {
			const string Path = std::string(filename) + FS_SLASH + '*';
			for_each_file_ex(Path.c_str(), 0, 0, FillPaths, PathCoder);
		}
		return 0;
	}
	else {
		int length = strlen(filename);
		char *_RelPath = (char*)malloc(length);
		string RelPath(make_relative_filename(_RelPath, DATA_PREFIX, filename, length));

		static USHORT Code = RESERVED_PATH_CODE_COUNT;
		static_cast<map<USHORT, string>* >(PathCoder)->insert(make_pair(Code, FixFilenameSlashes(RelPath)));

		++Code;
		if (Code > 1000) {
			_ERROR_;
			abort();
		}
		free(_RelPath);
		return 0;
	}
}


USHORT NetworkServer::GetCodeFromPath(const std::string &Path) {
	_SPARKLET_ASSERT(Path.length());

	for (USHORT x = 0; x < 5; ++x)
		if (PathCoderCache[x].first == Path)
			return PathCoderCache[x].second;

	for (std::map<USHORT, std::string>::iterator x = PathCoder.begin(); x != PathCoder.end(); ++x) {
		if (x->second == Path) {
			for (USHORT y = 4; y > 0; --y)
				PathCoderCache[y] = PathCoderCache[y - 1];
			PathCoderCache[0] = make_pair<std::string, USHORT>(x->second, x->first);

			return x->first;
		}
	}

	_ERROR_;
	_DISPLAY(Path);
	return 0;
}


float NetworkServer::GetViscosity() {
	return gGameOpt.MediumViscosity;
}


bool NetworkServer::Start(ServerOptions SO, GameOptions GO) {
	zigserver_c::set_server_tick(SO.Tick);
	zigserver_c::set_net_ratio(SO.NetRatio);
	zigserver_c::set_compression(SO.MinPacketCompressSize);
	zig_set_compression_level(3);
	zigserver_c::set_client_timeout(SO.ClientTimeout);

	_DISPLAY(SO.NetRatio);

	if (GO.MapFileNameList.size() == 0)
		return false;

	CurrentUID = RESERVED_OBJECT_UID_COUNT;

	gGameLogic = new GameLogic();

	SrvOpt = SO;
	gGameOpt = GO;

	MatchOver = false;
	ScoreChanged = false;
	BroadcastFrameSkippedCount = SrvOpt.NetRatio;
	TimeToWaitBeforeStartNewMatch = TIME_BETWEEN_MATCHES;
	//make it send immediately
	SendingPingResultsSkippedCount = (int)(SENDING_PING_RESULTS_INTERVAL / (SrvOpt.NetRatio * SrvOpt.Tick));

	if (!CreateWorld(gGameOpt.MapFileNameList.at(gGameOpt.CurrentMap))) {
		delete gGameLogic;
		return false;
	}

	MapListCache.clear();
	for (USHORT x = 0; x < gGameOpt.MapFileNameList.size(); ++x) {
		const std::string Path = FixFilenameSlashes(DATA_PREFIX + gGameOpt.MapFileNameList.at(x));
		XMLParser Parser(Path);
		const TiXmlElement *Root = Parser.GetRoot();
		MapListCache[XMLParser::GetValueS(Root, "name", "")] = Path;
	}

	_CHECKPOINT_;

	Running = true;
	return zigserver_c::start(SO.Port, SO.MaxClients);
}


void NetworkServer::Stop() {
	_SAY("Stop called!");
	zigserver_c::stop();
}


void NetworkServer::finish() {
	_SAY("finish called!!!");

	CurrentPoll.Reset();

	delete gGameLogic;

	DestroyWorld();

	for (USHORT x = 0; x < CollisionRectList.size(); ++x) {
		delete CollisionRectList[x]->Quad;
		delete CollisionRectList[x];
	}
	CollisionRectList.clear();

	ClientList.clear();

	SoundListPermanent.clear();
	Running = false;
}


bool NetworkServer::accept_client(address_c &client_addr, buffer_c &custom, buffer_c &answer) {
	//Don't even try to read data smaller than 7 bytes. It's not a guarantee that passing this test
	//the data will be correct. It's just a preliminary test.
	if (custom.size_left() < 7)
		return false;

	std::string PlayerName;

	try {
		//the first data is the protocol version
		const USHORT ProtocolVer = custom.getShort();

		//If client's and server's network protocol version don't match, recject the client.
		if (ProtocolVer != NSB_NET_PROTOCOL_VER) {
			answer << "Incompatibile network protocol";
			_SAY("Client rejected from " + client_addr.get_address());
			_SAY("Reason: Incompatibile network protocol.");
			return false;
		}

		//the second data is the password string
		const std::string ClientPassword = custom.getString();

		//if the servers requires password and the client-supplyed one doesn't match,
		//reject the client
		if (SrvOpt.Password != "" && SrvOpt.Password != ClientPassword) {
			answer << "Wrong password!";
			_SAY("Client rejected from " + client_addr.get_address());
			_SAY("Reason: Wrong password - " + ClientPassword);
			return false;
		}

		//The third data is client info
		const auto_ptr<const ClientInfo> CI(static_cast<const ClientInfo*>(custom.getObject()));
		PlayerName = UnambigouizePlayerName(CI->Name);

		//safety check
		if (CI->Team > LAST_TEAM) {
			_ERROR_;
			_SAY("Server received illegal connection packet, dropping connection.");
			_SAY("Client rejected from " + client_addr.get_address());
			return false;
		}
	} catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Server received illegal connection packet, dropping connection.");
		_SAY("Client rejected from " + client_addr.get_address());
		return false;
	} catch (...) {
		_ERROR_;
		_SAY("Unknown exception cought!");
		_SAY("Client rejected from " + client_addr.get_address());
		return false;
	}

	if (custom.size_left()) {
		answer << "Bad data!";
		_SAY("Client rejected from " + client_addr.get_address());
		_SAY("Reason: Malformed connection data.");
		return false;
	}

	//From now on, the client is accepted
	//Log that.
	_SAY("Client Accepted from " + client_addr.get_address());

	//put welcome message in the answer string
	answer << SrvOpt.WelcomeMessage;

	//put game options in the answer string
	answer << gGameOpt;

	//put the size of arena
	answer.putShort(ArenaSize.Width);
	answer.putShort(ArenaSize.Height);

	//put the NetRatio, client needs to know it for some reason
	answer.putShort(SrvOpt.NetRatio);

	//put player name that the server may have changed
	answer.putString(PlayerName);

	/*put coded paths ALWAYS the LAST one in the buffer. That's because of stupid coded paths
	reading routine*/
	//put coded path in the answer
	CodedPaths CP(PathCoder);
	answer << CP;

	return true;
}


void NetworkServer::client_connected(int client_id, buffer_c &custom) {
	/* The consistency of custom buffer is guaranteed because it passed tests in accept_client() */

	//skip
	custom.getShort();
	custom.getString();

	//The third data is client info
	ClientInfo *CI = static_cast<ClientInfo*>(custom.getObject());

	//prevent names collision
	CI->Name = UnambigouizePlayerName(CI->Name);

	ClientList.insert(ClientList.end(), make_pair(client_id, Client()));
	Client &Cli = get_element(ClientList, client_id);

	Cli.CameraOwnerSent = 0;
	Cli.LastPacketID = 0;
	Cli.SkippedPingResultDisplay = 0;
	Cli.LastPingValue = 0;
	Cli.Name = CI->Name;

	Cli.CameraGeo = Rect<float>(-1, -1, UNITS_PER_SCREEN_W, UNITS_PER_SCREEN_H);
	Cli.TargetCameraDim = Size(UNITS_PER_SCREEN_W, UNITS_PER_SCREEN_H);

	Cli.Team = CI->Team;
	Cli.Points = 0;

	if (gGameOpt.GT == GT_TEAM_DEATHMATCH) {
		if (TeamPointsList.find(CI->Team) == TeamPointsList.end())
			TeamPointsList.insert(make_pair(CI->Team, 0));
	}
	
	const OUIDInfo MainObjUID = CreatePlayerObject(CI, client_id);

	if (MainObjUID == 0) {
		_SAY("Disconnecting the client because couldn't create a player (malformed packet?).");
		
		//clean up
		zigserver_c::disconnect_client(client_id);
		delete CI;
		ClientList.erase(client_id);

		return;
	}
	
	Cli.MainObjectUID = MainObjUID;

	Cli.DataMsgStream = zigserver_c::create_stream(client_id, STREAM_CUMULATIVE_ACK);
	Cli.InfoMsgStream = zigserver_c::create_stream(client_id, STREAM_INDIVIDUAL_ACK);

	//notify all clients that a new player has connected
	NewPlayerInfo NPI;
	NPI.Name = CI->Name;
	NPI.Team = CI->Team;
	NPI.UID = client_id;
	buffer_c Msg;
	Msg << NPI;
	BroadcastMessage(Msg);

	//tell the newcomer about players already connected
	for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
		if (x->first != client_id) {
			NewPlayerInfo OtherPlayer;
			OtherPlayer.Name = x->second.Name;
			OtherPlayer.Team = x->second.Team;
			OtherPlayer.UID = x->first;
			Msg.clear();
			Msg << OtherPlayer;
			zigserver_c::send_message(client_id, Msg, x->second.InfoMsgStream, true);
		}
	}

	ScoreChanged = true;

	delete CI;

	_SAY("Client Connected!");
	_DISPLAY(client_id);
}


void NetworkServer::client_disconnected(int client_id, int zig_reason) {
	_SAY("Client Disconnected");
	_DISPLAY(client_id);

	/* The client was disconnected bofore the connection process finished. */
	if (ClientList.find(client_id) == ClientList.end()) {
		return;
	}

	switch (zig_reason) {
		case ZIGSERVER_DISCONNECTED:
			_SAY("Plain disconnection.");
		break;
		case ZIGSERVER_DISCONNECTED_TIMEOUT:
			_SAY("Disconnected due to time-out.");
		break;
		case ZIGSERVER_DISCONNECTED_PROTOCOL_VIOLATION:
			_SAY("Disconnected due to protocol violation.");
		break;
		case ZIGSERVER_DISCONNECTED_SOCKET_ERROR:
			_SAY("Disconnected due to socket error.");
		break;
		case ZIGSERVER_DISCONNECTED_SHUTDOWN:
			_SAY("Disconnected due to server shutdown.");
		break;
		default:
			_DISPLAY(zig_reason);
	}

	const Client &Cli = get_element(ClientList, client_id);
	get_element<OUIDInfo, Object*>(ObjectList, Cli.MainObjectUID)->Unlink();
	ClientList.erase(client_id);

	PlayerGoneInfo PGI;
	PGI.UID = client_id;
	buffer_c Msg;
	Msg << PGI;

	//notice all remaining clients about client disconnection
	for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
		zigserver_c::send_message(x->first, Msg, x->second.InfoMsgStream, true);

	for (map<TEAM, char>::iterator x = TeamPointsList.begin(); x != TeamPointsList.end(); ++x) {
		if (find_if(ClientList.begin(), ClientList.end(), SameTeam(x->first)) == ClientList.end()) {
			TeamPointsList.erase(x);
			break;
		}
	}

	ScoreChanged = true;
}


void NetworkServer::client_incoming_data(int client_id, buffer_c &in, int PacketID) {
	if (ClientList.find(client_id) == ClientList.end()) {
		_SAY("Illegal packet received, dropping it");
		return;
	}

	//dropping reordered and late packets
	if (PacketID <= get_element(ClientList, client_id).LastPacketID) {
		//_SAY("Packet dropped.");
		return;
	}

	get_element(ClientList, client_id).LastPacketID = PacketID;

	ProcessIncomingReliableMsg(client_id);
	ProcessIncomingUnreliableMsg(client_id, in);
}


void NetworkServer::ProcessIncomingReliableMsg(int CliID) {
	buffer_c Data;
	int StreamID;

	seek_first_stream(CliID);

	while ((StreamID = get_next_stream(CliID)) != -1) {
		while (zigserver_c::receive_message(CliID, Data, StreamID) == 1) {
			try {
				auto_ptr<serializable_c> Msg(static_cast<serializable_c *>(Data.getObject()));
				const int code = Msg->ZIG_GetTypeId();

				if (code == PlayerMessage::oCreator.GetTypeId()) {
					PlayerMessage *PM = static_cast<PlayerMessage*>(Msg.get());

					if (PM->Text.length() > 200) {
						_ERROR_;
						continue;
						//throw EX_INVALID_DATA;
					}

					buffer_c OutData;
					PM->From = CliID;
					OutData << *PM;
					if (PM->To.empty())
						BroadcastMessage(OutData);
					else if (PM->To == "/team/") {
						for (std::map<int, Client>::iterator it = ClientList.begin(); it != ClientList.end(); ++it) {
							if (it->second.Team == get_element(ClientList, CliID).Team)
								zigserver_c::send_message(it->first, OutData, it->second.InfoMsgStream, true);
						}
					}
					else {
						int ReceiverClientID = FindClientByName(PM->To);
						zigserver_c::send_message(ReceiverClientID, OutData, get_element(ClientList, ReceiverClientID).InfoMsgStream, true);
					}
				}
				else if (code == NewKickPollInfo::oCreator.GetTypeId()) {
					NewKickPollInfo *NKPI = static_cast<NewKickPollInfo*>(Msg.get());

					if (!CurrentPoll.Running()) {
						SendServerMessage(-1, get_element(ClientList, CliID).Name + " started a poll to kick " + NKPI->KickWho + ".");
						if (!NKPI->Reason.empty())
							SendServerMessage(-1, "Reason: " + NKPI->Reason);
						SendServerMessage(-1, "Please vote /yes or /no within a minute.");

						CurrentPoll.Start(POLL_KICK_PLAYER, FindClientByName(NKPI->KickWho), "", ClientList.size());
					}
					else
						SendServerMessage(CliID, "A new poll cannot be started now because one is already running.");
				}
				else if (code == StateInfoGeneric1::oCreator.GetTypeId()) { //vote yes or no to current pool
					const StateInfoGeneric1 *SIG = static_cast<const StateInfoGeneric1*>(Msg.get());
					CurrentPoll.Vote(CliID, SIG->State);

					SendServerMessage(-1, get_element(ClientList, CliID).Name + " voted " + (SIG->State ? "'yes'." : "'no'"));
				}
				else if (code == ServerRequestInfo::oCreator.GetTypeId()) {
					const ServerRequestInfo *SRI = static_cast<const ServerRequestInfo*>(Msg.get());
					ProcessRequestFromClient(SRI, CliID);
				}
				else if (code == ChangeMapCommandInfo::oCreator.GetTypeId()) {
					const ChangeMapCommandInfo *CMCI = static_cast<const ChangeMapCommandInfo*>(Msg.get());
					if (!CurrentPoll.Running()) {
						if (MapListCache.find(CMCI->MapName) != MapListCache.end()) {
							CurrentPoll.Start(POLL_CHANGE_MAP, 0, CMCI->MapName, ClientList.size());
							SendServerMessage(-1, get_element(ClientList, CliID).Name + " started a poll to change map to '" + CMCI->MapName + "'.");
							SendServerMessage(-1, "Please vote /yes or /no within a minute.");
						}
						else
							SendServerMessage(CliID, "Map named '" + CMCI->MapName + "' was not found on the server.");
					}
					else
						SendServerMessage(CliID, "A new poll cannot be started now because one is already running.");
				}
				else if (code == ChangeTeamCommandInfo::oCreator.GetTypeId()) {
					const ChangeTeamCommandInfo *CTCI = static_cast<const ChangeTeamCommandInfo*>(Msg.get());
					if (GetTeamColor(CTCI->Team) != 0) {
						SendServerMessage(-1, get_element(ClientList, CliID).Name + " switched to " + GetTeamName(CTCI->Team) + " team.");
						ChangeTeam(CliID, CTCI->Team);
					}
				}
				else {
					_ERROR_;
					_SAY("Unknown packet.");
				}
			} catch (zig_int_exceptions_t ex) {
				_ERROR_;
				_SAY("Server received illegal packet, dropping it.");
				//_SAY("Server received illegal packet, disconnecting the client.");
				//zigserver_c::disconnect_client(CliID);

				continue;
			}
		}
	}
}


void NetworkServer::ProcessIncomingUnreliableMsg(int CliID, buffer_c &in) {
	if (!in.size())
		return;

	try {
		auto_ptr<const KeySet> KeysDown(static_cast<const KeySet*>(in.getObject()));

		if (!KeysDown->empty()) {
			ControlledObject *CO = get_element(ClientList, CliID).ControlledObjList.back();
			CO->OnKeyDown(*KeysDown.get());

			if (KeysDown->find(IC_ZOOM_IN) != KeysDown->end())
				SetCameraZoom(get_element(ClientList, CliID).TargetCameraDim, -ZOOM_GRANULARITY_PERCENT);
			if (KeysDown->find(IC_ZOOM_OUT) != KeysDown->end())
				SetCameraZoom(get_element(ClientList, CliID).TargetCameraDim, ZOOM_GRANULARITY_PERCENT);
		}

		get_element(ClientList, CliID).KeysUp.push(*KeysDown.get());
	} catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Server received illegal packet, dropping it.");
		//_SAY("Server received illegal packet, disconnecting the client.");
		//zigserver_c::disconnect_client(CliID);

		return;
	}
}


void NetworkServer::ProcessRequestFromClient(const ServerRequestInfo *SRI, int CliID) {
	switch (SRI->RequestCode) {
		case SR_MAP_LIST:
			std::string MapList;

			for (std::map<std::string, std::string>::iterator it = MapListCache.begin();
				it != MapListCache.end(); ++it) {
				MapList += it->first + ", ";
			}
			MapList = MapList.substr(0, MapList.length() - 2);
			SendServerMessage(CliID, "Available maps are: " + MapList);
		break;
	}
}


void NetworkServer::SendServerMessage(int CliID, const std::string &Text) {
	ServerMessageInfo SMI;
	SMI.Text = Text;
	buffer_c buf;
	buf << SMI;

	if (CliID == -1)
		BroadcastMessage(buf);
	else
		zigserver_c::send_message(CliID, buf, get_element(ClientList, CliID).InfoMsgStream, true);
}


void NetworkServer::ChangeTeam(int CliID, TEAM Team) {
	TEAM OldTeam = get_element(ClientList, CliID).Team;
	get_element(ClientList, CliID).Team = Team;
	get_element(PhysicalObjList, get_element(ClientList, CliID).MainObjectUID)->Identity.Team = Team;

	if (count_if(ClientList.begin(), ClientList.end(), SameTeam(OldTeam)) == 0)
		TeamPointsList.erase(OldTeam);

	if (TeamPointsList.find(Team) == TeamPointsList.end())
		TeamPointsList.insert(make_pair(Team, 0));

	if (gGameOpt.GT == GT_TEAM_DEATHMATCH)
		get_element(ClientList, CliID).Points = 0;

	ScoreChanged = true;

	ChangeTeamCommandInfo CTCI;
	CTCI.Team = Team;
	CTCI.Who = CliID;
	buffer_c buf;
	buf << CTCI;
	BroadcastMessage(buf);
}


void NetworkServer::SetCameraZoom(Size &CameraDim, float Percent) {
	const int OffsetW = (int)(Percent / 100.f * UNITS_PER_SCREEN_W);
	const int OffsetH = (int)(Percent / 100.f * UNITS_PER_SCREEN_H);

	if (Percent > 0) {
		CameraDim.Width += OffsetW;
		if (CameraDim.Width > ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_W)
			CameraDim.Width = (USHORT)(ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_W);

		CameraDim.Height += OffsetH;
		if (CameraDim.Height > ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_H)
			CameraDim.Height = (USHORT)(ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_H);
	}
	else {
		CameraDim.Width += OffsetW;
		if (CameraDim.Width < ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_W)
			CameraDim.Width = (USHORT)(ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_W);

		CameraDim.Height += OffsetH;
		if (CameraDim.Height < ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_H)
			CameraDim.Height = (USHORT)(ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_H);
	}
}



void NetworkServer::ProcessOutgoingGameStateMsg() {
	buffer_c Data;

	if (ScoreChanged) {
		ScoreChanged = false;
		PutScore(Data);
		BroadcastMessage(Data);
	}

	SendingPingResultsSkippedCount++;
	if (SendingPingResultsSkippedCount >= SENDING_PING_RESULTS_INTERVAL / (SrvOpt.NetRatio * SrvOpt.Tick)) {
		SendingPingResultsSkippedCount = 0;

		PingInfo PI;
		for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
			PI.push_back(x->second.LastPingValue);
		Data.clear();
		Data << PI;
		BroadcastMessage(Data);
	}
}


void NetworkServer::ProcessOutgoingGameDataMsg() {
	for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
		const int &ClientID = x->first;
		Client &ActualCli = x->second;

		const OUIDInfo CameraOwner = ActualCli.lCameraOwner.back();
		const Size NewCameraSize = CalcCameraDim(ActualCli.CameraGeo, ActualCli.TargetCameraDim);
		const Point<float> TargetCameraLoc = GetCameraTargetSmart(GetCenter(get_element(lObjectList, CameraOwner)->GetGeo()), NewCameraSize, ArenaSize);
		const Point<float> NewCameraLoc = CalcCameraLoc(ActualCli.CameraGeo, TargetCameraLoc);
		ActualCli.CameraGeo = Rect<float>(NewCameraLoc, NewCameraSize);

		//data to be sent
		buffer_c RelData;
		buffer_c UnrelData;

		PutSoundInfo(ActualCli, UnrelData);

		PutCameraInfo(ActualCli, RelData);

		MessageCoder RelMC;
		MessageCoder UnrelMC;

		for (MAP_OBJ_ITER Obj = lObjectList.begin(); Obj != lObjectList.end(); Obj++) {
			const OUIDInfo ObjectID = Obj->first;
			const Object *ActualObj = Obj->second;

			UnrelMC.Begin(ObjectID);
			RelMC.Begin(ObjectID);

			switch (GetRequedMsgDataType(ActualCli, ActualObj, ObjectID)) {
				case MDT_COMPLETE:
					ActualObj->GetCompleteData(RelMC);
				break;

				case MDT_CHANGED:
					ActualObj->GetChangedData(SrvOpt.ReliableDeltaPackets ? RelMC : UnrelMC);
				break;

				case MDT_JUST_UID:
					ActualObj->GetEmptyData(RelMC);
				break;
			}

			RelMC.End();
			UnrelMC.End();
		}

		ManageRelevantObjects(ActualCli, RelMC);
		RelMC.GetCodedMessage(RelData);
		zigserver_c::send_message(ClientID, RelData, get_element(ClientList, ClientID).DataMsgStream, true);

		if (!SrvOpt.ReliableDeltaPackets)
			UnrelMC.GetCodedMessage(UnrelData);

		zigserver_c::send_packet(ClientID, UnrelData);
	}
}


int NetworkServer::GetRequedMsgDataType(Client &Cli, const Object *Obj, OUIDInfo ObjID) {
	if (Obj->IsInsideCamera(Cli.CameraGeo)) {
		if (Cli.RelevantObjects.find(ObjID) == Cli.RelevantObjects.end()) {
			Cli.RelevantObjects.insert(ObjID);
			return MDT_COMPLETE;
		}

		return MDT_CHANGED;
	}

	if (Cli.RelevantObjects.find(ObjID) != Cli.RelevantObjects.end()) {
		Cli.RelevantObjects.erase(ObjID);
		return MDT_JUST_UID;
	}

	return MDT_NONE;
}


void NetworkServer::ManageRelevantObjects(Client &Cli,  MessageCoder &MC) {
	stack<OUIDInfo> ToRemove;

	for (set<OUIDInfo>::iterator iter = Cli.RelevantObjects.begin(); iter != Cli.RelevantObjects.end(); ++iter) {
		const OUIDInfo ObjID = *iter;
		if (lObjectList.find(ObjID) == lObjectList.end()) {
			MC.PutRemove(ObjID);
			ToRemove.push(ObjID);
		}
	}

	while (!ToRemove.empty()) {
		Cli.RelevantObjects.erase(ToRemove.top());
		ToRemove.pop();
	}
}


void NetworkServer::AddSound(const Sound *S, bool Untunable) {
	if (Untunable) {
		Sound *Snd = new Sound(*S);
		SoundList.push_back(Snd);
	}
	else //if (SoundListPermanent.find(S->SoundUID) != SoundListPermanent.end())
		SoundListPermanent.insert(std::make_pair(S->SoundUID, S));
}


void NetworkServer::RemoveSound(const OUIDInfo &SoundUID) {
	SoundListPermanent.erase(SoundUID);
}


void CalcPanAndVolume(const Point<float> &CameraGeo, const Point<float> &Source, UCHAR &Pan, UCHAR &Volume) {
	const float range = ACOUSTICAL_RANGE;
	const float distance = abs(Source - CameraGeo);

	if (distance <= range)
		Volume = (UCHAR)(255.f * (1.f - (distance / range)));
	else
		Volume = 0;

	const float x = (CameraGeo.x - Source.x);
	const float pan_extreme = ACOUSTICAL_SWEEP_EXTREME;
	if (x < -pan_extreme)
		Pan = 255;
	else if (x > pan_extreme)
		Pan = 0;
	else
		Pan = 127 - (UCHAR)(128.f * (x / pan_extreme));
}


void NetworkServer::PutSoundInfo(Client &Cli, buffer_c &Data) {
	//remember this position and skip one byte, this is the place where we will put
	//the counter eventually
	const int NSIUCounterPos = Data.getpos();
	Data.skip(sizeof(UCHAR));
	int NSIUCounter = 0;

	//put non-permanent sounds, thay are always new to the client
	//we're limited to 256 sounds (protocol limit), we should look at 'proirity'
	//not just send first 256 sounds
	for (USHORT x = 0; x < SoundList.size() && NSIUCounter < 255; ++x) {
		const Sound *const S = SoundList.at(x);

		if (IsInsideAcousticalRange(Cli.CameraGeo, S->Source)) {
			NewSoundInfoUntunable *NSIU = new NewSoundInfoUntunable;

			NSIU->PathCode = S->PathCode;
			NSIU->SoundUID = S->SoundUID;
			NSIU->Priority = S->Priority;
			NSIU->PlayMode = S->PlayMode;

			UCHAR abs_volume;
			CalcPanAndVolume(Cli.CameraGeo, S->Source, NSIU->Pan, abs_volume);
			NSIU->Volume = static_cast<UCHAR>(abs_volume * (S->InitVolume / 255.f));

			Data << *NSIU;
			delete NSIU;
			++NSIUCounter;
		}
	}

	//seek to the place reserved for counter and put it there
	Data.seek(NSIUCounterPos);
	_SPARKLET_ASSERT(NSIUCounter < 256);
	Data.putByte(NSIUCounter);
	//seek back to the end of the data
	Data.seek_end();

	//remember this position and skip one byte, this is the place where we will put
	//the counter eventually
	const int NSICounterPos = Data.getpos();
	Data.skip(sizeof(UCHAR));
	int NSICounter = 0;

	//put permanent sounds that are new to the client
	for (std::map<const OUIDInfo, const Sound *const>::iterator it = SoundListPermanent.begin();
		it != SoundListPermanent.end() && NSICounter < 255;
		++it) {
			const OUIDInfo &SID = it->first;
			const Sound *const S = it->second;

			if (IsInsideAcousticalRange(Cli.CameraGeo, S->Source)) {
				if (Cli.SoundsListening.find(SID) == Cli.SoundsListening.end()) {
					Cli.SoundsListening.insert(SID);

					NewSoundInfo *NSI = new NewSoundInfo;

					NSI->PathCode = S->PathCode;
					NSI->SoundUID = S->SoundUID;
					NSI->Priority = S->Priority;
					NSI->PlayMode = S->PlayMode;

					Data << *NSI;
					delete NSI;
					++NSICounter;
				}
			}
	}

	//seek to the place reserved for counter and put it there
	Data.seek(NSICounterPos);
	_SPARKLET_ASSERT(NSICounter < 256);
	Data.putByte(NSICounter);
	//seek back to the end of the data
	Data.seek_end();

	//remember this position and skip one byte, this is the place where we will put
	//the counter eventually
	const int ASICounterPos = Data.getpos();
	Data.skip(sizeof(UCHAR));
	int ASICounter = 0;

	//put the updates for sounds the client is listening to
	for (std::map<const OUIDInfo, const Sound *const>::iterator it = SoundListPermanent.begin();
		it != SoundListPermanent.end() && ASICounter < 255;
		++it) {
			const OUIDInfo &SID = it->first;
			const Sound *const S = it->second;

			if (IsInsideAcousticalRange(Cli.CameraGeo, S->Source)) {
				if (Cli.SoundsListening.find(SID) != Cli.SoundsListening.end()) {
					AdjustSoundInfo *ASI = new AdjustSoundInfo;

					ASI->SoundUID = S->SoundUID;

					UCHAR abs_volume;
					CalcPanAndVolume(Cli.CameraGeo, S->Source, ASI->Pan, abs_volume);
					ASI->Volume = static_cast<UCHAR>(abs_volume * (S->InitVolume / 255.f));

					Data << *ASI;
					delete ASI;
					++ASICounter;
				}
			}
	}

	//seek to the place reserved for counter and put it there
	Data.seek(ASICounterPos);
	_SPARKLET_ASSERT(ASICounter < 256);
	Data.putByte(ASICounter);
	//seek back to the end of the data
	Data.seek_end();

	std::stack<set<OUIDInfo>::iterator> ToRemove;
	for (set<OUIDInfo>::iterator it = Cli.SoundsListening.begin(); it != Cli.SoundsListening.end(); ++it) {
		//remove permanent sounds from the clients that don't exist any more
		if (SoundListPermanent.find(*it) == SoundListPermanent.end())
			ToRemove.push(it);
		//remove permanent sounds from the clients that are out of the client's reach
		else if (!IsInsideAcousticalRange(Cli.CameraGeo, SoundListPermanent[*it]->Source))
			ToRemove.push(it);
	}

	while (!ToRemove.empty()) {
		Cli.SoundsListening.erase(ToRemove.top());
		ToRemove.pop();
	}
}


inline bool NetworkServer::IsInsideAcousticalRange(const Point<float> &CameraGeo, const Point<float> &Source) {
	return abs(CameraGeo - Source) <= ACOUSTICAL_RANGE;
}


void NetworkServer::client_ping_result(int client_id, int ping_value) {
	std::map<int, Client>::iterator it = ClientList.find(client_id);
	if (it == ClientList.end())
		return;

	Client &Cli = it->second;

	Cli.LastPingValue = ping_value;

	Cli.SkippedPingResultDisplay++;
	if (Cli.SkippedPingResultDisplay == 5) {
		Cli.SkippedPingResultDisplay = 0;
		std::ostringstream ss;
		ss << "Ping results: client_id=" << client_id << ", ping=" << ping_value << "ms";
		_SAY(ss.str());
	}
}


bool NetworkServer::get_server_info(address_c &, buffer_c &custom, buffer_c &answer) {
	ServerInfo SI;
	SI.GameType = gGameOpt.GT;
	SI.Name = SrvOpt.Name;
	SI.PlayerLimit = SrvOpt.MaxClients;
	SI.PlayersConnected = ClientList.size();
	SI.ProtocolVersion = NSB_NET_PROTOCOL_VER;
	SI.Port = SrvOpt.Port;
	SI.NetRatio = SrvOpt.NetRatio;
	SI.RequirePassword = !SrvOpt.Password.empty();

	const std::string MapFilename = FixFilenameSlashes(DATA_PREFIX + gGameOpt.MapFileNameList[gGameOpt.CurrentMap]);
	XMLParser Parser(MapFilename);
	const TiXmlElement *Root = Parser.GetRoot();
	SI.MapName = XMLParser::GetValueS(Root, "name", "");

	answer << SI;

#ifdef ENABLE_SERVER_LOG_OBTAINING
	/* This is the way to retrive logfile from sparklet server. Is it a security issue? Is there anything
	 * in logfile that's secret? The answer is no.
	 */
	if (custom.size() == 1) {
		if (custom.getByte() == 0) {
			FILE *f = fopen("sparklet.log", "r");
			if (!f)
				return true; //still return true

			buffer_c LogFile;
			for (int c = fgetc(f); c != EOF; c = fgetc(f))
				LogFile.putByte(c);
			fclose(f);

			LogFile.seek(0);
			LogFile.zip(answer);
		}
	}
#endif

	return true;
}



void NetworkServer::ProcessPollResults() {
	PollResults PR;
	std::string KickedPlayerName;

	CurrentPoll.GetResults(PR);

	if (PR.No < PR.Yes) {
		if (PR.PT == POLL_KICK_PLAYER) {
			if (ClientList.find(PR.arg) == ClientList.end())
				return; //player left the game in the meatime

			/* Remember kicked player name because the client will be
			 * removed from the list. */
			KickedPlayerName = get_element(ClientList, PR.arg).Name;

			_SAY("Players voted to kick a player from the game.");
			buffer_c bye;
			bye.putByte(NSB_CLI_DISCONNECTED_KICK_VOTED);
			zigserver_c::disconnect_client(PR.arg, bye);
		}
		else {
			_SAY("Players voted to change the map.");
			MatchOver = true;
			gGameOpt.NextMap = PR.argstr;
		}
	}

	ServerMessageInfo SMI0, SMI1, SMI2;
	std::ostringstream oss1, oss2;
	if (PR.EndedEarly)
	oss1 << "Poll ended earlier because enought players had voted.";
	SMI0.Text = oss1.str();
	oss2 << "Vote results: " << PR.Yes << " yes, " << PR.No << " no.";
	SMI1.Text = oss2.str();

	if (PR.PT == POLL_KICK_PLAYER) {
		if (PR.Yes > PR.No)
			SMI2.Text = KickedPlayerName + " has been kicked!";
		else
			SMI2.Text = KickedPlayerName + " has not been kicked!";
	}
	else {
		if (PR.Yes > PR.No)
			SMI2.Text = "Changing map to '" + PR.argstr + "'...";
		else
			SMI2.Text = "Staying on the current map.";
	}

	buffer_c buf;
	buf << SMI0;
	BroadcastMessage(buf);
	buf.clear();
	buf << SMI1;
	BroadcastMessage(buf);
	buf.clear();
	buf << SMI2;
	BroadcastMessage(buf);
}


void NetworkServer::Update() {
	if (!Running)
		return;

	if (CurrentPoll.Tick()) {
		ProcessPollResults();
	}

	bool BroadcastFrameNow = false;
	//check whether it is the time to broadcast frame
	if (BroadcastFrameSkippedCount == 0) {
		BroadcastFrameSkippedCount = SrvOpt.NetRatio;
		BroadcastFrameNow = true;
	}
	BroadcastFrameSkippedCount--;

	DoObjectRemoval();

	lObjectList.clear();
	lPlayerObjList.clear();
	lPhysicalObjList.clear();
	lFloatingObjList.clear();

	lObjectList.insert(ObjectList.begin(), ObjectList.end());
	lPlayerObjList.insert(PlayerObjList.begin(), PlayerObjList.end());
	lPhysicalObjList.insert(PhysicalObjList.begin(), PhysicalObjList.end());
	lFloatingObjList.insert(FloatingObjList.begin(), FloatingObjList.end());

	for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
		x->second.lCameraOwner = x->second.CameraOwner;

	if (BroadcastFrameNow) {
		for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
			queue<set<INPUT_COMMAND> > &Keys = (*x).second.KeysUp;
			while (!Keys.empty()) {
				ControlledObject *CO = (*x).second.ControlledObjList.back();
				CO->OnKeyUp(Keys.front());
				Keys.pop();
			}
		}
	}

	for (MAP_OBJ_ITER Obj = lObjectList.begin(); Obj != lObjectList.end(); Obj++)
		Obj->second->OnUpdate();

	CheckCollisions();

	for (MAP_PLA_OBJ_ITER PlaObj = lPlayerObjList.begin(); PlaObj != lPlayerObjList.end(); ++PlaObj) {
		for (MAP_FLO_OBJ_ITER FloObj = lFloatingObjList.begin(); FloObj != lFloatingObjList.end(); ++FloObj) {
			if (PlaObj->second->IsOver(FloObj->second))
				FloObj->second->OnOver(PlaObj->second);
		}
	}

	if (!MatchOver)
		MatchOver = CheckForWinner();
	else {
		TimeToWaitBeforeStartNewMatch -= gTimeElapsed;
		if (TimeToWaitBeforeStartNewMatch < 0) {
			TimeToWaitBeforeStartNewMatch = TIME_BETWEEN_MATCHES;
			MatchOver = !StartNewMatch();
		}
	}

	if (BroadcastFrameNow) {
		for (MAP_OBJ_ITER Obj = lObjectList.begin(); Obj != lObjectList.end(); Obj++)
			Obj->second->CheckData();

		ProcessOutgoingGameStateMsg();
		ProcessOutgoingGameDataMsg();

		//Clear list of non-permanent sounds
		for (USHORT x = 0; x < SoundList.size(); x++)
			delete SoundList[x];
		SoundList.clear();
	}
}


void NetworkServer::DoObjectRemoval() {
	while (!ObjectToKill.empty()) {
		if (ObjectList.find(ObjectToKill.top()) != ObjectList.end()) {
			for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
				ReleaseCamera(x->first, ObjectToKill.top());
			delete ObjectList.find(ObjectToKill.top())->second;
			ObjectList.erase(ObjectToKill.top());
		}
		ObjectToKill.pop();
	}
}


void NetworkServer::CheckCollisions() {
	/*for (MAP_PHY_OBJ_ITER Obj1 = lPhysicalObjList.begin(); Obj1 != lPhysicalObjList.end(); ++Obj1) {
		for (MAP_PHY_OBJ_ITER Obj2 = lPhysicalObjList.begin(); Obj2 != lPhysicalObjList.end(); ++Obj2) {
			if (Obj1->first != Obj2->first && Obj1->second->AreColliding(Obj2->second))
				Obj1->second->OnCollide(Obj2->second);
		}
	}*/
	const USHORT crls = CollisionRectList.size();

	for (MAP_PHY_OBJ_ITER Obj = lPhysicalObjList.begin(); Obj != lPhysicalObjList.end(); ++Obj) {
		const Rect<float> &ObjRect = Obj->second->GetGeo();
		int begin = (USHORT)ObjRect.y / QuadSize * QuadsPerW + (USHORT)ObjRect.x / QuadSize;
		USHORT end = ((USHORT)(ObjRect.y + ObjRect.Height) / QuadSize + 1) * QuadsPerW;

		if (begin < 0)
			begin = 0;

		if (end > crls)
			end = crls;

		for (USHORT r = begin; r < end; ++r) {
			const Rect<float> *const Quad = CollisionRectList[r]->Quad;
			if (CheckRectCollision(Quad->x, Quad->y, Quad->Width, Quad->Height,
				ObjRect.x, ObjRect.y, ObjRect.Width, ObjRect.Height)) {
					CollisionRectList[r]->List.push_back(Obj->second);
			}
		}
	}

	for (USHORT x = 0; x < CollisionRectList.size(); ++x) {
		CloseObjects *const CloseObj = CollisionRectList[x];
		for (USHORT y = 0; y < CloseObj->List.size(); ++y) {
			PhysicalObject *const first = CloseObj->List[y];
			for (USHORT z = y; z < CloseObj->List.size(); ++z) {
				PhysicalObject *const second = CloseObj->List[z];
				if (y != z && first->AreColliding(second)) {
					first->OnCollide(second);
					second->OnCollide(first);
				}
			}
		}
		CloseObj->List.clear();
	}
}


bool NetworkServer::StartNewMatch() {
	buffer_c ReconnectAgain;
	ReconnectAgain.putByte(NSB_CLI_DISCONNECTED_GAME_ENDED);

	/* We need a copy of a list of clients ID because disconnect_client() calls the client_disconnected() callback
	 * immediately where ClientList is modified. so we can't loop thought it. */
	int *ClientID = new int[ClientList.size()];
	int i = 0, s;
	for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x, ++i)
		ClientID[i] = x->first;

	// disconnect each client, telling them to reconect again
	s = i;
	for (i = 0; i < s; ++i)
		zigserver_c::disconnect_client(ClientID[i], ReconnectAgain);

	delete[] ClientID;

	if (!gGameOpt.NextMap.empty()) {
		for (USHORT x = 0; x < gGameOpt.MapFileNameList.size(); ++x) {
			const std::string Path = FixFilenameSlashes(DATA_PREFIX + gGameOpt.MapFileNameList[x]);
			XMLParser Parser(Path);
			const TiXmlElement *Root = Parser.GetRoot();
			if (XMLParser::GetValueS(Root, "name", "") == gGameOpt.NextMap) {
				gGameOpt.CurrentMap = x;
				break;
			}
		}
		gGameOpt.NextMap.clear();
	}
	else {
		++gGameOpt.CurrentMap;
		if (gGameOpt.CurrentMap >= (int)gGameOpt.MapFileNameList.size())
			gGameOpt.CurrentMap = 0;
	}

	Stop();
	Start(SrvOpt, gGameOpt);

	return true;
}


void NetworkServer::DestroyWorld() {
	for (MAP_OBJ_ITER Obj = ObjectList.begin(); Obj != ObjectList.end(); Obj++)
		delete Obj->second;

	ObjectList.clear();
	PlayerObjList.clear();
	PhysicalObjList.clear();
	FloatingObjList.clear();

	lObjectList.clear();
	lPlayerObjList.clear();
	lPhysicalObjList.clear();
	lFloatingObjList.clear();

	while (!ObjectToKill.empty())
		ObjectToKill.pop();

	TeamPointsList.clear();
}


bool NetworkServer::CreateWorld(const std::string &MapFilename) {
	try {
		XMLParser Parser(std::string(DATA_PREFIX) + MapFilename);
		const TiXmlElement *Root = Parser.GetRoot();
		if (!Root)
			return false;

		gGameOpt.MediumViscosity = XMLParser::GetValueL(Root, VISCOSITY_STR, 10);

		const USHORT ArenaSizeWidth = XMLParser::GetValueL(Root, W_STR, 0);
		const USHORT ArenaSizeHeight = XMLParser::GetValueL(Root, H_STR, 0);
		_SPARKLET_ASSERT(ArenaSizeHeight && ArenaSizeWidth);
		ArenaSize = Size(ArenaSizeWidth, ArenaSizeHeight);

		Size ArenaSizeRounded(ArenaSize);
		//round to next multiple of QuadSize if needed
		if (ArenaSize.Height % QuadSize || ArenaSize.Width % QuadSize) {
			ArenaSizeRounded.Width = ArenaSize.Width - (ArenaSize.Width % QuadSize) + QuadSize;
			ArenaSizeRounded.Height = ArenaSize.Height - (ArenaSize.Height % QuadSize) + QuadSize;
		}

		QuadsPerW = ArenaSizeRounded.Width / QuadSize;
		USHORT QuadsPerH = ArenaSizeRounded.Height / QuadSize;

		CollisionRectList.reserve(QuadsPerW * QuadsPerH);
		for (USHORT y = 0; y < QuadsPerH; ++y) {
			for (USHORT x = 0; x < QuadsPerW; ++x) {
				CollisionRectList.push_back(new CloseObjects(new Rect<float>(x * QuadSize, y * QuadSize, QuadSize, QuadSize)));
			}
		}

		const TiXmlElementList* Att = XMLParser::GetAllChildren(Root, "*");
		for (USHORT x = 0; x < Att->size(); ++x) {
			const Object* Obj = gInitializer->CreateObject(Att->at(x));

			if (!Obj) {
				_ERROR_;
				_DISPLAY(XMLParser::GetName(Att->at(x)));
			}
		}
		delete Att;

		return true;
	} catch (const ExSparkletException &ese) {
		_ERROR_;
		_SAY(ese.GetMsg());
		return false;
	}

}


OUIDInfo NetworkServer::Add(Object *Obj) {
	++CurrentUID;
	ObjectList.insert(ObjectList.end(), make_pair(CurrentUID, Obj));

	Obj->SetArenaSize(ArenaSize);

	return CurrentUID;
}


void NetworkServer::Add(PhysicalObject *Obj, OUIDInfo ObjectUID) {
	PhysicalObjList.insert(PhysicalObjList.end(), make_pair(ObjectUID, Obj));
}


void NetworkServer::Add(PlayerObject *Obj, OUIDInfo ObjectUID) {
	PlayerObjList.insert(PlayerObjList.end(), make_pair(ObjectUID, Obj));
}


void NetworkServer::Add(FloatingObject *Obj, OUIDInfo ObjectUID) {
	FloatingObjList.insert(FloatingObjList.end(), make_pair(ObjectUID, Obj));
}


void NetworkServer::SetCamera(int ClientID, OUIDInfo CameraOwner) {
	get_element(ClientList, ClientID).CameraOwner.push_back(CameraOwner);
}


void NetworkServer::ReleaseCamera(int ClientID, OUIDInfo CameraOwner) {
	get_element(ClientList, ClientID).CameraOwner.remove(CameraOwner);

	//_SPARKLET_ASSERT(get_element(ClientList, ClientID).CameraOwner.size());
}


void NetworkServer::SetInputObject(int ClientID, ControlledObject *InputObject) {
	if (get_element(ClientList, ClientID).ControlledObjList.size())
		get_element(ClientList, ClientID).ControlledObjList.back()->LostControl();

	get_element(ClientList, ClientID).ControlledObjList.push_back(InputObject);

	InputObject->GotControl();
}


void NetworkServer::ReleaseInputObject(int ClientID, ControlledObject *InputObject) {
	InputObject->LostControl();

	get_element(ClientList, ClientID).ControlledObjList.remove(InputObject);

	if (get_element(ClientList, ClientID).ControlledObjList.size())
		get_element(ClientList, ClientID).ControlledObjList.back()->GotControl();
}


void NetworkServer::RemoveObject(OUIDInfo ObjectUID) {
	ObjectToKill.push(ObjectUID);
}


void NetworkServer::RemovePhysicalObj(OUIDInfo ObjectUID) {
	PhysicalObjList.erase(ObjectUID);
}


void NetworkServer::RemovePlayerObj(OUIDInfo ObjectUID) {
	PlayerObjList.erase(ObjectUID);
}


void NetworkServer::RemoveFloatingObj(OUIDInfo ObjectUID) {
	FloatingObjList.erase(ObjectUID);
}


OUIDInfo NetworkServer::CreatePlayerObject(ClientInfo *CI, int ClientID) {
	try {
		XMLParser Parser(string(DATA_PREFIX) + CI->ShipPath);
		const TiXmlElement *Root = Parser.GetRoot();

		if (!Root || XMLParser::GetName(Root) != PLAYER_STR) {
			_ERROR_;
			return OUIDInfo(0);
		}

		// No need to bother destroying Player object if exception happens. Class' Object
		// contructor is the only place where an exception can occur.
		Player *New = new Player(Root, ClientID);

		New->SetName(CI->Name);
		New->Identity.ClientID = ClientID;
		New->Identity.Team = CI->Team;
		New->Identity.OwnerObjUID = New->GetObjUID();

		return New->GetObjUID();
	}
	catch (const ExSparkletException &ese) {
		_ERROR_;
		return OUIDInfo(0);
	}
}


void NetworkServer::SomeoneGotKilled(const IDENTITY *Who, const IDENTITY *Whose) {
	GameMessage GM;

	if (gGameOpt.GT == GT_TEAM_DEATHMATCH) {
		//plain suicide - the player killed himself by shooting
		if (Who->OwnerObjUID == Whose->OwnerObjUID) {
			GM.WhoKilled = Who->ClientID;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = true;
			gNetworkServer->OnTeamSuicide(Whose);
		}
		//indirect suicide - the player killed himself using 3rd object eg. a wall
		else if (Who->Team == TEAM_NO_TEAM) {
			GM.WhoKilled = 0;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = false;
			gNetworkServer->OnTeamSuicide(Whose);
		}
		//team mate homocide - someone from the same team killed the player
		//other team homocide - someone from other team killed the player
		else {
			//in some strange cases (e.g. the player shoots, disconnects, and kills someone, in that
			//order) it's perfectly right that the client doesn't exist
			if (ClientList.find(Who->ClientID) == ClientList.end())
				return;

			GM.WhoKilled = Who->ClientID;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = true;
			gNetworkServer->OnTeamHomocide(Who, Whose);
		}
	}
	else if (gGameOpt.GT == GT_DEATHMATCH) {
		//plain suicide
		if (Who->OwnerObjUID == Whose->OwnerObjUID) {
			GM.WhoKilled = Who->ClientID;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = true;
			gNetworkServer->OnSuicide(Whose);
		}
		else if (Who->OwnerObjUID == 0) {
			GM.WhoKilled = 0;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = false;
			gNetworkServer->OnSuicide(Whose);
		}
		//plain homocide
		else {
			//in some strange cases (e.g. the player shoots, disconnects, and kills someone, in that
			//order) it's perfectly right that the client doesn't exist
			if (ClientList.find(Who->ClientID) == ClientList.end())
				return;

			GM.WhoKilled = Who->ClientID;
			GM.WhoWasKilled = Whose->ClientID;
			GM.KilledWithWeapon = true;
			gNetworkServer->OnHomocide(Who, Whose);
		}
	}

	buffer_c Data;
	Data << GM;
	BroadcastMessage(Data);
}


void NetworkServer::OnTeamHomocide(const IDENTITY *Who, const IDENTITY *Whose) {
	//team mate homocide
	if (Who->Team == Whose->Team) {
		get_element(ClientList, Who->ClientID).Points--;
		get_element(TeamPointsList, Who->Team)--;
	}
	else { //other team homocide
		get_element(ClientList, Who->ClientID).Points++;
		get_element(TeamPointsList, Who->Team)++;
	}

	ScoreChanged = true;

}


void NetworkServer::OnTeamSuicide(const IDENTITY *Who) {
	if (ClientList.find(Who->ClientID) == ClientList.end()) {
		_ERROR_;
	}

	get_element(ClientList, Who->ClientID).Points--;
	get_element(TeamPointsList, Who->Team)--;
	ScoreChanged = true;
}


void NetworkServer::OnHomocide(const IDENTITY *Who, const IDENTITY *Whose) {
	get_element(ClientList, Who->ClientID).Points++;
	ScoreChanged = true;
}


void NetworkServer::OnSuicide(const IDENTITY *Who) {
	if (ClientList.find(Who->ClientID) == ClientList.end()) {
		_ERROR_;
	}

	get_element(ClientList, Who->ClientID).Points--;
	ScoreChanged = true;
}


//put all players and teams score
void NetworkServer::PutScore(buffer_c &Data) {
	if (gGameOpt.GT == GT_DEATHMATCH) {
		ScoreInfo SI;
		for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
			SI.push_back(x->second.Points);
		Data << SI;
	}
	else if (gGameOpt.GT == GT_TEAM_DEATHMATCH) {
		ScoreInfo SI;
		for (map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
			SI.push_back(x->second.Points);
		Data << SI;

		ScoreInfo SITeam;
		for (map<TEAM, char>::iterator x = TeamPointsList.begin(); x != TeamPointsList.end(); ++x)
			SITeam.push_back(x->second);
		Data << SITeam;
	}
}


//put camera owner object ID if it changed
void NetworkServer::PutCameraInfo(Client &Cli, buffer_c &Data) {
	OUIDInfo CameraOwner = Cli.lCameraOwner.back();

	if (CameraOwner != Cli.CameraOwnerSent) {
		Data << (UCHAR)1 << CameraOwner << Cli.MainObjectUID;
		Cli.CameraOwnerSent = CameraOwner;
	}
	else
		Data << (UCHAR)0;
}


bool NetworkServer::CheckForWinner() {
	if (gGameOpt.GT == GT_DEATHMATCH) {
		for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
			if (x->second.Points >= gGameOpt.PointsMax) {
				HonorTheWinner(x->first);
				return true;
			}
		}
	}
	else if (gGameOpt.GT == GT_TEAM_DEATHMATCH) {
		TEAM WinningTeam = TEAM_NO_TEAM;
		for (map<TEAM, char>::iterator x = TeamPointsList.begin(); x != TeamPointsList.end(); ++x) {
			if (x->second >= gGameOpt.PointsMax)
				WinningTeam = x->first;
		}

		if (WinningTeam == TEAM_NO_TEAM)
			return false;

		/* one team won */
	    /* now find a player from that team with most score and honor him */
		int WinnerCliID;
		char MostPoints = 0;
		for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
			if (x->second.Team == WinningTeam) {
				if (MostPoints <= x->second.Points) {
					MostPoints = x->second.Points;
					WinnerCliID = x->first;
				}
			}
		}

		HonorTheWinner(WinnerCliID);

		return true;
	}

	return false;
}


void NetworkServer::HonorTheWinner(const int &CliID) {
	const OUIDInfo &WinnerObjectUID = get_element(ClientList, CliID).MainObjectUID;

	for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x)
		SetCamera(x->first, WinnerObjectUID);

	WinnerInfo WI;
	WI.ClientID = CliID;
	buffer_c Data;
	Data << WI;
	BroadcastMessage(Data);
}


int NetworkServer::FindClientByName(const std::string &Name) {
	for (std::map<int, Client>::iterator it = ClientList.begin(); it != ClientList.end(); ++it) {
		if (CaseInsensitiveMatch(it->second.Name, Name))
			return it->first;
	}

	_ERROR_;
	throw EX_INVALID_DATA;
}


void NetworkServer::BroadcastMessage(buffer_c &data) {
	for (std::map<int, Client>::iterator it = ClientList.begin(); it != ClientList.end(); ++it)
		zigserver_c::send_message(it->first, data, it->second.InfoMsgStream, true);
}


const std::string NetworkServer::UnambigouizePlayerName(const std::string &Name) {
	std::string NewName = Name;
again:
	for (std::map<int, Client>::iterator x = ClientList.begin(); x != ClientList.end(); ++x) {
		if (CaseInsensitiveMatch(NewName, x->second.Name)) {
			NewName += "_";
			goto again;
		}
	}

	return NewName;
}


