/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "network_messages.h"

#include <bitset>

#ifdef WIN32
	#include "sparklet_utils.h"
#endif

#include <cassert>


ZIG_SERIALIZABLE_CLASS_CPP(KeySet)

void KeySet::write(buffer_c &out) const {
	//out.putByte(std::set<INPUT_COMMAND>::size());
	for (std::set<INPUT_COMMAND>::const_iterator x = std::set<INPUT_COMMAND>::begin(); x != std::set<INPUT_COMMAND>::end(); ++x)
		out << static_cast<UCHAR>(*x);
}


void KeySet::read(buffer_c &in) {
	/*const unsigned char Size = in.getByte();
	for (unsigned char x = 0; x < Size; x++)
		std::set<INPUT_COMMAND>::insert(static_cast<INPUT_COMMAND>(in.getByte()));*/

	if (in.size_left() > 128) {
		_ERROR_;
		return;
	}

	while (in.size_left())
		std::set<INPUT_COMMAND>::insert(static_cast<INPUT_COMMAND>(in.getByte()));
}



ZIG_SERIALIZABLE_CLASS_CPP(OUIDInfo)

/*
This is the limit value for object unique ID (OUID). It is the result from using
buffer_c::get255Ms() and buffer_c::set255Ms() functions which are limited to -+256M.
Since OUID is unsinged this limit is doubled.
*/
#define LIMIT_512M 536870910


void OUIDInfo::write(buffer_c &out) const {
	out.put256Ms(OUID - LIMIT_512M / 2);
}


void OUIDInfo::read(buffer_c &in) {
	OUID = in.get256Ms() + LIMIT_512M / 2;
}


OUIDInfo& OUIDInfo::operator ++() {
	OUID++;

	if (OUID == LIMIT_512M) {
		_ERROR_;
		_SAY("Internal limitation reached, will crash probably!!!");
		//such a nice place to throw an exception
	}

	return *this;
}


ZIG_SERIALIZABLE_CLASS_CPP(ClientInfo)

void ClientInfo::write(buffer_c &out) const {
	out << ShipPath << Name << static_cast<UCHAR>(Team);
}


void ClientInfo::read(buffer_c &in) {
	in >> ShipPath >> Name;
	Team = static_cast<TEAM>(in.getByte());
}



//2^15 / PI / 2
#define ANGF 5210.f

ZIG_SERIALIZABLE_CLASS_CPP(VelocityInfo)

void VelocityInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(abs(Velocity) < 3276.f);

	out.put32K(lroundf(abs(Velocity) * 10.f));
	out.put32K(lroundf(arg(Velocity) * ANGF) + 16384);

	_SPARKLET_ASSERT(Loc.x < 16384);
	_SPARKLET_ASSERT(Loc.y < 16384);
	out.put32K(lroundf(Loc.x) + 16384);
	out.put32K(lroundf(Loc.y) + 16384);
}


void VelocityInfo::read(buffer_c &in) {
	const USHORT lng = in.get32K();
	const USHORT ang = in.get32K();
	Velocity = polar(lng / 10.f, (ang - 16384.f) / ANGF);

	Loc.x = in.get32K() - 16384;
	Loc.y = in.get32K() - 16384;
}



ZIG_SERIALIZABLE_CLASS_CPP(FullVelocityInfo)

void FullVelocityInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(abs(Velocity) < 3276.f);

	out.put32K(lroundf(abs(Velocity) * 10.f));
	out.put32K(lroundf(arg(Velocity) * ANGF) + 16384);

	if (abs(Impulse) > 65535.f)
		out.putShort(65535);
	else
		out.putShort(lroundf(abs(Impulse)));

	out.put32K(lroundf(arg(Impulse) * ANGF) + 16384);

	_SPARKLET_ASSERT(Loc.x < 16384);
	_SPARKLET_ASSERT(Loc.y < 16384);
	out.put32K(lroundf(Loc.x) + 16384);
	out.put32K(lroundf(Loc.y) + 16384);
}


void FullVelocityInfo::read(buffer_c &in) {
	const USHORT lng1 = in.get32K();
	const USHORT ang1 = in.get32K();
	Velocity = polar(lng1 / 10.f, (ang1 - 16384.f) / ANGF);

	const USHORT lng2 = in.getShort();
	const USHORT ang2 = in.get32K();
	Impulse = polar((float)lng2, (ang2 - 16384.f) / ANGF);

	Loc.x = in.get32K() - 16384;
	Loc.y = in.get32K() - 16384;
}



ZIG_SERIALIZABLE_CLASS_CPP(RotationInfo)

void RotationInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(abs(Direction) <= M_PI);
	_SPARKLET_ASSERT(abs(Rotation) < 3.27f);
	out.putShorts(lroundf(Direction * 10000));
	out.putShorts(lroundf(Rotation * 10000));
}


void RotationInfo::read(buffer_c &in) {
	Direction = in.getShorts() / 10000.f;
	Rotation = in.getShorts() / 10000.f;
}



ZIG_SERIALIZABLE_CLASS_CPP(FullRotationInfo)

void FullRotationInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(abs(Direction) <= M_PI);
	_SPARKLET_ASSERT(abs(Rotation) < 6.55f);
	_SPARKLET_ASSERT(abs(lroundf(Torque)) < 16384);

	out.putShorts(lroundf(Direction * 10000));
	out.putShorts(lroundf(Rotation * 5000));
	out.put32K(lroundf(Torque) + 16384);
}


void FullRotationInfo::read(buffer_c &in) {
	Direction = in.getShorts() / 10000.f;
	Rotation = in.getShorts() / 5000.f;
	Torque = in.get32K() - 16384.f;
}



ZIG_SERIALIZABLE_CLASS_CPP(ParticleInfo)

void ParticleInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(LifetimeLeft <= 2.5);
	out.putByte(lroundf(LifetimeLeft * 100.f));
	out << VI;
}


void ParticleInfo::read(buffer_c &in) {
	LifetimeLeft = in.getByte() / 100.f;
	VelocityInfo *MItmp = static_cast<VelocityInfo*>(in.getObject());
	VI = *MItmp;
	delete MItmp;
}



ZIG_SERIALIZABLE_CLASS_CPP(FullObjInfo)

void FullObjInfo::write(buffer_c &out) const {
	out.put32K(XMLFilePathCode);
	out.put32K(Surface);
	out.put32K(Dim.Width);
	out.put32K(Dim.Height);

	_SPARKLET_ASSERT(lroundf(1.f / Scale) < 256);
	_SPARKLET_ASSERT(lroundf(1.f / Scale) > 0);
	out.putByte(lroundf(1.f / Scale));
}


void FullObjInfo::read(buffer_c &in) {
	XMLFilePathCode = in.get32K();
	Surface = in.get32K();
	Dim.Width = in.get32K();
	Dim.Height = in.get32K();

	Scale = 1.f / in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(EngineInfo)


void EngineInfo::write(buffer_c &out) const {
	std::bitset<8> Info;

	_SPARKLET_ASSERT(Count <= 8);

	for (USHORT x = 0; x < Count; ++x)
		Info[x] = Engine[x];

	out.putByte(Info.to_ulong());
}


void EngineInfo::read(buffer_c &in) {
	const UCHAR Val = in.getByte();
	Count = 8;
	std::bitset<8> Info(Val);

	Engine = new bool[Count];

	for (USHORT x = 0; x < Count; ++x)
		Engine[x] = Info[x];
}


bool EngineInfo::operator!=(const EngineInfo &EI) {
	bool RetVal = false;

	for (USHORT x = 0; x < Count; ++x)
		RetVal |= (Engine[x] != EI.Engine[x]);

	return RetVal;
}



ZIG_SERIALIZABLE_CLASS_CPP(WeaponInfo)


void WeaponInfo::write(buffer_c &out) const {
	_SPARKLET_ASSERT(Index < 16);
	_SPARKLET_ASSERT(Ammo < 4096);
	_SPARKLET_ASSERT(BarStatus < 4);
	_SPARKLET_ASSERT(WStatus < 4);
	_SPARKLET_ASSERT(4 == sizeof(ULONG));

	ULONG Info = 0;

	Info |= Index << (32 - 4); //28
	Info |= Ammo << (32 - 4 - 12); //16
	Info |= BarStatus << (32 - 4 - 12 - 2); //14
	Info |= WStatus << (32 - 4 - 12 - 2 - 2); //12

	out.putLong(Info);
}


void WeaponInfo::read(buffer_c &in) {
	ULONG Info = in.getLong();

	Index = Info >> (32 - 4) & 0xf;
	Ammo = Info >> (32 - 4 - 12) & 0xfff;
	BarStatus = Info >> (32 - 4 - 12 - 2) & 0x3;
	WStatus = Info >> (32 - 4 - 12 - 2 - 2) & 0x3;
}


bool WeaponInfo::operator!=(const WeaponInfo &WI) {
	return (Index != WI.Index)
		|| (Ammo != WI.Ammo)
		|| (BarStatus != WI.BarStatus)
		|| (WStatus != WI.WStatus);
}



ZIG_SERIALIZABLE_CLASS_CPP(ShipState)


void ShipState::write(buffer_c &out) const {
	out << Energy << ChassisFrame;
}


void ShipState::read(buffer_c &in) {
	in >> Energy >> ChassisFrame;
}


bool ShipState::operator!=(const ShipState &SS) {
	return (Energy != SS.Energy)
		|| (ChassisFrame != SS.ChassisFrame);
}



ZIG_SERIALIZABLE_CLASS_CPP(ShipInfo)


void ShipInfo::write(buffer_c &out) const {
	out << Name << static_cast<char>(Team);
}


void ShipInfo::read(buffer_c &in) {
	in >> Name;
	Team = static_cast<TEAM>(in.getByte());
}


bool ShipInfo::operator!=(const ShipInfo &SI) {
	return (Name != SI.Name)
		|| (Team != SI.Team);
}



ZIG_SERIALIZABLE_CLASS_CPP(AspectInfo)


void AspectInfo::write(buffer_c &out) const {
	out.putByte(Visible);
}


void AspectInfo::read(buffer_c &in) {
	Visible = in.getByte();
}


bool AspectInfo::operator!=(const AspectInfo &AI) {
	return (Visible != AI.Visible);
}



ZIG_SERIALIZABLE_CLASS_CPP(CodedPaths)

void CodedPaths::write(buffer_c &out) const {
	for (std::map<USHORT, std::string>::const_iterator x = begin(); x != end(); ++x) {
		out.put32K(x->first);
		out.putString(x->second);
	}
}


void CodedPaths::read(buffer_c &in) {
	while (in.size_left()) {
		const USHORT x = in.get32K();
		const std::string s = in.getString();
		std::map<USHORT, std::string>::insert(make_pair(x, s));
	}
}



ZIG_SERIALIZABLE_CLASS_CPP(NewPlayerInfo)

void NewPlayerInfo::write(buffer_c &out) const {
	out << Name;
	out.put32K(UID);
	out << (UCHAR)Team;
}


void NewPlayerInfo::read(buffer_c &in) {
	in >> Name;
	UID = in.get32K();
	Team = (TEAM)in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(PlayerGoneInfo)

void PlayerGoneInfo::write(buffer_c &out) const {
	out.put32K(UID);
}


void PlayerGoneInfo::read(buffer_c &in) {
	UID = in.get32K();
}



ZIG_SERIALIZABLE_CLASS_CPP(ScoreInfo)

void ScoreInfo::write(buffer_c &out) const {
	out.putByte(std::vector<char>::size());
	for (USHORT x = 0; x < std::vector<char>::size(); ++x)
		out.putBytes(std::vector<char>::operator[](x));
}


void ScoreInfo::read(buffer_c &in) {
	const UCHAR Count = in.getByte();
	for (int x = 0; x < Count; ++x)
		std::vector<char>::push_back(in.getBytes());
}



ZIG_SERIALIZABLE_CLASS_CPP(PingInfo)

void PingInfo::write(buffer_c &out) const {
	out.putByte(std::vector<USHORT>::size());
	for (USHORT x = 0; x < std::vector<USHORT>::size(); ++x)
		out.put32K(std::vector<USHORT>::operator[](x));
}


void PingInfo::read(buffer_c &in) {
	const UCHAR Count = in.getByte();
	for (int x = 0; x < Count; ++x)
		std::vector<USHORT>::push_back(in.get32K());
}


void PingInfo::push_back(USHORT p) {
		if (p > 32767) p = 32767;
		std::vector<USHORT>::push_back(p);
}



ZIG_SERIALIZABLE_CLASS_CPP(WinnerInfo)

void WinnerInfo::write(buffer_c &out) const {
	out.put32K(ClientID);
}


void WinnerInfo::read(buffer_c &in) {
	ClientID = in.get32K();
}



ZIG_SERIALIZABLE_CLASS_CPP(StateInfoGeneric1)

void StateInfoGeneric1::write(buffer_c &out) const {
	out.putByte(State);
}


void StateInfoGeneric1::read(buffer_c &in) {
	State = in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(StateInfoGeneric2)

void StateInfoGeneric2::write(buffer_c &out) const {
	out.putByte(State);
}


void StateInfoGeneric2::read(buffer_c &in) {
	State = in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(PlayerMessage)

void PlayerMessage::write(buffer_c &out) const {
	out << Text << To << From;
}


void PlayerMessage::read(buffer_c &in) {
	in >> Text >> To >> From;
}



ZIG_SERIALIZABLE_CLASS_CPP(GameMessage)

void GameMessage::write(buffer_c &out) const {
	out.put32K(WhoKilled);
	out.put32K(WhoWasKilled);
	out.putByte(KilledWithWeapon);
}


void GameMessage::read(buffer_c &in) {
	WhoKilled = in.get32K();
	WhoWasKilled = in.get32K();
	KilledWithWeapon = in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(ServerInfo)

void ServerInfo::write(buffer_c &out) const {
	out << Name << ProtocolVersion << PlayersConnected << PlayerLimit << Port << MapName << NetRatio;
	out.putByte(static_cast<UCHAR>(GameType));
	out.putByte(RequirePassword);
}


void ServerInfo::read(buffer_c &in) {
	in >> Name >> ProtocolVersion >> PlayersConnected >> PlayerLimit >> Port >> MapName >> NetRatio;
	GameType = static_cast<GAME_TYPE>(in.getByte());
	RequirePassword = in.getByte();
}



ZIG_SERIALIZABLE_CLASS_CPP(GameOptions)

void GameOptions::write(buffer_c &out) const {
	out.putByte(static_cast<UCHAR>(GT));
	out.putByte(PointsMax);
	out.putLong((NLulong)(MediumViscosity * 1000.f));
}


void GameOptions::read(buffer_c &in) {
	GT = static_cast<GAME_TYPE>(in.getByte());
	PointsMax = in.getByte();
	MediumViscosity = in.getLong() / 1000.f;
}



ZIG_SERIALIZABLE_CLASS_CPP(NewSoundInfo)

void NewSoundInfo::write(buffer_c &out) const {
	out << SoundUID << PathCode << Priority << PlayMode;
}


void NewSoundInfo::read(buffer_c &in) {
	const OUIDInfo *si = static_cast<OUIDInfo*>(in.getObject());
	SoundUID = *si;
	delete si;

	in >> PathCode >> Priority >> PlayMode;
}



ZIG_SERIALIZABLE_CLASS_CPP(NewSoundInfoUntunable)

void NewSoundInfoUntunable::write(buffer_c &out) const {
	out << SoundUID << PathCode << Volume << Pan << Priority << PlayMode;
}


void NewSoundInfoUntunable::read(buffer_c &in) {
	const OUIDInfo *si = static_cast<OUIDInfo*>(in.getObject());
	SoundUID = *si;
	delete si;

	in >> PathCode >> Volume >> Pan >> Priority >> PlayMode;
}



ZIG_SERIALIZABLE_CLASS_CPP(AdjustSoundInfo)

void AdjustSoundInfo::write(buffer_c &out) const {
	out << SoundUID << Volume << Pan;
}


void AdjustSoundInfo::read(buffer_c &in) {
	const OUIDInfo *si = static_cast<OUIDInfo*>(in.getObject());
	SoundUID = *si;
	delete si;

	in >> Volume >> Pan;
}


ZIG_SERIALIZABLE_CLASS_CPP(NewKickPollInfo)

void NewKickPollInfo::write(buffer_c &out) const {
	out << KickWho << Reason;
}


void NewKickPollInfo::read(buffer_c &in) {
	in >> KickWho >> Reason;
}


ZIG_SERIALIZABLE_CLASS_CPP(ServerRequestInfo)

void ServerRequestInfo::write(buffer_c &out) const {
	out << (UCHAR)RequestCode;
}


void ServerRequestInfo::read(buffer_c &in) {
	UCHAR Code;
	in >> Code;
	RequestCode = (SERVER_REQUEST)Code;
}


ZIG_SERIALIZABLE_CLASS_CPP(ChangeMapCommandInfo)

void ChangeMapCommandInfo::write(buffer_c &out) const {
	out << MapName;
}


void ChangeMapCommandInfo::read(buffer_c &in) {
	in >> MapName;
}


ZIG_SERIALIZABLE_CLASS_CPP(ServerMessageInfo)

void ServerMessageInfo::write(buffer_c &out) const {
	out << Text;
}


void ServerMessageInfo::read(buffer_c &in) {
	in >> Text;
}


ZIG_SERIALIZABLE_CLASS_CPP(ChangeTeamCommandInfo)

void ChangeTeamCommandInfo::write(buffer_c &out) const {
	out.putByte(Team);
	out.putLong(Who);
}


void ChangeTeamCommandInfo::read(buffer_c &in) {
	Team = (TEAM)in.getByte();
	Who = in.getLong();
}
