/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SERVER_ONLY


#ifndef NETWORK_CLIENT_H
#define NETWORK_CLIENT_H


#include <alleggl.h>
#ifdef WIN32
	#include <winalleg.h>
#endif

#include <string>
#include <vector>
#include <map>

#include "agl_bitmap.h"
#include "text_object.h"

#include "structures.h"
#include "fonts.h"

#include "network_messages.h"
#include "xmlparser.h"

#include "sound_sample.h"

#include <zig/zigclient.h>
#include <zig/buffer.h>


#define MAP_CLIOBJ_ITER std::map<OUIDInfo, CliObject*>::iterator
#define MAP_RENOBJ_ITER std::multimap<USHORT, CliObject*>::iterator


class Statusbar;
class TextObject;
class CliObject;
class ScreenMask;
class MessageLister;
class MessageWriter;
class FPSDisplay;
class Star;
class GameCmd;


struct ClientOptions {
	std::string ServerAddress;
	std::string Name;
	std::string Password;

	TEAM Team;
	std::string XMLFileShipPath;
};


class NetworkClient : private zigclient_c {
public:
	NetworkClient();
	~NetworkClient();

	bool Connect(const string &GuiPath, const ClientOptions &CO);
	void Disconnect(int reason = 0);

	void Process() {zigclient_c::process_nonblocking(); if (zigclient_c::is_connected() && Displaying) PollMessageWriter();}
	void Update();
	void PollInput(const std::set<INPUT_COMMAND> *InputCommand);

	bool HighLoad() {return zigclient_c::is_connected();}

	bool Render();

	void GetServerInfo(address_c &addr, buffer_c &req) {zigclient_c::get_server_info(addr, req);}
	void GetAllServersInfo(buffer_c &req) {zigclient_c::get_local_server_info(req);}

	void Show() {Displaying = true;}
	void Hide() {Displaying = false;}

	float GetViscosity() {return GameOpt.MediumViscosity;}

	static int NetRatio;

private:
	std::map<OUIDInfo, CliObject*> CliObjList;
	std::multimap<USHORT, CliObject*> RendObjList;
	std::multimap<USHORT, CliObject*> GuiRendObjList;

	std::vector<const Star*> StarList;

	OUIDInfo CameraOwner;
	OUIDInfo MainObject;
	Rect<float> CameraGeo;
	Size TargetCameraDim;

	bool Rendered;
	bool Displaying;
	USHORT DisplayLogoSkip;

	AGLBitmap Logo;
	TextObject *ConnectionStatusText, *GameInfoText, *GameStatusText, *GameMessageText;
	std::vector<TextObject*> TextList;
	void SetConnectionStatusText(const std::string &Text);
	void SetGameInfoText(const std::string &Text);
	void SetGameStatusText(const std::string &Text);
	void SetGameMessageText(const std::string &Text);

	bool MatchOver;

	float Uptime;
	bool WantsReconnect;
	float TimeToWaitBrforeReconect;
	bool Reconnect();
	std::string ServerAddress;

	std::map<USHORT, std::string> PathCoder;

	Statusbar *Stb;
	ScreenMask *ScrMask;
	MessageLister *MsgLister;
	MessageWriter *MsgWriter;
	FPSDisplay *Statistics;

	struct ScoreT {
		ScoreT() : PointsDisplay(MESSAGE_BIG), NameDisplay(MESSAGE_BIG), PingDisplay(VERDANA_B_8) {};
		char Points;
		UCHAR Rank;
		char Advantage;

		TextObject PointsDisplay;
		TextObject NameDisplay;
		TextObject PingDisplay;
	};
	std::map<TEAM, ScoreT> TeamScoreList;

	struct Oponent {
		ScoreT Score;
		TEAM Team;
		std::string Name;
		USHORT Ping;
	};
	std::map<int, Oponent> OponentList;

	bool DisplayingScoreTable;
	USHORT DisplayingScoreTableCounter;
	void DisplayScoreTable();

	void UpdateCamera();
	void SetCameraZoom(float Offset);

	USHORT MessageWriterActiveCounter;
	USHORT StatisticsActiveCounter;

	int DataMsgStream;
	int InfoMsgStream;

	int MyCliUID;
	TEAM MyTeam;

	GameOptions GameOpt;
	ClientOptions CliOpt;
	std::string ClientGuiPath;
	std::string Name;
	Size ArenaSize;

	bool AutoHideMsgWriter;

	int LastPacketID;

	CliObject* CreateObject(const FullObjInfo *FOI, const OUIDInfo &ObjID);
	void DestroyObject(OUIDInfo ObjUID);

	void DeliverMsg(CliObject *CO, UCHAR PendingCount, buffer_c &in);

	void ProcessInUnreliableMsg(buffer_c &in);
	void ProcessInReliableMsg();
	void ParseGamePacket(buffer_c &Data);
	void ProcessGameStateMsg(buffer_c &Data, bool &ScoreChanged);

	void ProcessOutReliableMsg();

	void DisplayGameMessage(const GameMessage* GM);

	bool InitRenderEngine();
	bool LoadGUI(const TiXmlElement *Root);

	bool Start(const std::string &GuiPath);
	void Stop();

	void BuildStars();

	CliObject* LoadObjectFromMap(const TiXmlElement *Root, const string Name);

	void OponentConnected(const NewPlayerInfo* Info);
	void OponentDisconnected(const PlayerGoneInfo* Info);

	void SomeoneWon(const WinnerInfo* WI);

	void ReadScore(const ScoreInfo *SI, buffer_c &Msg);
	void ReadPingInfo(const PingInfo *PI);

	void RemoveBogusTeams();
	void RankPlayers();

	void SendPlayerMessage(const std::string &Text);
	void SendPlayerMessage(const std::string &Text, const std::string &Name);
	void SendTeamMessage(const std::string &Text);
	void DisplayPlayerMessage(const PlayerMessage* PM);
	void PollMessageWriter();

	void DisplayServerMessage(const std::string Text);
	bool CheckForPlayer(const std::string &Name);
	void ExecuteCmd(std::vector<std::string> &WordList, const std::string &Params);
	void ChangeTeam(const ChangeTeamCommandInfo *CTCI);

	void ReadSounds(buffer_c &in);
	void PlaySound(const NewSoundInfoUntunable *NSIU);
	void PlaySound(const NewSoundInfo *NSI);
	std::map<OUIDInfo, SoundSample*>::iterator AdjustSound(const AdjustSoundInfo *ASI);
	std::map<OUIDInfo, SoundSample*> SoundSampleList;

	virtual void connected(buffer_c &hello);
	virtual void connection_timed_out();
	virtual void connection_refused(buffer_c &reason, int code_reason);
	virtual void disconnected(bool server_initiated, buffer_c &reason, int code_reason, int zig_reason);
	virtual void incoming_data(buffer_c &in, int packet_id);
	virtual void incoming_server_info(address_c &addr, buffer_c &info, int time_ms, bool timedout);

	//to be removed for ziglite
	//virtual void channel_opened(int, channel_c*) {}

	struct SameTeam : unary_function<std::pair<int, Oponent>, bool> {
		SameTeam (const TEAM &T) : Team(T) {};
		bool operator() (const std::pair<int, Oponent> &opo) const {
			return opo.second.Team == Team;
		}
	private:
		const TEAM &Team;
	};
};

std::string GetRandomBrightParticle();
std::string GetRandomRedParticle();
std::string GetRandomBlueParticle();


#endif

#endif //SERVER_ONLY
