/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SERVER_ONLY


#include <alleggl.h>


#ifdef RootWindow
	#undef RootWindow
#endif


#include "network_client.h"
#include "network_messages.h"
#include "message_coder.h"

#include "initializer.h"
#include "bitmap_server.h"
#include "sparklet_utils.h"

#include "cli_object.h"
#include "fps_display.h"
#include "statusbar.h"
#include "screen_mask.h"
#include "message_lister.h"
#include "message_writer.h"
#include "cli_player_object.h"
#include "player_object.h"

#include "exceptions.h"

#include "star.h"


#ifdef WIN32
	#include "sparklet_utils.h"
#endif

#include <stack>
#include <algorithm>

#define SURFACE_STR std::string("surface")


//client sends very few data, don't event try compressing it
#define MIN_COMPRESSION_SIZE -1


using namespace std;

NetworkClient *gNetworkClient = NULL;

extern BitmapServer *gBmpSrv;
extern Initializer *gInitializer;
extern float gTimeElapsed;


//this should be overriden by NetowrkClient::connected before used
int NetworkClient::NetRatio = 5;



NetworkClient::NetworkClient() :
	zigclient_c(gLC),
	Displaying(false),
	Logo("bitmaps/gui/logo.tga"),
	TimeToWaitBrforeReconect(3),
	AutoHideMsgWriter(true) {

	gNetworkClient = this;

	DisplayLogoSkip = 0;
	DisplayingScoreTableCounter = 0;
	MessageWriterActiveCounter = 0;
	StatisticsActiveCounter = 0;

	ConnectionStatusText = new TextObject(Point<float>(0, 0), "", MESSAGE_BIG, makeacol32(128, 128, 255, 255));
	GameInfoText = new TextObject(MESSAGE_BIG);
	GameStatusText = new TextObject(Point<float>(0, 0), "", MESSAGE_BIG, makeacol32(255, 255, 255, 255));
	GameMessageText = new TextObject(Point<float>(0, 0), "", MESSAGE_BIG, makeacol32(128, 128, 255, 255));
	TextList.push_back(ConnectionStatusText);
	TextList.push_back(GameInfoText);
	TextList.push_back(GameStatusText);
	TextList.push_back(GameMessageText);

	zigclient_c::start();

	MatchOver = false;
	WantsReconnect = false;
	DisplayingScoreTable = false;
}


NetworkClient::~NetworkClient() {
	while (!TextList.empty()) {
		delete TextList.back();
		TextList.pop_back();
	}

	zigclient_c::stop();
}


bool NetworkClient::Start(const std::string &GuiPath) {
	const string FullGuiPath = string(DATA_PREFIX) + GuiPath;

	if (!exists(FullGuiPath.c_str())) {
		_ERROR_;
		return false;
	}

	XMLParser Parser(FullGuiPath);
	const TiXmlElement *Root = Parser.GetRoot();

	if (!InitRenderEngine()) {
		_ERROR_;
		return false;
	}

	if (!LoadGUI(Root)) {
		_ERROR_;
		const std::string Msg = "Somehow you managed GUI loader to fail. Bravo!";
		_SAY(Msg);
		if (is_windowed_mode())
			allegro_message((std::string("Fatal error!\n") + Msg).c_str());
		return false;
	}

	return true;
}


void NetworkClient::Stop() {
	_SAY("Stop called!");

	for (MAP_CLIOBJ_ITER x = CliObjList.begin(); x != CliObjList.end(); ++x)
		delete x->second;

	for (std::map<OUIDInfo, SoundSample*>::iterator it = SoundSampleList.begin();
		it != SoundSampleList.end(); ++it)
			delete it->second;

	for (USHORT x = 0; x < StarList.size(); ++x)
		delete StarList[x];

	CliObjList.clear();
	RendObjList.clear();
	GuiRendObjList.clear();
	OponentList.clear();
	TeamScoreList.clear();
	StarList.clear();
	SoundSampleList.clear();
}


void NetworkClient::Disconnect(int reason) {
	_SAY("Disconnect called");
	zigclient_c::disconnect(reason);
	//zigclient_c::stop();
	Rendered = false;
}


bool NetworkClient::Connect(const string &GuiPath, const ClientOptions &CO) {
	if (!Start(GuiPath))
		return false;

	ClientGuiPath = GuiPath;

	ClientInfo CI;
	CI.ShipPath = CO.XMLFileShipPath;
	CI.Team = CO.Team;
	CI.Name = CO.Name;

	//will be needed when reconnecting
	CliOpt = CO;

	//Connection data structure the server is expecting:
	//ushort: network protocol version
	//string: password
	//client info
	buffer_c ConnectData;
	ConnectData.putShort(NSB_NET_PROTOCOL_VER);
	ConnectData << CO.Password;
	ConnectData << CI;

	zigclient_c::set_connect_data(ConnectData);
	zigclient_c::set_compression(MIN_COMPRESSION_SIZE);
	zigclient_c::set_server_timeout(8);
	zigclient_c::set_connect_timeout(8);
	//zigclient_c::set_shutdown_timeout(8);

	address_c Address(CO.ServerAddress);

	if (!Address.valid() || !Address.get_port()) {
		_SAY("Malformed IP address passed to the client.");
		return false;
	}

	ServerAddress = CO.ServerAddress;

	SetConnectionStatusText("CONNECTING TO " + CO.ServerAddress);

	return zigclient_c::connect(Address);
}


bool NetworkClient::Reconnect() {
	if (!Start(ClientGuiPath))
		return false;

	/*Some data may have chenaged (team for example) and we have to set_connect_data() again.*/
	ClientInfo CI;
	CI.ShipPath = CliOpt.XMLFileShipPath;
	CI.Team = CliOpt.Team;
	CI.Name = CliOpt.Name;

	//Connection data structure the server is expecting:
	//ushort: network protocol version
	//string: password
	//client info
	buffer_c ConnectData;
	ConnectData.putShort(NSB_NET_PROTOCOL_VER);
	ConnectData << CliOpt.Password;
	ConnectData << CI;

	zigclient_c::set_connect_data(ConnectData);

	address_c Address(ServerAddress);

	SetConnectionStatusText("RECONNECTING");

	return zigclient_c::connect(Address);
}


bool NetworkClient::InitRenderEngine() {
	CameraOwner = 0;
	MainObject = 0;
	CameraGeo = Rect<float>(0.f, 0.f, UNITS_PER_SCREEN_W, UNITS_PER_SCREEN_H);
	TargetCameraDim = Size(UNITS_PER_SCREEN_W, UNITS_PER_SCREEN_H);

	Rendered = false;

	return true;
}


bool NetworkClient::LoadGUI(const TiXmlElement *Root) {
	Statistics = static_cast<FPSDisplay*>(LoadObjectFromMap(Root, FPS_DISPLAY_STR));
	if (!Statistics)
		return false;

	Stb = static_cast<Statusbar*>(LoadObjectFromMap(Root, STATUSBAR_STR));
	if (!Stb)
		return false;

	ScrMask = static_cast<ScreenMask*>(LoadObjectFromMap(Root, SCREEN_MASK_STR));
	if (!ScrMask)
		return false;

	MsgLister = static_cast<MessageLister*>(LoadObjectFromMap(Root, MESSAGE_LISTER_STR));
	if (!MsgLister)
		return false;

	MsgWriter = static_cast<MessageWriter*>(LoadObjectFromMap(Root, MESSAGE_WRITER_STR));
	if (!MsgWriter)
		return false;

	return true;
}


void NetworkClient::connected(buffer_c &data) {
	_CHECKPOINT_;

	try {
		//The first data should be the hello message from the server.
		std::string HelloMsg = data.getString();
		//Display the string messge.
		MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_SERVER_MESSAGE);
		MsgLister->Feed("\\server\\: " + HelloMsg);

		//The second data is GameOptions
		const auto_ptr<GameOptions> GO(static_cast<GameOptions*>(data.getObject()));
		GameOpt = *(GO.get());

		//The third data are two unsigned shorts representing arena size
		ArenaSize.Width = data.getShort();
		ArenaSize.Height = data.getShort();

		if (ArenaSize.Width < 500 ||  ArenaSize.Width > 10000 ||
			ArenaSize.Height < 500 ||  ArenaSize.Height > 10000)
				throw EX_EOB;

		//The forth data should be an unsigned short representing server's net ratio,
		//set by the user when starting the server
		NetworkClient::NetRatio = data.getShort();
		_DISPLAY(NetworkClient::NetRatio);

		//The fifth data is player name. Server may have changed the name in order to prevent name collisions.
		Name = data.getString();
		Stb->SetPlayerName(Name);

		//The LAST data should be the list of coded paths.
		const auto_ptr<CodedPaths> CP(static_cast<CodedPaths*>(data.getObject()));
		PathCoder.clear();
		PathCoder.insert(CP.get()->begin(), CP.get()->end());
	} catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Client received illegal packet, dropping connection.");

		Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
		return;
	}

	Uptime = 0;
	MatchOver = false;
	DisplayingScoreTable = false;
	LastPacketID = 0;

	_SAY("Connected!");
	SetConnectionStatusText("");
	SetGameStatusText("");

	BuildStars();

	DataMsgStream = zigclient_c::create_stream(STREAM_CUMULATIVE_ACK);
	InfoMsgStream = zigclient_c::create_stream(STREAM_INDIVIDUAL_ACK);
}


void NetworkClient::connection_timed_out() {
	_SAY("connection timed out");
	SetConnectionStatusText("CONNECTION TIMED OUT");
	Stop();
}


void NetworkClient::connection_refused(buffer_c &reason, int code_reason) {
	std::string msg;

	if (code_reason == ZIGCLIENT_CONNECTFAILED) {
		reason >> msg;
		_DISPLAY(msg);
	}

	_DISPLAY(code_reason);
	_SAY("Connection refused");
	SetConnectionStatusText("CONNECTION REFUSED: " + msg);
	Stop();
}


void NetworkClient::disconnected(bool server_initiated, buffer_c &reason, int code_reason, int zig_reason) {
	_DISPLAY(code_reason);
	_DISPLAY(zig_reason);
	_SAY("Client disconectned");

	if (zig_reason == LEETCLIENT_DISCONNECTED)
		zig_reason = code_reason;

	switch (zig_reason) {
		case ZIGCLIENT_DISCONNECTED_TIMEOUT:
			SetConnectionStatusText("DISCONNECTED FROM SERVER DUE TO SERVER TIMEOUT");
		break;
		case ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION:
			SetConnectionStatusText("DISCONNECTED FROM SERVER DUE TO PROTOCOL VIOLATION");
		break;
		case ZIGCLIENT_DISCONNECTED_SOCKET_ERROR:
			 SetConnectionStatusText("DISCONNECTED FROM SERVER DUE TO INTERNAL SOCKET ERROR");
		break;
		default:
			SetConnectionStatusText("DISCONNECTED FROM SERVER");
	}

	if (server_initiated) {
		if (reason.size() == 1) {
			UCHAR b = reason.getByte();
			if (b == NSB_CLI_DISCONNECTED_GAME_ENDED) {
				WantsReconnect = true;
				SetConnectionStatusText("RECONNECTING SOON...");
			}
			else if (b == NSB_CLI_DISCONNECTED_KICK_VOTED) {
				SetConnectionStatusText("PLAYERS VOTED TO KICK YOU FROM THE GAME");
			}
		}
	}

	Stop();
}


void NetworkClient::incoming_server_info(address_c &addr, buffer_c &info, int time_ms, bool timedout) {
	_SAY("wtf?");
}


void NetworkClient::incoming_data(buffer_c &in, int PacketID) {
	//dropping late and duplicate packets
	if (PacketID <= LastPacketID) {
		_SAY("Packet dropped.");
		return;
	}

	LastPacketID = PacketID;

	ProcessInReliableMsg();
	ProcessInUnreliableMsg(in);
}


void NetworkClient::ProcessInUnreliableMsg(buffer_c &in) {
	if (!in.size())
		return;

	buffer_c Data;

	try {
		ReadSounds(in);
		if (in.size_left()) {
			MessageCoder::Decode(in, Data);
			ParseGamePacket(Data);
		}
	} catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Client received illegal packet, dropping connection.");
		Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
		return;
	} catch (ExSparkletException &ex) {
		_ERROR_;
		_DISPLAY(ex.GetMsg());
		Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
		return;
	} catch (...) {
		_ERROR_;
		_SAY("Unknown exception cought, dropping connection.");
		Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
		return;
	}
}


void NetworkClient::ProcessInReliableMsg() {
	buffer_c Msg;
	int StreamID;
	bool ScoreChanged = false;

	seek_first_stream();

	while ((StreamID = get_next_stream()) != -1) {
		while (zigclient_c::receive_message(Msg, StreamID) > 0) {
			try {
				if (StreamID == DataMsgStream) {
					const UCHAR HasCameraOwner = Msg.getByte();
					if (HasCameraOwner) {
						const OUIDInfo *co(static_cast<OUIDInfo*>(Msg.getObject()));
						const OUIDInfo *mo(static_cast<OUIDInfo*>(Msg.getObject()));
						CameraOwner = *co;
						MainObject = *mo;
						delete co;
						delete mo;
					}

					buffer_c Data;
					MessageCoder::Decode(Msg, Data);
					ParseGamePacket(Data);

					//_SPARKLET_ASSERT(CliObjList.find(CameraOwner) != CliObjList.end());
				}
				else if (StreamID == InfoMsgStream) {
					ProcessGameStateMsg(Msg, ScoreChanged);
				}
				else {
					_ERROR_;
					_DISPLAY(StreamID);
				}
			} catch (zig_int_exceptions_t ex) {
				_ERROR_;
				_SAY("Client received illegal packet, dropping connection.");
				Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
				return;
			} catch (ExSparkletException &ex) {
				_ERROR_;
				_DISPLAY(ex.GetMsg());
				Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
				return;
			} catch (...) {
				_ERROR_;
				_SAY("Unknown exception cought, dropping connection.");
				Disconnect(ZIGCLIENT_DISCONNECTED_PROTOCOL_VIOLATION);
				return;
			}
		}
	}

	if (ScoreChanged) {
		RankPlayers();

		if (GameOpt.GT == GT_DEATHMATCH) {
			const Oponent *We = &get_element(OponentList, MyCliUID);
			Stb->SetScore(We->Score.Points, GameOpt.PointsMax, We->Score.Advantage, We->Score.Rank, OponentList.size());
		}
		else if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
			const ScoreT *We = &get_element(TeamScoreList, MyTeam);
			Stb->SetScore(We->Points, GameOpt.PointsMax, We->Advantage, We->Rank, TeamScoreList.size());
		}
	}
}


/*
game packet structure:
	OUIDInfo "ObjectUID"	- Object Unique ID - the UID of the object to which the following mesages apply onto
	UCHAR "Pending" 		- Number of messages pending for this object
		-> if (Pending == 0) <-- special case - remove the object, no more messages for this object
	serializable_c			- "Pending" messages pending
		-> if the object with "ObjectUID" UID doesn't exist the first message is FullObjInfo
	... and again ...
	OUIDInfo
	UCHAR
	serializable_c
	...
*/
void NetworkClient::ParseGamePacket(buffer_c &Data) {
	while (Data.size_left()) {
		//the data is object UID
		const OUIDInfo* ObjectUID(static_cast<const OUIDInfo*>(Data.getObject()));

		//a number of messages pending for this object.
		UCHAR Pending = Data.getByte();

		if (Pending == 0) { //0 is the special case.
			//if 0 then just remove the object and continue
			if (CliObjList.find(*ObjectUID) != CliObjList.end()) {
				DestroyObject(*ObjectUID);
				delete ObjectUID;
				continue;
			}
			_ERROR_;
			delete ObjectUID;
			continue;
		}

		if (CliObjList.find(*ObjectUID) == CliObjList.end()) {
			const FullObjInfo *FOI(static_cast<FullObjInfo*>(Data.getObject()));
			CliObject *CO = CreateObject(FOI, *ObjectUID);
			CliObjList.insert(CliObjList.end(), make_pair(*ObjectUID, CO));
			--Pending;
			delete FOI;
		}

		CliObject *CO = get_element(CliObjList, *ObjectUID);
		DeliverMsg(CO, Pending, Data);
		delete ObjectUID;
	}
}


void NetworkClient::ProcessGameStateMsg(buffer_c &Msg, bool &ScoreChanged) {
	const serializable_c *Info(Msg.getObject());
	const int code = Info->ZIG_GetTypeId();

	if (code == NewPlayerInfo::oCreator.GetTypeId()) {
		OponentConnected(static_cast<const NewPlayerInfo*>(Info));
		ScoreChanged = true;
	}
	else if (code == PlayerGoneInfo::oCreator.GetTypeId()) {
		OponentDisconnected(static_cast<const PlayerGoneInfo*>(Info));
		RemoveBogusTeams();
		ScoreChanged = true;
	}
	else if (code == ScoreInfo::oCreator.GetTypeId()) {
		ReadScore(static_cast<const ScoreInfo*>(Info), Msg);
		ScoreChanged = true;
	}
	else if (code == GameMessage::oCreator.GetTypeId()) {
		DisplayGameMessage(static_cast<const GameMessage*>(Info));
	}
	else if (code == WinnerInfo::oCreator.GetTypeId()) {
		SomeoneWon(static_cast<const WinnerInfo*>(Info));
	}
	else if (code == PlayerMessage::oCreator.GetTypeId()) {
		DisplayPlayerMessage(static_cast<const PlayerMessage*>(Info));
	}
	else if (code == PingInfo::oCreator.GetTypeId()) {
		ReadPingInfo(static_cast<const PingInfo*>(Info));
	}
	else if (code == ServerMessageInfo::oCreator.GetTypeId()) {
		const ServerMessageInfo *SMI = static_cast<const ServerMessageInfo*>(Info);
		DisplayServerMessage(SMI->Text);
	}
	else if (code == ChangeTeamCommandInfo::oCreator.GetTypeId()) {
		const ChangeTeamCommandInfo *CTCI = static_cast<const ChangeTeamCommandInfo*>(Info);
		ChangeTeam(CTCI);
	}
	else {
		_ERROR_;
	}

	delete Info;
}


void NetworkClient::DisplayServerMessage(const std::string Text) {
	MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_SERVER_MESSAGE);
	MsgLister->Feed(Text);
}


void NetworkClient::ReadScore(const ScoreInfo *SI, buffer_c &Msg) {
	_SPARKLET_ASSERT(OponentList.size() == SI->size());

	map<int, Oponent>::iterator iter = OponentList.begin();
	for (USHORT x = 0; x < SI->size(); ++x) {
		iter->second.Score.Points = (*SI)[x];
		++iter;
	}

	if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
		const ScoreInfo *SITeam = static_cast<const ScoreInfo*>(Msg.getObject());
		_SPARKLET_ASSERT(TeamScoreList.size() == SITeam->size());

		map<TEAM, ScoreT>::iterator iter2 = TeamScoreList.begin();
		for (USHORT x = 0; x < SITeam->size(); ++x) {
			iter2->second.Points = (*SITeam)[x];
			++iter2;
		}

		delete SITeam;
	}
}


void NetworkClient::ChangeTeam(const ChangeTeamCommandInfo *CTCI) {
	TEAM OldTeam;
	Oponent *Opo = &OponentList.find(CTCI->Who)->second;

	OldTeam = Opo->Team;
	Opo->Team = CTCI->Team;
	if (CTCI->Who == MyCliUID) {
		CliOpt.Team = CTCI->Team;
		MyTeam = CTCI->Team;
	}

	if (count_if(OponentList.begin(), OponentList.end(), SameTeam(OldTeam)) == 0)
		TeamScoreList.erase(OldTeam);

	if (TeamScoreList.find(CTCI->Team) == TeamScoreList.end())
		TeamScoreList.insert(make_pair(CTCI->Team, ScoreT()));
}


void NetworkClient::ReadPingInfo(const PingInfo *PI) {
	_SPARKLET_ASSERT(OponentList.size() == PI->size());

	map<int, Oponent>::iterator iter = OponentList.begin();
	for (USHORT x = 0; x < PI->size(); ++x) {
		iter->second.Ping = (*PI)[x];
		++iter;
	}
}


void NetworkClient::ProcessOutReliableMsg() {
}


void NetworkClient::ReadSounds(buffer_c &in) {
	const UCHAR NewSoundsUntunableCount = in.getByte();
	for (UCHAR x = 0; x < NewSoundsUntunableCount; ++x) {
		const NewSoundInfoUntunable* NSIU(static_cast<NewSoundInfoUntunable*>(in.getObject()));
		PlaySound(NSIU);
		delete NSIU;
	}

	const UCHAR NewSoundsCount = in.getByte();
	for (UCHAR x = 0; x < NewSoundsCount; ++x) {
		const NewSoundInfo* NSI(static_cast<NewSoundInfo*>(in.getObject()));
		PlaySound(NSI);
		delete NSI;
	}

	std::map<OUIDInfo, SoundSample*> SoundSampleKeep;
	const UCHAR AdjustSoundsCount = in.getByte();
	for (UCHAR x = 0; x < AdjustSoundsCount; ++x) {
		const AdjustSoundInfo* ASI(static_cast<AdjustSoundInfo*>(in.getObject()));
		std::map<OUIDInfo, SoundSample*>::iterator it = AdjustSound(ASI);
		delete ASI;
		if (it != SoundSampleList.end()) {
			SoundSampleKeep.insert(std::make_pair(it->first, it->second));
			//remove it from the list to prevent deleting the object later
			SoundSampleList.erase(it);
		}
	}
	for (std::map<OUIDInfo, SoundSample*>::iterator it = SoundSampleList.begin(); it != SoundSampleList.end(); ++it)
		delete it->second;

	SoundSampleList = SoundSampleKeep;
}


void NetworkClient::PlaySound(const NewSoundInfoUntunable *NSIU) {
	SoundSample *SS = new SoundSample(NSIU, PathCoder[NSIU->PathCode]);
	SS->Play();
}


void NetworkClient::PlaySound(const NewSoundInfo *NSI) {
	SoundSample *SS = new SoundSample(NSI, PathCoder[NSI->PathCode]);
	SoundSampleList.insert(std::make_pair(NSI->SoundUID, SS));
	SS->Play();
}


std::map<OUIDInfo, SoundSample*>::iterator NetworkClient::AdjustSound(const AdjustSoundInfo *ASI) {
	std::map<OUIDInfo, SoundSample*>::iterator it = SoundSampleList.find(ASI->SoundUID);

	if (it != SoundSampleList.end())
		it->second->Adjust(ASI);

	return it;
}


void NetworkClient::Update() {
	++DisplayLogoSkip;
	++DisplayingScoreTableCounter;
	++MessageWriterActiveCounter;
	++StatisticsActiveCounter;

	for (USHORT x = 0; x < TextList.size(); ++x)
		TextList[x]->Update();

	if (!zigclient_c::is_connected()) {
		if (WantsReconnect) {
			TimeToWaitBrforeReconect -= gTimeElapsed;
			if (TimeToWaitBrforeReconect < 0) {
				Reconnect();
				WantsReconnect = false;
				TimeToWaitBrforeReconect = 4;
			}
		}
		return;
	}

	Uptime += gTimeElapsed;

	ProcessOutReliableMsg();

	UpdateCamera();

	for (MAP_CLIOBJ_ITER Obj = CliObjList.begin(); Obj != CliObjList.end(); ++Obj)
		Obj->second->OnUpdate();

	Rendered = false;
}


void NetworkClient::UpdateCamera() {
	if (CameraOwner != OUIDInfo(0)) {
		const std::map<OUIDInfo, CliObject*>::const_iterator it = CliObjList.find(CameraOwner);
		//In some cases it is possible that the object holding the camera goes out of the camera. That is
		//because camera won't follow objects outside the map.
		if (it == CliObjList.end())
			return;
		const CliObject *CO = it->second;
		const Size NewCameraSize = CalcCameraDim(CameraGeo, TargetCameraDim);
		const Point<float> TargetCameraLoc = GetCameraTargetSmart(GetCenter(CO->GetGeo()), NewCameraSize, ArenaSize);
		const Point<float> NewCameraLoc = CalcCameraLoc(CameraGeo, TargetCameraLoc);
		CameraGeo = Rect<float>(NewCameraLoc, NewCameraSize);

	}
}


void NetworkClient::SetConnectionStatusText(const std::string &Text) {
	//simplyfies things, instead of involving text length routines
	const int FONT_WIDTH = 17;
	const Point<float> Loc(UNITS_PER_SCREEN_W / 2 - FONT_WIDTH * Text.size() / 2, UNITS_PER_SCREEN_H / 20);
	ConnectionStatusText->SetLoc(Loc);
	ConnectionStatusText->SetText(Text);
}


void NetworkClient::SetGameInfoText(const std::string &Text) {
	//simplyfies things, instead of involving text length routines
	const int FONT_WIDTH = 17;
	const Point<float> Loc(UNITS_PER_SCREEN_W / 2 - FONT_WIDTH * Text.size() / 2, UNITS_PER_SCREEN_H / 20 + 60);
	GameInfoText->SetLoc(Loc);
	GameInfoText->SetText(Text);
	GameInfoText->FadeOut();
	GameInfoText->SetColor(makeacol32(128, 128, 255, 255));
}


void NetworkClient::SetGameMessageText(const std::string &Text) {
	//simplyfies things, instead of involving text length routines
	const int FONT_WIDTH = 17;
	const Point<float> Loc(UNITS_PER_SCREEN_W / 2 - FONT_WIDTH * Text.size() / 2, UNITS_PER_SCREEN_H / 20 + 90);
	GameMessageText->SetLoc(Loc);
	GameMessageText->SetText(Text);
	GameMessageText->FadeOut();
	GameMessageText->SetColor(makeacol32(128, 128, 255, 255));
}


void NetworkClient::SetGameStatusText(const std::string &Text) {
	//simplyfies things, instead of involving text length routines
	const int FONT_WIDTH = 17;
	const Point<float> Loc(UNITS_PER_SCREEN_W / 2 - FONT_WIDTH * Text.size() / 2, UNITS_PER_SCREEN_H / 20 + 30);
	GameStatusText->SetLoc(Loc);
	GameStatusText->SetText(Text);
}


bool NetworkClient::Render() {
	if (!Displaying)
		return false;

	if (!zigclient_c::is_connected() && !(DisplayLogoSkip % 4)) {
		//render the logo
		Logo.Render(Rect<float>(0, 0, Logo.Dim.Width, Logo.Dim.Height),
			Rect<float>(0, 0, UNITS_PER_SCREEN_W, UNITS_PER_SCREEN_H), 0);

		for (USHORT x = 0; x < TextList.size(); ++x)
			TextList[x]->Display();

		glFlush();
		allegro_gl_flip();
		glClear(GL_COLOR_BUFFER_BIT);
		glLoadIdentity();
	}
	else if (!Rendered) {
		glDisable(GL_TEXTURE_2D);
		glEnable(GL_POINT_SMOOTH);
		for_each(StarList.begin(), StarList.end(), DrawStar(CameraGeo));
		glDisable(GL_POINT_SMOOTH);
		glEnable(GL_TEXTURE_2D);

		//handle zoom
		const float ZoomRatioW = (float)UNITS_PER_SCREEN_W / CameraGeo.Width;
		const float ZoomRatioH = (float)UNITS_PER_SCREEN_H / CameraGeo.Height;
		glScalef(ZoomRatioW, ZoomRatioH, 1);

		//render "physical" objects
		for (MAP_RENOBJ_ITER x = RendObjList.begin(); x != RendObjList.end(); ++x)
			x->second->Render();

		//make gui graphics immune to zoom
		glLoadIdentity();

		//gui elements follow
		for (MAP_RENOBJ_ITER x = GuiRendObjList.begin(); x != GuiRendObjList.end(); ++x)
			x->second->Render();

		for (USHORT x = 0; x < TextList.size(); ++x)
			TextList[x]->Display();

		if (DisplayingScoreTable)
			DisplayScoreTable();

		Rendered = true;

		//flush & flip the whole thing, it's too easy with OGL, what's the fun? :(
		glFlush();
		allegro_gl_flip();
		glClear(GL_COLOR_BUFFER_BIT);
		glLoadIdentity();

		return true;
	}

	return false;
}


void NetworkClient::BuildStars() {
	const ULONG StarsCount = ((ULONG)ArenaSize.Width * ArenaSize.Height) / 7000;

	StarList.reserve(StarsCount);
	for (USHORT i = 0; i < StarsCount; ++i) {
		const int DepthMin = 2;
		const int DepthMax = 15;
		const int DepthDiff = DepthMax - DepthMin;

		const int x = (rand() % (ArenaSize.Width + UNITS_PER_SCREEN_W));
		const int y = (rand() % (ArenaSize.Height + UNITS_PER_SCREEN_H));
		const int z = DepthMin + (rand() % DepthDiff);
		const UCHAR Depth = 255 - (UCHAR)(((float)(z - DepthMin) / (float)DepthDiff) * 255.f);
		StarList.push_back(new Star(Point<float>(x, y), z, Depth));
	}
}


CliObject* NetworkClient::CreateObject(const FullObjInfo *FOI, const OUIDInfo &ObjID) {
	USHORT PathCode = FOI->XMLFilePathCode;
	string Path;

	if (PathCode < RESERVED_PATH_CODE_COUNT) {
		switch (PathCode) {
			case RANDOM_YELLOW_PARTICLE:
				Path = GetRandomBrightParticle();
			break;
			case RANDOM_BLUE_PARTICLE:
				Path = GetRandomBlueParticle();
			break;
			case RANDOM_RED_PARTICLE:
				Path = GetRandomRedParticle();
			break;
			default:
				_ERROR_;
		}
	}
	else {
		std::map<USHORT, std::string>::iterator it = PathCoder.find(PathCode);
		if (it == PathCoder.end()) {
			_ERROR_;
			_DISPLAY((int)PathCode);
			_DISPLAY(PathCoder.size());
			abort();
			return NULL;
		}

		Path = it->second;
	}

	XMLParser Parser(string(DATA_PREFIX) + Path);
	const TiXmlElement *Root = Parser.GetRoot();

	CliObject *RetVal;

	if (ObjID == MainObject) {
		_SPARKLET_ASSERT(XMLParser::GetName(Root) == PLAYER_OBJECT_STR);
		RetVal = new CliPlayerObject(Root, FOI->Dim, FOI->Scale, Stb);
		_SPARKLET_ASSERT(RetVal);
	}
	else {
		RetVal = gInitializer->CreateCliObject(Root, FOI->Dim, FOI->Scale);
		_SPARKLET_ASSERT(RetVal);
	}

	RetVal->SetCamera(&CameraGeo);
	RendObjList.insert(make_pair(FOI->Surface, RetVal));

	return RetVal;
}


void NetworkClient::DestroyObject(OUIDInfo ObjectUID) {
	MAP_CLIOBJ_ITER iter = CliObjList.find(ObjectUID);
	_SPARKLET_ASSERT(iter != CliObjList.end());

	for (MAP_RENOBJ_ITER x = RendObjList.begin(); x != RendObjList.end(); ++x) {
		if (x->second == iter->second) {
			RendObjList.erase(x);
			break;
		}
	}

	delete iter->second;
	CliObjList.erase(iter);
}


CliObject* NetworkClient::LoadObjectFromMap(const TiXmlElement *Root, const string ItemName) {
	const TiXmlElementList* Child = XMLParser::GetChildren(Root, ItemName);

	if (Child->size() == 0)
		return NULL;

	const TiXmlElement *Element = Child->at(0);
	delete Child;

	Rect<float> Geo(
		XMLParser::GetValueF(Element, X_STR, 0.f),
		XMLParser::GetValueF(Element, Y_STR, 0.f),
		XMLParser::GetValueL(Element, W_STR, 0),
		XMLParser::GetValueL(Element, H_STR, 0)
	);

	if (!Geo.Width || !Geo.Height) {
		_SAY("Invalid item's width and/or height");
		return NULL;
	}

	const USHORT Surface = XMLParser::GetValueL(Element, SURFACE_STR, 10);
	const string RelPath = XMLParser::GetValueS(Element, SOURCE_STR, "bad");
	const string FullPath = FixFilenameSlashes(string(DATA_PREFIX) + RelPath);

	if (!exists(FullPath.c_str()) || RelPath == "bad") {
		_SAY("Invalid item's 'source' tag");
		return NULL;
	}

	GuiObject *New = NULL;

	try {
		XMLParser Parser(FullPath);
		const TiXmlElement *ObjRoot = Parser.GetRoot();
		New = static_cast<GuiObject*>(gInitializer->CreateCliObject(ObjRoot, Geo, 1.f));
		if (!New)
			return NULL;
	}
	catch (const ExFileNotFound& fnf) {
		return NULL;
	}

	New->SetLoc(Geo);
	New->SetCamera(&CameraGeo);

	static OUIDInfo CliObjUID = 0;
	++CliObjUID;
	_SPARKLET_ASSERT(CliObjUID < OUIDInfo(RESERVED_OBJECT_UID_COUNT));
	CliObjList.insert(make_pair(CliObjUID, New));
	GuiRendObjList.insert(make_pair(Surface, New));

	return New;
}


void NetworkClient::DeliverMsg(CliObject *CO, UCHAR PendingCount, buffer_c &in) {
	for (UCHAR MsgCounter = 0; MsgCounter < PendingCount; ++MsgCounter) {
		const serializable_c *Msg(in.getObject());
		const int code = Msg->ZIG_GetTypeId();

		if (code == VelocityInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const VelocityInfo*>(Msg));
		else if (code == RotationInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const RotationInfo*>(Msg));
		else if (code == FullRotationInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const FullRotationInfo*>(Msg));
		else if (code == FullVelocityInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const FullVelocityInfo*>(Msg));
		else if (code == EngineInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const EngineInfo*>(Msg));
		else if (code == WeaponInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const WeaponInfo*>(Msg));
		else if (code == ShipState::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const ShipState*>(Msg));
		else if (code == ShipInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const ShipInfo*>(Msg));
		else if (code == ShipInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const ShipInfo*>(Msg));
		else if (code == AspectInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const AspectInfo*>(Msg));
		else if (code == ParticleInfo::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const ParticleInfo*>(Msg));
		else if (code == StateInfoGeneric1::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const StateInfoGeneric1*>(Msg));
		else if (code == StateInfoGeneric2::oCreator.GetTypeId())
			CO->SetMsg(static_cast<const StateInfoGeneric2*>(Msg));
		else
			_ERROR_;

		delete Msg;
	}
}


std::string GetRawText(const std::string &Line, const USHORT WordsToSkip) {
	unsigned int From = 0;

	for (int x = 0; x < WordsToSkip; x++) {
		unsigned int lFrom = Line.find(' ', From);
		if (lFrom == std::string::npos)
			return "";
		From += lFrom;
	}

	return Line.substr(From + 1);
}


bool NetworkClient::CheckForPlayer(const std::string &lName) {
	for (std::map<int, Oponent>::iterator it = OponentList.begin(); it != OponentList.end(); ++it) {
		if (it->second.Name == lName)
			return true;
	}

	return false;
}


void NetworkClient::ExecuteCmd(std::vector<std::string> &WordList, const std::string &Params) {
	static const char* CmdList[] = {
#define GAME_CMD(cmd) #cmd,
#include "game_commads.h"
#undef GAME_CMD
	};
	//shouldn't this be updated automagically?
	const USHORT CmdCount = 12;

	const std::string &Command = WordList[0];

	bool Found = false;
	for (USHORT x = 0; x < CmdCount; x++) {
		if (CaseInsensitiveMatch(Command, CmdList[x])) {
			Found = true;
			break;
		}
	}

	if (!Found) {
		MsgLister->Feed("Unknown command '/" + Command + "'.");
		MsgLister->Feed("Type '/help' for list of available commands.");
		return;
	}

	if (CaseInsensitiveMatch(Command, "help")) {
		if (WordList.size() == 1) {
			std::string Msg;
			for (USHORT x = 0; x < CmdCount; x++)
				Msg += std::string("/") + CmdList[x]  + " ";
			MsgLister->Feed("Available commands:");
			MsgLister->Feed(Msg);
			MsgLister->Feed("Use '/help <command>' to display command usage.");
		}
		else {
#define SHOW_HELP1(cmd, help1)         \
			if (WordList[1] == #cmd) { \
				MsgLister->Feed(help1);\
				return;                \
			}
#define SHOW_HELP2(cmd, help1, help2)  \
			if (WordList[1] == #cmd) { \
				MsgLister->Feed(help1);\
				MsgLister->Feed(help2);\
				return;                \
			}
#define SHOW_HELP3(cmd, help1, help2, help3) \
			if (WordList[1] == #cmd) {       \
				MsgLister->Feed(help1);      \
				MsgLister->Feed(help2);      \
				MsgLister->Feed(help3);      \
				return;                      \
			}

			SHOW_HELP1(help, "Display a list of available commands.");
			SHOW_HELP1(version, "Display version information.");
			SHOW_HELP1(playerlist, "Display a list of players playing the game.");
			SHOW_HELP1(yes, "Vote 'yes' to current poll.");
			SHOW_HELP1(no, "Vote 'no' to current poll.");
			SHOW_HELP2(msg, "Send a message to someone.", "usage: /msg <nick> <message>");
			SHOW_HELP2(msgteam, "Send a message to all members of your team.", "usage: /msgteam <message>");
			SHOW_HELP2(kick, "Start a new poll to kick someone from the game.", "usage: /kick <nickname> <reason>");
			SHOW_HELP2(nohide, "Do not hide message writer/lister automatically.", "Issue it again to make it hidable.");
			SHOW_HELP1(maplist, "Display list of available maps on the server.");
			SHOW_HELP3(chmap, "Start a poll to change the map.", "usage: /chmap <map_name>", "Use /maplist to get list of maps.");
			SHOW_HELP3(chteam, "Join the other team.", "usage: /chteam <team>", "<team> is blue, red, green or yellow");

#undef SHOW_HELP1
#undef SHOW_HELP2
#undef SHOW_HELP3
		}

		return;
	}

	if (CaseInsensitiveMatch(Command, "version")) {
		MsgLister->Feed(SPARKLET_VERSION_STR);
		return;
	}

	if (CaseInsensitiveMatch(Command, "msg")) {
		if (WordList.size() == 1) {
			std::vector<std::string> W;
			W.push_back("help");
			W.push_back("msg");
			ExecuteCmd(W, "msg");
		}
		else {
			if (!CheckForPlayer(WordList[1])) {
				MsgLister->Feed("No player named '" + WordList[1] + "'.");
				return;
			}

			SendPlayerMessage(GetRawText(Params, 1), WordList[1]);
		}
		return;
	}

	if (CaseInsensitiveMatch(Command, "msgteam")) {
		SendTeamMessage(Params);
		return;
	}

	if (CaseInsensitiveMatch(Command, "playerlist")) {
		std::string List;
		for (std::map<int, Oponent>::iterator it = OponentList.begin(); it != OponentList.end(); ++it)
			List += it->second.Name + " ";
		MsgLister->Feed(List);

		return;
	}

	if (CaseInsensitiveMatch(Command, "no")) {
		StateInfoGeneric1 SIG;
		SIG.State = 0;
		buffer_c buf;
		buf << SIG;
		zigclient_c::send_message(buf, InfoMsgStream, true);
		return;
	}

	if (CaseInsensitiveMatch(Command, "yes")) {
		StateInfoGeneric1 SIG;
		SIG.State = 1;
		buffer_c buf;
		buf << SIG;
		zigclient_c::send_message(buf, InfoMsgStream, true);
		return;
	}

	if (CaseInsensitiveMatch(Command, "kick")) {
		if (WordList.size() < 2) {
			std::vector<std::string> W;
			W.push_back("help");
			W.push_back("kick");
			ExecuteCmd(W, "kick");
		}
		else {
			if (!CheckForPlayer(WordList[1])) {
				MsgLister->Feed("No player named '" + WordList[1] + "'.");
				return;
			}

			NewKickPollInfo NKPI;
			NKPI.KickWho = WordList[1];
			NKPI.Reason = GetRawText(Params, 1);
			buffer_c buf;
			buf << NKPI;
			zigclient_c::send_message(buf, InfoMsgStream, true);
		}

		return;
	}

	if (CaseInsensitiveMatch(Command, "nohide")) {
		AutoHideMsgWriter = !AutoHideMsgWriter;
		MsgLister->SetAutohide(AutoHideMsgWriter);
		return;
	}

	if (CaseInsensitiveMatch(Command, "maplist")) {
		ServerRequestInfo SRI;
		SRI.RequestCode = SR_MAP_LIST;
		buffer_c buf;
		buf << SRI;
		zigclient_c::send_message(buf, InfoMsgStream, true);
		return;
	}

	if (CaseInsensitiveMatch(Command, "chmap")) {
		ChangeMapCommandInfo CMCI;
		CMCI.MapName = Params;
		buffer_c buf;
		buf << CMCI;
		zigclient_c::send_message(buf, InfoMsgStream, true);
		return;
	}

	if (CaseInsensitiveMatch(Command, "chteam")) {
		if (WordList.size() < 2) {
			std::vector<std::string> W;
			W.push_back("help");
			W.push_back("chteam");
			ExecuteCmd(W, "chteam");
			return;
		}

		ChangeTeamCommandInfo CTCI;
		CTCI.Team = GetTeamFromName(WordList[1]);

		if (CTCI.Team == TEAM_NO_TEAM)
			MsgLister->Feed("No such team. Available teams are: blue, red, green and yellow.");
		else {
			buffer_c buf;
			buf << CTCI;
			zigclient_c::send_message(buf, InfoMsgStream, true);
		}

		return;
	}
}


void NetworkClient::PollMessageWriter() {
	while (keypressed()) {
		if (!MsgWriter->GetVisible()) {
			if ((readkey() & 0xff) == '/') {
				MsgWriter->SetVisible(true);
				MsgWriter->PutChar('/');
			}
		}
		else {
			int key_val = readkey();
			if (isprint(key_val & 0xff))
				MsgWriter->PutChar(key_val);
			else if (key_val >> 8 == KEY_BACKSPACE)
				MsgWriter->PopChar();
			else if (key_val >> 8 == KEY_UP)
				MsgWriter->HistoryBackward();
			else if (key_val >> 8 == KEY_DOWN)
				MsgWriter->HistoryForward();
			else if (key_val >> 8 == KEY_ENTER || key_val >> 8 == KEY_ENTER_PAD) {
				MsgWriter->HistoryPush();
				const std::string Line = MsgWriter->GetText();
				/* If it starts with a '/' it is a command. */
				if (!Line.empty() && Line[0] == '/') {
					std::vector<std::string> WordList;
					std::string Params;
					std::istringstream iss(Line);
					std::string w;

					//split the words
					iss >> w;
					while (!w.empty()) {
						WordList.push_back(w);
						w.clear();
						iss >> w;
					}

					//cut the first word out
					unsigned int Pos = Line.find(' ');
					if (Pos != std::string::npos)
						Params = Line.substr(Pos + 1, Line.length());

					MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_NORMAL);
					MsgLister->Feed(Line);

					// remove the '/'
					WordList[0] = WordList[0].substr(1, WordList[0].length()-1);

					MsgLister->Ident();
					MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_NORMAL);
					ExecuteCmd(WordList, Params);
					MsgLister->Deident();
				}
				else
					SendPlayerMessage(Line);

				MsgWriter->Clear();
				if (AutoHideMsgWriter)
					MsgWriter->SetVisible(false);
			}
		}
	}
}


void NetworkClient::PollInput(const std::set<INPUT_COMMAND> *InputCommand) {
	if (!zigclient_c::is_connected() || !Displaying)
		return;

	if (MessageWriterActiveCounter > 10 &&
		InputCommand->find(IC_SWITCH_MESSAGE_WRITER) != InputCommand->end()) {
			MsgWriter->SetVisible(!MsgWriter->GetVisible());
			MessageWriterActiveCounter = 0;
	}

	if (StatisticsActiveCounter > 10 &&
		InputCommand->find(IC_SWITCH_STATISTICS) != InputCommand->end()) {
			Statistics->SetVisible(!Statistics->GetVisible());
			StatisticsActiveCounter = 0;
	}

	if (InputCommand->find(IC_ZOOM_IN) != InputCommand->end())
		SetCameraZoom(-ZOOM_GRANULARITY_PERCENT);
	if (InputCommand->find(IC_ZOOM_OUT) != InputCommand->end())
		SetCameraZoom(ZOOM_GRANULARITY_PERCENT);

	if (MatchOver)
		return;

	if (DisplayingScoreTableCounter > 10 &&
		InputCommand->find(IC_SWITCH_SCORE_TABLE_DISPLAY) != InputCommand->end()) {
			DisplayingScoreTable = !DisplayingScoreTable;
			DisplayingScoreTableCounter = 0;
	}

	std::set<INPUT_COMMAND> Empty;

	if (MsgWriter->GetVisible())
		InputCommand = &Empty;

	KeySet KS(*InputCommand);
	buffer_c out;
	out << KS;

	zigclient_c::send_packet(out);

	clear_keybuf();
}


void NetworkClient::SetCameraZoom(float Percent) {
	const int OffsetW = (int)(Percent / 100.f * UNITS_PER_SCREEN_W);
	const int OffsetH = (int)(Percent / 100.f * UNITS_PER_SCREEN_H);

	if (Percent > 0) {
		TargetCameraDim.Width += OffsetW;
		if (TargetCameraDim.Width > ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_W)
			TargetCameraDim.Width = (USHORT)(ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_W);

		TargetCameraDim.Height += OffsetH;
		if (TargetCameraDim.Height > ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_H)
			TargetCameraDim.Height = (USHORT)(ZOOM_OUT_MAX_RATIO * UNITS_PER_SCREEN_H);
	}
	else {
		TargetCameraDim.Width += OffsetW;
		if (TargetCameraDim.Width < ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_W)
			TargetCameraDim.Width = (USHORT)(ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_W);

		TargetCameraDim.Height += OffsetH;
		if (TargetCameraDim.Height < ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_H)
			TargetCameraDim.Height = (USHORT)(ZOOM_IN_MAX_RATIO * UNITS_PER_SCREEN_H);
    }
}


void NetworkClient::SendPlayerMessage(const std::string &Text, const std::string &lName) {
	PlayerMessage PM;
	PM.Text = Text;
	PM.To = lName;
	buffer_c Data;
	Data << PM;

	zigclient_c::send_message(Data, InfoMsgStream, true);
}


void NetworkClient::SendPlayerMessage(const std::string &Text) {
	SendPlayerMessage(Text, "");
}


void NetworkClient::SendTeamMessage(const std::string &Text) {
	//let's hope no-one will pick "/team/" as a name
	//even if someone does it's no hurt
	SendPlayerMessage(Text, "/team/");
}


void NetworkClient::DisplayPlayerMessage(const PlayerMessage* PM) {
	if (PM->To.empty()) //everyone received this message
		MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_CHAT_MESSAGE);
	else if (PM->To == "/team/") //a team message
		MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_TEAM_MESSAGE);
	else //private message, PM->To equals to this player name
		MsgLister->SetFontColor(MSG_LSTR_FONT_COLOR_PRIVATE_MESSAGE);

	MsgLister->Feed(get_element(OponentList, (int)PM->From).Name + ": " + PM->Text);
}


void NetworkClient::DisplayGameMessage(const GameMessage* GM) {
	const Oponent &WhoWasKilled = get_element(OponentList, (int)GM->WhoWasKilled);

	if (!GM->KilledWithWeapon) {
		if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
			if (WhoWasKilled.Team == MyTeam) {
				if (GM->WhoWasKilled == MyCliUID)
					SetGameMessageText("Who gave you the driving license?");
				else
					SetGameMessageText(WhoWasKilled.Name + " from your team exploded for no reason");
			}
			else
				SetGameMessageText(WhoWasKilled.Name + " from " + GetTeamName(WhoWasKilled.Team)
					+ " team just exploded.");
		}
		else if (GameOpt.GT == GT_DEATHMATCH) {
			if (GM->WhoWasKilled == MyCliUID)
				SetGameMessageText("Who gave you the driving license?");
			else
				SetGameMessageText(WhoWasKilled.Name + " was bored and just exploded");
		}

		return;
	}

	const Oponent &WhoKilled = get_element(OponentList, (int)GM->WhoKilled);

	if (GM->WhoKilled == MyCliUID && GM->WhoWasKilled == MyCliUID) {
		SetGameMessageText("Do you even know how to use this weapon?");
		return;
	}

	if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
		if (WhoKilled.Team == WhoWasKilled.Team) {
			if (WhoKilled.Team == MyTeam) {
				if (GM->WhoKilled == MyCliUID)
					SetGameMessageText("you killed " + WhoWasKilled.Name + " from your team");
				else if (GM->WhoWasKilled == MyCliUID)
					SetGameMessageText(WhoKilled.Name + " from your team killed you, hehe");
				else if (GM->WhoWasKilled == GM->WhoKilled)
					SetGameMessageText("Clumsiness inside your team! "
						+ WhoKilled.Name + " killed himself");
				else
					SetGameMessageText("Stupidity inside your team! "
						+ WhoKilled.Name + " killed " + WhoWasKilled.Name);
			}
			else {
				if (GM->WhoKilled == GM->WhoWasKilled)
					SetGameMessageText("Clumsiness inside " + GetTeamName(WhoKilled.Team) + " team! "
						+ WhoKilled.Name + " killed himself.");
				else
					SetGameMessageText("Stupidity inside " + GetTeamName(WhoKilled.Team) + " team! "
						+ WhoKilled.Name + " killed " + WhoWasKilled.Name);
			}
		}
		else if (WhoKilled.Team == MyTeam) {
			if (GM->WhoKilled == MyCliUID)
				SetGameMessageText("you killed " + WhoWasKilled.Name + " from "
					+ GetTeamName(WhoWasKilled.Team) + " team.");
			else
				SetGameMessageText(WhoKilled.Name + " from your team killed "
					+ WhoWasKilled.Name + " from " + GetTeamName(WhoWasKilled.Team) + " team");
		}
		else if (WhoWasKilled.Team == MyTeam) {
			if (GM->WhoWasKilled == MyCliUID)
				SetGameMessageText(WhoKilled.Name + " from " + GetTeamName(WhoKilled.Team) + " team killed you");
			else
				SetGameMessageText(WhoKilled.Name + " from " + GetTeamName(WhoKilled.Team) + " team killed "
					+ WhoWasKilled.Name + " from your team");
		}
		else
			SetGameMessageText(WhoKilled.Name + " from " + GetTeamName(WhoKilled.Team) + " team killed "
				+ WhoWasKilled.Name + " from " + GetTeamName(WhoWasKilled.Team) + " team");
	}
	else if (GameOpt.GT == GT_DEATHMATCH) {
		if (GM->WhoKilled == GM->WhoWasKilled)
			SetGameMessageText(WhoKilled.Name + " killed himself. What a fun thing to do.");
		else if (GM->WhoKilled == MyCliUID)
			SetGameMessageText("you killed " + WhoWasKilled.Name);
		else if (GM->WhoWasKilled == MyCliUID)
			SetGameMessageText(WhoKilled.Name + " killed you");
		else
			SetGameMessageText(WhoKilled.Name + " killed " + WhoWasKilled.Name);
	}
}


void NetworkClient::OponentConnected(const NewPlayerInfo* Info) {
	Oponent Opo;
	Opo.Name = Info->Name;
	Opo.Team = Info->Team;
	Opo.Ping = 0;
	Opo.Score.Points = 0;

	OponentList.insert(make_pair(Info->UID, Opo));

	if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
		if (TeamScoreList.find(Info->Team) == TeamScoreList.end())
			TeamScoreList.insert(make_pair(Info->Team, ScoreT()));
	}

	if (Name == Info->Name) {
		MyCliUID = Info->UID;
		MyTeam = Info->Team;
	}

	//skip first few notifications because this are usually not clients who have just connected
	if (Uptime > 2) {
		if (GameOpt.GT == GT_DEATHMATCH)
			SetGameInfoText(Info->Name + " joined the game");
		else if (GameOpt.GT == GT_TEAM_DEATHMATCH)
			SetGameInfoText(Info->Name + " joined the " + GetTeamName(Info->Team) + " team");
	}
}


void NetworkClient::OponentDisconnected(const PlayerGoneInfo* Info) {
	SetGameInfoText(get_element(OponentList, (int)Info->UID).Name + " left the game");

	OponentList.erase(Info->UID);
}


void NetworkClient::RankPlayers() {
	if (GameOpt.GT == GT_DEATHMATCH) {
		typedef multimap<char, Oponent*, greater<char> > SBPType;
		SBPType SortedByPoints;

		for (map<int, Oponent>::iterator x = OponentList.begin(); x != OponentList.end(); ++x)
			SortedByPoints.insert(make_pair(x->second.Score.Points, &x->second));

		int c = 1;
		int lp = -1;
		for (SBPType::iterator x = SortedByPoints.begin(); x != SortedByPoints.end(); ++x) {
			if (lp != x->second->Score.Points && lp != -1)
				++c;
			lp = x->second->Score.Points;
			x->second->Score.Rank = c;
			if (x != SortedByPoints.begin())
				x->second->Score.Advantage = x->second->Score.Points -
					SortedByPoints.begin()->second->Score.Points;
		}

		if (SortedByPoints.size() > 1)
			SortedByPoints.begin()->second->Score.Advantage =
				SortedByPoints.begin()->second->Score.Points -
				(++SortedByPoints.begin())->second->Score.Points;
		else
			SortedByPoints.begin()->second->Score.Advantage = 0;
	}
	else if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
		typedef multimap<char, ScoreT*, greater<char> > SBPType;
		SBPType SortedByPoints;

		for (map<TEAM, ScoreT>::iterator x = TeamScoreList.begin(); x != TeamScoreList.end(); ++x)
			SortedByPoints.insert(make_pair(x->second.Points, &x->second));

		int c = 1;
		int lp = -1;
		for (SBPType::iterator x = SortedByPoints.begin(); x != SortedByPoints.end(); ++x) {
			if (lp != x->first && lp != -1)
				++c;
			lp = x->first;
			x->second->Rank = c;

			if (x != SortedByPoints.begin())
				x->second->Advantage = x->second->Points - SortedByPoints.begin()->second->Points;
		}

		if (SortedByPoints.size() > 1)
			SortedByPoints.begin()->second->Advantage = SortedByPoints.begin()->second->Points -
				(++SortedByPoints.begin())->second->Points;
		else
			SortedByPoints.begin()->second->Advantage = 0;
	}
}


void NetworkClient::RemoveBogusTeams() {
	stack<map<TEAM, ScoreT>::iterator> ToRemove;

	for (map<TEAM, ScoreT>::iterator x = TeamScoreList.begin(); x != TeamScoreList.end(); ++x) {
		if (find_if(OponentList.begin(), OponentList.end(), SameTeam(x->first)) == OponentList.end())
			ToRemove.push(x);
	}

	while (!ToRemove.empty()) {
		TeamScoreList.erase(ToRemove.top());
		ToRemove.pop();
	}
}


void NetworkClient::DisplayScoreTable() {
	if (GameOpt.GT == GT_DEATHMATCH) {
		typedef multimap<char, Oponent*, greater<char> > SBPType;
		SBPType SortedByPoints;

		for (map<int, Oponent>::iterator x = OponentList.begin(); x != OponentList.end(); ++x)
			SortedByPoints.insert(make_pair(x->second.Score.Points, &x->second));

		int c = 0;
		for (SBPType::iterator x = SortedByPoints.begin(); x != SortedByPoints.end(); ++x) {
			TextObject *const PlayerName = &x->second->Score.NameDisplay;
			TextObject *const Points = &x->second->Score.PointsDisplay;
			TextObject *const Ping = &x->second->Score.PingDisplay;

			Points->SetLoc(Point<float>(UNITS_PER_SCREEN_W / 3 * 2, UNITS_PER_SCREEN_H / 4 + c * 50));
			Points->SetColor(makeacol(255, 255, 0, 191));
			Points->SetText(itoa(x->first));
			Points->Display();

			PlayerName->SetLoc(Point<float>(UNITS_PER_SCREEN_W / 3, UNITS_PER_SCREEN_H / 4 + c * 50));
			PlayerName->SetColor(makeacol(255, 191, 0, 191));
			PlayerName->SetText(string(itoa(c + 1)) + ". " + x->second->Name);
			PlayerName->Display();

			Ping->SetLoc(Point<float>(UNITS_PER_SCREEN_W / 3 * 2, UNITS_PER_SCREEN_H / 4 + c * 50 + 30));
			Ping->SetColor(makeacol(255, 255, 0, 191));
			Ping->SetText("ping: " + itoa(x->second->Ping));
			Ping->Display();

			++c;
		}
	}
	else if (GameOpt.GT == GT_TEAM_DEATHMATCH) {
		int TeamCounter = 0;
		for (map<TEAM, ScoreT>::iterator team = TeamScoreList.begin(); team != TeamScoreList.end(); ++team) {
			++TeamCounter;
			typedef multimap<char, Oponent*, greater<char> > SBPType;
			SBPType SortedByPoints;

			const float name_x = UNITS_PER_SCREEN_W / (TeamScoreList.size() * 2) + UNITS_PER_SCREEN_W / TeamScoreList.size() * (TeamCounter - 1) - 110;
			const float points_x = UNITS_PER_SCREEN_W / (TeamScoreList.size() * 2) + UNITS_PER_SCREEN_W / TeamScoreList.size() * (TeamCounter - 1) + 110;

			for (map<int, Oponent>::iterator opo = OponentList.begin(); opo != OponentList.end(); ++opo) {
				if (opo->second.Team == team->first)
					SortedByPoints.insert(make_pair(opo->second.Score.Points, &opo->second));
			}

			int PlayerCounter = 0;
			for (SBPType::iterator x = SortedByPoints.begin(); x != SortedByPoints.end(); ++x) {
				TextObject *const PlayerName = &x->second->Score.NameDisplay;
				TextObject *const Points = &x->second->Score.PointsDisplay;
				TextObject *const Ping = &x->second->Score.PingDisplay;

				PlayerName->SetLoc(Point<float>(name_x, UNITS_PER_SCREEN_H / 3 + PlayerCounter * 50));
				PlayerName->SetColor(GetTeamColor(x->second->Team));
				PlayerName->SetText(string(itoa(PlayerCounter + 1)) + ". " + x->second->Name);
				PlayerName->Display();

				Points->SetLoc(Point<float>(points_x, UNITS_PER_SCREEN_H / 3 + PlayerCounter * 50));
				Points->SetColor(GetTeamColor(x->second->Team));
				Points->SetText(itoa(x->second->Score.Points));
				Points->Display();

				Ping->SetLoc(Point<float>(points_x, UNITS_PER_SCREEN_H / 3 + PlayerCounter * 50 + 30));
				Ping->SetColor(GetTeamColor(x->second->Team));
				Ping->SetText("ping: " + itoa(x->second->Ping));
				Ping->Display();

				++PlayerCounter;
			}

			team->second.PointsDisplay.SetLoc(Point<float>(points_x, UNITS_PER_SCREEN_H / 3 + PlayerCounter * 50));
			team->second.PointsDisplay.SetColor(makeacol32(255, 0, 0, 191));
			team->second.PointsDisplay.SetText(itoa(team->second.Points));
			team->second.PointsDisplay.Display();

			team->second.NameDisplay.SetLoc(Point<float>(name_x + 40, UNITS_PER_SCREEN_H / 3 - 50));
			team->second.NameDisplay.SetColor(makeacol32(255, 255, 255, 191));
			team->second.NameDisplay.SetText(GetTeamName(team->first) + " TEAM");
			team->second.NameDisplay.Display();
		}
	}
}


void NetworkClient::SomeoneWon(const WinnerInfo* WI) {
	if (GameOpt.GT == GT_DEATHMATCH) {
		if (WI->ClientID == MyCliUID)
			SetGameStatusText("YOU ARE THE WINNER!");
		else
			SetGameStatusText(get_element(OponentList, (int)WI->ClientID).Name + " wins the match!");
	}
	else if (GameOpt.GT == GT_TEAM_DEATHMATCH)
		SetGameStatusText(GetTeamName(get_element(OponentList, (int)WI->ClientID).Team) + " TEAM WINS THE MATCH!");

	//send to server the message saying that all keys are released
	//(yeah, like he's going believe this one)
	std::set<INPUT_COMMAND> EmptyInput;
	PollInput(&EmptyInput);

	DisplayingScoreTable = true;
	MatchOver = true;
	ScrMask->SetVisible(true);
}


std::string GetRandomBrightParticle() {
	const int r = rand() % 3;
	switch (r) {
		case 0:
			return "defs/objects/particles/yellow.nso";
		case 1:
			return "defs/objects/particles/orange.nso";
	}
	return "defs/objects/particles/orange.nso";
}


std::string GetRandomRedParticle() {
	const int r = rand() % 3;
	switch (r) {
		case 0:
			return "defs/objects/particles/red.nso";
		case 1:
			return "defs/objects/particles/orange.nso";
	}
	return "defs/objects/particles/orange.nso";
}


std::string GetRandomBlueParticle() {
	const int r = rand() % 3;
	switch (r) {
		case 0:
			return "defs/objects/particles/blue.nso";
		case 1:
			return "defs/objects/particles/blue.nso";
	}
	return "defs/objects/particles/green.nso";
}


#endif //SERVER_ONLY
