/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "sprite.h"

#include "missile.h"

#include "particle.h"
#include "sound.h"
#include "sparklet_utils.h"
#include "network_server.h"

#define FLY_SOUND_STR std::string("fly_sound")


extern NetworkServer *gNetworkServer;
extern float gTimeElapsed;


Missile::Missile(const TiXmlElement* Att, int lClientID, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	Projectile(Att, Scale),
	ControlledObject(lClientID),
	CameraClientID(lClientID),
	OldRotation(0.f),
	Rotation(0.f),
	NextFocusedObject(NULL) {

	//capture the camera
	gNetworkServer->SetCamera(CameraClientID, ObjectUID);

	const TiXmlElement *Root = Parser->GetRoot();

	BaseSprite = Object::GetBaseSprite(Root);

	FlySound = new Sound(XMLParser::GetChild(Root, FLY_SOUND_STR), Geo);
	gNetworkServer->AddSound(FlySound, false);
}


Missile::~Missile() {
	delete FlySound;
}


void Missile::OnUpdate() {
	BaseSprite->SetDirection(arg(Velocity));

	Velocity += (Impulse + CollidingImpulse) / Mass;
	Velocity = polar(abs(Velocity), BoundAngle(arg(Velocity) + Rotation * gTimeElapsed));
	Projectile::OnUpdate();
}


bool Missile::OnKeyDown(const std::set<INPUT_COMMAND> &Key) {
	//update rotation
	if (ControlledObject::Pressed(IC_TURN_LEFT, Key)) {
		Rotation = -1.8f;
	}
	if (ControlledObject::Pressed(IC_TURN_RIGHT, Key)) {
		Rotation = 1.8f;
	}
	if (ControlledObject::Pressed(IC_SECONDARY_FIRE, Key)) {
		Explode();
		Unlink();
	}

	return true;
}


bool Missile::OnKeyUp(const std::set<INPUT_COMMAND> &Key) {
	//set rotation to zero upon keyup event
	if (!ControlledObject::Pressed(IC_TURN_LEFT, Key) && !ControlledObject::Pressed(IC_TURN_RIGHT, Key))
		Rotation = 0;

	return true;
}


void Missile::GetCompleteData(MessageCoder &MC) const {
	RotationInfo *RI = new RotationInfo;
	//Why do we have to divide by 2??? To make it work. Maybe someday I'll find out why.
	RI->Direction = arg(Velocity) / 2.f;
	RI->Rotation = Rotation / 2.f;
	MC.Put(RI);

	Projectile::GetCompleteData(MC);
}


void Missile::CheckData() {
	if (OldRotation != Rotation) {
		RotationChanged = true;
		OldRotation = Rotation;
	}
	else
		RotationChanged = false;

	Projectile::CheckData();
}


void Missile::GetChangedData(MessageCoder &MC) const {
	if (RotationChanged) {
		RotationInfo *RI = new RotationInfo;
		//Why do we have to divide by 2??? To make it work. Maybe someday I'll find out why.
		RI->Direction = arg(Velocity) / 2.f;
		RI->Rotation = Rotation / 2.f;
		MC.Put(RI);
	}

	Projectile::GetChangedData(MC);
}


void Missile::Damage(const PhysicalObject *Obj) {
	Explode();
	Unlink();

	Projectile::Damage(Obj);
}


Particle* Missile::Explode() {
	gNetworkServer->RemoveSound(FlySound->SoundUID);
	//rember the object to pass the camera
	NextFocusedObject = Projectile::Explode();

	return NextFocusedObject;
}


void Missile::Unlink() {
	//release the camera
	gNetworkServer->ReleaseCamera(CameraClientID, ObjectUID);

	//pass the camera to the next object
	//this is done in order for the user to see the explosion, rather than just jumping back to the
	//ship immediately
	if (NextFocusedObject)
		NextFocusedObject->TakeCamera(CameraClientID);

	ControlledObject::Unlink();
	Projectile::Unlink();
}
