/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "message_coder.h"


#ifdef WIN32
	#include "sparklet_utils.h"
#endif


#include <vector>
#include <cmath>
#include <numeric>
#include <algorithm>


using namespace std;


MessageCoder::~MessageCoder() {
	for_each(FOIOList.begin(), FOIOList.end(), FOIODestroy());
	for_each(PIOList.begin(), PIOList.end(), PIODestroy());
}


void MessageCoder::Put(const serializable_c *inf) {
	CommonData << *inf;
	delete inf;

	++MsgCounter;
}


void MessageCoder::Put(const FullObjInfo *FOI) {
	FOIOList.insert(FOIObject(new OUIDInfo(*CurrentObjUID), FOI));
}


void MessageCoder::Put(const ParticleInfo *PI) {
	PIOList.insert(PIObject(new OUIDInfo(*CurrentObjUID), PI));
}


void MessageCoder::PutRemove(const OUIDInfo Die) {
	RemoveList.insert(Die);
}


void MessageCoder::Begin(const OUIDInfo &ObjUID) {
	CurrentObjUID = new OUIDInfo(ObjUID);
	MsgCounter = 0;
	CommonData.clear();
}


void MessageCoder::End() {
	if (MsgCounter != 0) {
		RawData << *CurrentObjUID;
		RawData.putByte(MsgCounter);
		RawData.putBlock(CommonData.data(), CommonData.size());
	}

	delete CurrentObjUID;
}


void MessageCoder::GetCodedMessage(buffer_c &buf) {
	CodeFOI(buf);
	CodePI(buf);

	buf.put32K(RawData.size());
	buf.putBlock(RawData.data(), RawData.size());

	CodeRemove(buf);
}


void MessageCoder::Decode(buffer_c &cod, buffer_c &dec) {
	DecodeFOI(cod, dec);
	DecodePI(cod, dec);

	const USHORT RawLength = cod.get32K();
	dec.putBlock(cod.data_cur(), RawLength);
	cod.seek(cod.getpos() + RawLength);

	DecodeRemove(cod, dec);

	dec.seek(0);
}


inline bool MessageCoder::AreFOIONear::operator() (const FOIObject& x, const FOIObject& y) const {
	return (*(y.ObjID) - *(x.ObjID) == 1
		&& y.FOI->XMLFilePathCode == x.FOI->XMLFilePathCode
		&& y.FOI->Scale == x.FOI->Scale
		&& y.FOI->Surface == x.FOI->Surface
		&& y.FOI->Dim.Width == x.FOI->Dim.Width
		&& y.FOI->Dim.Height == x.FOI->Dim.Height
		&& x.FOI->Dim.Width == x.FOI->Dim.Height);
}


inline bool MessageCoder::ArePIONear::operator() (const PIObject& x, const PIObject& y) const {
	return (*(y.ObjID) - *(x.ObjID) == 1
		&& fabs(y.PI->VI.Loc.x - x.PI->VI.Loc.x) <= 8.f
		&& fabs(y.PI->VI.Loc.y - x.PI->VI.Loc.y) <= 8.f
		&& fabs(abs(y.PI->VI.Velocity) - abs(x.PI->VI.Velocity)) < 128.f);
}


inline MessageCoder::PIODiff MessageCoder::DiffPIO::operator()(const PIObject& x, const PIObject& y) const {
	PIODiff PIOD;

	PIOD.Angle = arg(x.PI->VI.Velocity);
	PIOD.LifeTime = x.PI->LifetimeLeft;

	PIOD.LocDiff.x = x.PI->VI.Loc.x - y.PI->VI.Loc.x;
	PIOD.LocDiff.y = x.PI->VI.Loc.y - y.PI->VI.Loc.y;

	PIOD.VelocityDiff = abs(x.PI->VI.Velocity) - abs(y.PI->VI.Velocity);

	return PIOD;
}


//255 / PI
#define ANGF 40.58575f

inline void MessageCoder::WritePIODiff::operator() (const PIODiff &x) const {
	Data.putBytes(lroundf(x.Angle * ANGF));
	Data.putByte(lroundf(x.LifeTime * 100.f));
	Data.putBytes(lroundf(x.VelocityDiff));

	const UCHAR b = ((lroundf(x.LocDiff.y) + 8) << 4) | (lroundf(x.LocDiff.x) + 8);
	Data.putByte(b);
}


inline void MessageCoder::ReadPIODiff::operator() (PIODiff &x) const {
	x.Angle = Data.getBytes() / ANGF;
	x.LifeTime = Data.getByte() / 100.f;
	x.VelocityDiff = Data.getBytes();

	const UCHAR b = Data.getByte();
	x.LocDiff.x = (b >> 4) - 8;
	x.LocDiff.y = (b & 0x0f) - 8;
}


inline void MessageCoder::WriteFOIOcod::operator() (const FOIObject &x) {
	Data << *(x.ObjID) << *(x.FOI);
}


inline void MessageCoder::WritePIOcod::operator() (const PIObject &x) {
	Data << *(x.ObjID) << *(x.PI);
}


inline void MessageCoder::WriteFOIOdec::operator() (const FOIObject &x) {
	Data << *(x.ObjID) << (UCHAR)1 << *(x.FOI);
}


inline void MessageCoder::WritePIOdec::operator() (const PIObject &x) {
	Data << *(x.ObjID) << (UCHAR)1 << *(x.PI);
}


const UCHAR CODED = 0x80;
const UCHAR NOT_CODED = 0x0;


void MessageCoder::CodeFOI(buffer_c &Data) {
	Data.put32K(FOIOList.size());

	FOIOListType::iterator last;
	FOIOListType::iterator last_last = FOIOList.begin();
	FOIOListType::iterator first = FOIOList.begin();

	vector<FOIOListType::value_type> CannotCode;

	while (first != FOIOList.end()) {
		last = adjacent_find(first, FOIOList.end(), binary_negate<AreFOIONear>(AreFOIONear()));

		if (last != FOIOList.end())
			++last;

		FOIOListType::iterator until = first;

		do {
			if (distance(until, last) > 127)
				advance(until, 127);
			else
				until = last;
				
			USHORT SimilarObjectCount = distance(first, until);

			if (SimilarObjectCount > 1) {
				//write header and first FOI
				const UCHAR CodHeader = CODED | SimilarObjectCount - 1;
				Data << CodHeader << *(first->ObjID) << *first->FOI;

				//write unique data, but skip the first object because we alread wrote it
				bool is_first = true;
				for (FOIOListType::iterator x = first; x != until; ++x) {
					if (!is_first)
						Data.putByte((*x).FOI->Dim.Width);
					is_first = false;
				}

				//save FOIs that can't be coded
				for (FOIOListType::iterator x = last_last; x != first; ++x)
					CannotCode.push_back(*x);

				last_last = until;
				first = until;
			}
		} while (until != last);

		first = last;
	}

	while (last_last != FOIOList.end()) {
		CannotCode.push_back(*last_last);
		++last_last;
	}

	vector<FOIOListType::value_type>::iterator firstcc = CannotCode.begin();
	vector<FOIOListType::value_type>::iterator lastcc = firstcc;
	
	do {
		if (distance(lastcc, CannotCode.end()) > 127)
			advance(lastcc, 127);
		else
			lastcc = CannotCode.end();

		const USHORT NotSimilarObjectCount = distance(firstcc, lastcc);
		if (NotSimilarObjectCount) {
			const UCHAR Header = NOT_CODED | NotSimilarObjectCount;
			Data << Header;
			for_each(firstcc, lastcc, WriteFOIOcod(Data));
		}

		firstcc = lastcc;
	} while(lastcc != CannotCode.end());
}


void MessageCoder::CodePI(buffer_c &Data) {
	Data.put32K(PIOList.size());

	PIOListType::iterator last;
	PIOListType::iterator last_last = PIOList.begin();
	PIOListType::iterator first = PIOList.begin();

	vector<PIOListType::value_type> CannotCode;

	while (first != PIOList.end()) {
		last = adjacent_find(first, PIOList.end(), binary_negate<ArePIONear>(ArePIONear()));

		if (last != PIOList.end())
			++last;

		PIOListType::iterator until = first;

		do {
			if (distance(until, last) > 127)
				advance(until, 127);
			else
				until = last;

			USHORT SimilarObjectCount = distance(first, until);
			if (SimilarObjectCount > 1) {
				vector<PIODiff> Diff(SimilarObjectCount);

				adjacent_difference(first, until, Diff.begin(), DiffPIO());

				//write header and first PI
				const UCHAR CodHeader = CODED | SimilarObjectCount - 1;
				Data << CodHeader << *(first->ObjID) << *first->PI;

				//write differences
				for_each(Diff.begin() + 1, Diff.end(), WritePIODiff(Data));

				//save PIs that can't be coded
				for (PIOListType::iterator x = last_last; x != first; ++x)
					CannotCode.push_back(*x);

				last_last = until;
				first = until;
			}
		} while(until != last);

		first = last;
	}

	//write POIs that cannot be coded
	while (last_last != PIOList.end()) {
		CannotCode.push_back(*last_last);
		++last_last;
	}

	vector<PIOListType::value_type>::iterator firstcc = CannotCode.begin();
	vector<PIOListType::value_type>::iterator lastcc = firstcc;
	
	do {
		if (distance(lastcc, CannotCode.end()) > 127)
			advance(lastcc, 127);
		else
			lastcc = CannotCode.end();

		const USHORT NotSimilarObjectCount = distance(firstcc, lastcc);
		if (NotSimilarObjectCount) {
			const UCHAR Header = NOT_CODED | NotSimilarObjectCount;
			Data << Header;
			for_each(firstcc, lastcc, WritePIOcod(Data));
		}

		firstcc = lastcc;
	} while(lastcc != CannotCode.end());
}


struct IsDiffOne : public std::binary_function <OUIDInfo, OUIDInfo, bool> {
	bool operator() (const OUIDInfo &x, const OUIDInfo &y) const {
		return y - x == 1;
	}
};


struct WriteRemove {
	WriteRemove(buffer_c &lData) : Data(lData) {};
	void operator() (const OUIDInfo &x) {
		Data << x;
	}
private:
	buffer_c &Data;
};


void MessageCoder::CodeRemove(buffer_c &Data) {
	Data.put32K(RemoveList.size());

	RemoveListType::iterator last;
	RemoveListType::iterator last_last = RemoveList.begin();
	RemoveListType::iterator first = RemoveList.begin();

	vector<OUIDInfo> CannotCode;

	while (first != RemoveList.end()) {
		last = adjacent_find(first, RemoveList.end(), binary_negate<IsDiffOne>(IsDiffOne()));

		if (last != RemoveList.end())
			++last;

		RemoveListType::iterator until = first;

		do {
			if (distance(until, last) > 127)
				advance(until, 127);
			else
				until = last;

			USHORT ArrayedIDsCount = distance(first, until);
			if (ArrayedIDsCount > 1) {
				//write header and first ObjID
				const UCHAR CodHeader = CODED | ArrayedIDsCount;
				Data << CodHeader << *first;

				//save ObjIDs that can't be coded
				for (RemoveListType::iterator x = last_last; x != first; ++x)
					CannotCode.push_back(*x);

				last_last = until;
				first = until;
			}
		} while (until != last);

		first = last;
	}

	//write ObjIDs that can't be coded, if there are some
	while (last_last != RemoveList.end()) {
		CannotCode.push_back(*last_last);
		++last_last;
	}

	vector<RemoveListType::value_type>::iterator firstcc = CannotCode.begin();
	vector<RemoveListType::value_type>::iterator lastcc = firstcc;
	
	do {
		if (distance(lastcc, CannotCode.end()) > 127)
			advance(lastcc, 127);
		else
			lastcc = CannotCode.end();

		const USHORT NotSimilarObjectCount = distance(firstcc, lastcc);
		if (NotSimilarObjectCount) {
			const UCHAR Header = NOT_CODED | NotSimilarObjectCount;
			Data << Header;
			for_each(firstcc, lastcc, WriteRemove(Data));
		}

		firstcc = lastcc;
	} while(lastcc != CannotCode.end());
}


void MessageCoder::DecodeFOI(buffer_c &cod, buffer_c &dec) {
	const USHORT n = cod.get32K();
	USHORT x = 0;

	while (x < n) {
		const UCHAR Header = cod.getByte();
		const char PendingObjects = Header & 0x7f; //0x80 - 1 = 01111111

		if (Header & CODED) {
			//read the first object, it is consistent
			const OUIDInfo *FirstObjID = static_cast<OUIDInfo*>(cod.getObject());
			const FullObjInfo *FirstFOI = static_cast<FullObjInfo*>(cod.getObject());
			FOIObject FirstFOIO(FirstObjID, FirstFOI);

			vector<FOIObject> FOIO;
			FOIO.reserve(PendingObjects + 1);
			FOIO.push_back(FirstFOIO);

			for (USHORT c = 0; c < PendingObjects; ++c) {
				//read per-object unique data
				const UCHAR Width = cod.getByte();
				//create new object similar to the first one
				FullObjInfo *FOI = new FullObjInfo(*FirstFOI);
				FOI->Dim = Size(Width, Width);
				const OUIDInfo *NewID = new OUIDInfo(*FirstObjID + c + 1);
				FOIO.push_back(FOIObject(NewID, FOI));
			}

			for_each(FOIO.begin(), FOIO.end(), WriteFOIOdec(dec));
			for_each(FOIO.begin(), FOIO.end(), FOIODestroy());

			x += PendingObjects + 1;
		}
		else {
			for (int y = 0; y < PendingObjects; ++y) {
				const OUIDInfo *OID = static_cast<OUIDInfo*>(cod.getObject());
				const FullObjInfo *FOI = static_cast<FullObjInfo*>(cod.getObject());
				dec << *OID << (UCHAR)1 << *FOI;
				delete OID;
				delete FOI;
			}

			x += PendingObjects;
		}
	}
}


void MessageCoder::DecodePI(buffer_c &cod, buffer_c &dec) {
	const USHORT n = cod.get32K();
	USHORT x = 0;

	while (x < n) {
		const UCHAR Header = cod.getByte();
		const char PendingObjects = Header & 0x7f; //0x80 - 1 = 01111111

		if (Header & CODED) {
			const OUIDInfo *FirstObjID = static_cast<OUIDInfo*>(cod.getObject());
			const ParticleInfo *FirstPI = static_cast<ParticleInfo*>(cod.getObject());
			PIObject FirstPIO(FirstObjID, FirstPI);
			OUIDInfo ObjID(*FirstObjID);

			vector<PIODiff> PIOD(PendingObjects);
			for_each(PIOD.begin(), PIOD.end(), ReadPIODiff(cod));

			vector<PIObject> PIO(PendingObjects + 1);

			//hand-made implementation similar to sdt::partial_sum
			//not quite sure what's the difference, but who cares
			vector<PIODiff>::iterator item = PIOD.begin();
			vector<PIObject>::iterator out = PIO.begin();
			*out = FirstPIO;

			while (item != PIOD.end()) {
				const PIObject *Before = &(*out);
				const PIODiff *Current = &(*item);
				++item;

				const float MotLen = Current->VelocityDiff + abs(Before->PI->VI.Velocity);
				const float Angle = Current->Angle;
				const float Time = Current->LifeTime;
				const float LocX = Current->LocDiff.x + Before->PI->VI.Loc.x;
				const float LocY = Current->LocDiff.y + Before->PI->VI.Loc.y;

				ParticleInfo *PI = new ParticleInfo;
				PI->VI.Velocity = polar(MotLen, Angle);
				PI->VI.Loc = Point<float>(LocX, LocY);
				PI->LifetimeLeft = Time;

				++ObjID;
				const OUIDInfo *UID = new OUIDInfo(ObjID);
				++out;
				*out = PIObject(UID, PI);
			}

			for_each(PIO.begin(), PIO.end(), WritePIOdec(dec));
			for_each(PIO.begin(), PIO.end(), PIODestroy());

			x += PendingObjects + 1;
		}
		else {
			for (int y = 0; y < PendingObjects; ++y) {
				const OUIDInfo *OID = static_cast<OUIDInfo*>(cod.getObject());
				const ParticleInfo *PI = static_cast<ParticleInfo*>(cod.getObject());
				dec << *OID << (UCHAR)1 << *PI;
				delete OID;
				delete PI;
			}

			x += PendingObjects;
		}
	}
}


void MessageCoder::DecodeRemove(buffer_c &cod, buffer_c &dec) {
	const USHORT n = cod.get32K();
	USHORT x = 0;

	while (x < n) {
		const UCHAR Header = cod.getByte();
		const char PendingIDs = Header & 0x7f; //0x80 - 1 = 01111111

		if (Header & CODED) {
			const OUIDInfo *FirstObjID = static_cast<OUIDInfo*>(cod.getObject());
			for (USHORT c = 0; c < PendingIDs; ++c) {
				const OUIDInfo *OID = new OUIDInfo(*FirstObjID + OUIDInfo(c));
				dec << *OID << (UCHAR)0;
				delete OID;
			}

			x += PendingIDs;
			delete FirstObjID;
		}
		else {
			for (int y = 0; y < PendingIDs; ++y) {
				const OUIDInfo *OID = static_cast<OUIDInfo*>(cod.getObject());
				dec << *OID << (UCHAR)0;
				delete OID;
			}

			x += PendingIDs;
		}
	}
}
