/***************************************************************************
                          game.cpp  -  description
                             -------------------
    begin                : Fri Feb 28 2003
    copyright            : (C) 2003 by Milan Mimica
    email                : milan.mimica@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "game.h"

#ifndef SERVER_ONLY
#include <alleggl.h>
#endif
#include <allegro.h>

#ifdef WIN32
	#include <winalleg.h>
#endif

//workaround for MASkinG and xlib collision
#undef RootWindow
#undef MessageBox

#include "game_gui.h"
#include "network_server.h"
#include "network_client.h"
#include "game_input.h"

#include "sparklet_utils.h"

#include "bitmap_server.h"
#include "font_server.h"
#include "sample_server.h"
#include "xmlserver.h"
#include "initializer.h"

#include "exceptions.h"

#ifdef SERVER_ONLY
	#include <unistd.h>
	#include <signal.h>
	#include <stdlib.h>
	#include <time.h>
	#include <sys/time.h>
#endif


#include <queue>

const int Interval = 50;
float gTimeElapsed = 1.f / (float)Interval;
const int gInputRatio = 5;

volatile bool Finish = false;
volatile int TickCounter = 0;

#ifndef SERVER_ONLY

	bool SwitchDisplay = false;

	extern bool DoFullRedraw;

	std::queue<int> FrameQueue;
	volatile int Frames = 0;
	volatile int gFPS = 200;
	const int FramesInterval = 4;

	FontServer *gFntSrv = NULL;

#endif //SERVER_ONLY

BitmapServer *gBmpSrv = NULL;
SampleServer *gSmpSrv = NULL;
XMLServer *gXMLSrv = NULL;
Initializer *gInitializer = NULL;

#ifndef SERVER_ONLY
bool Game::Start() {
	const std::string ConfigPath(FixFilenameSlashes(std::string(DATA_PREFIX) + std::string("profile.ini")));
	_DISPLAY(ConfigPath);
	set_config_file(ConfigPath.c_str());

	push_config_state();

	if (!zig_init(gLC)) _ERROR_;
	if (!InitTimers()) { _ERROR_; return false; }
	if (!InitGraphics()) { _ERROR_; return false; }
	if (!InitSound()) _ERROR_;
	if (!GameGUI::Init()) { _ERROR_; return false; }

	_CHECKPOINT_;

	if (!PerformOGLTest()) {
		const char *msg ="Sorry!\n"
				"No required OpenGL support was found on this computer.\n"
				"We suggest installing the latest graphic card driver.";
		_ERROR_;
		_SAY(msg);
		if (is_windowed_mode())
			allegro_message(msg);
		return false;
	}

	install_keyboard();
	install_mouse();

#ifdef ALLEGRO_WINDOWS
	/*Alert if crapy and bugy MS drivers are detected.*/
	if (AreMicrosoftDrivers()) {
		allegro_message("WARNING!!!\n"
			"Sparklet has detected graphic card drivers provided by Microsoft Windows that\n"
			"DO NOT include proper OpenGL support. The game will probably crash. We strongly\n"
			"recommend to install latest drivers provided by graphic card manufacturer!");
	}
#endif

	pop_config_state();

	gXMLSrv = new XMLServer;
	gBmpSrv = new BitmapServer;
	gSmpSrv = new SampleServer;
	try {
		gFntSrv = new FontServer;
	} catch (const ExFileNotFound &fnf) {
		_ERROR_;
		_SAY(fnf.GetMsg());
		return false;
	}

	_CHECKPOINT_;

	gInitializer = new Initializer;

	{

	NetworkServer Server;
	NetworkClient Client;
	GameInput Input;
	GameGUI Gui(&Server, &Client);
	_CHECKPOINT_;

	const bool yield = get_config_int("server", "yield", false);

	USHORT Counter1 = 0;
	USHORT Counter2 = 0;

	while (!Finish) {
		if (SwitchDisplay && Counter2 >= 20) {
			OnDisplaySwitch(&Client, &Gui);
			Counter2 = 0;
		}
		SwitchDisplay = false;

		Finish = Gui.Process();
		Server.Process();
		Client.Process();

		while (TickCounter) {
			Server.Update();
			Client.Update();
			Input.Process();

			Counter1++;
			if (Counter1 == gInputRatio) {
				Counter1 = 0;
				Client.PollInput(&Input.Update());
				SwitchDisplay |= Input.IsSwitchKeyPressed();
			}

			++Counter2;

			TickCounter--;
			if (TickCounter > 10)
				TickCounter = 0;
		}

		if (Client.Render())
			++Frames;

		if (Server.HighLoad() || Client.HighLoad()) {
			if (yield)
				rest(0);
			else
				rest(1);
		}
		else
			rest(5);
	}

	_CHECKPOINT_;

	Client.Disconnect();
	Server.Stop();

	/*
		Clent, Server and Gui are gone. We can delete other objects now.
	*/
	}

	GameGUI::DeInit();
	delete gInitializer;
	zig_shutdown(gLC);
	delete gFntSrv;
	delete gBmpSrv;
	delete gSmpSrv;
	delete gXMLSrv;

	flush_config_file();

	return true;
}
#endif //SERVER_ONLY


#ifdef SERVER_ONLY
/* CTRL-C */
void sigint_handler (int) {
	Finish = true;
}
#endif


#ifdef SERVER_ONLY
void SP_Sleep(long ms) {
	timespec delay, rem;
	delay.tv_sec = 0;
	delay.tv_nsec = ms * 1000 * 1000;

	while (true) {
		if (nanosleep(&delay, &rem) != 0) {
			if (errno == EINTR)
				delay = rem;
			else
				return;
		}
		else
			return;
	}
}
#endif


#ifdef SERVER_ONLY
bool Game::StartServerOnly(const std::string &ProfileFilePath, bool RunInBackground) {
	struct sigaction sa;
	memset(&sa, 0, sizeof(struct sigaction));
	sa.sa_handler = sigint_handler;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);

	set_config_file(ProfileFilePath.c_str());
	_DISPLAY(ProfileFilePath);

	push_config_state();

	if (!zig_init(gLC)) { _ERROR_; return false; }

	pop_config_state();

	_CHECKPOINT_;

	gXMLSrv = new XMLServer;
	gSmpSrv = new SampleServer;
	gBmpSrv = new BitmapServer;

	_CHECKPOINT_;

	gInitializer = new Initializer;

	{

	NetworkServer Server;
	_CHECKPOINT_;
	ServerOptions SO;
	SO.MaxClients = get_config_int("server", "max_players", 10);
	SO.WelcomeMessage = get_config_string("server", "msg", "");
	SO.Name = get_config_string("server", "name", "sparklet server");
	SO.Password = get_config_string("server", "password", "");
	SO.Port = get_config_int("server", "port", 2020);
	SO.Tick = gTimeElapsed;
	SO.NetRatio = get_config_int("server", "net_ratio", 5);
	SO.ClientTimeout = get_config_int("server", "client_timeout", 8);
	SO.MinPacketCompressSize = get_config_int("server", "compress", 0) ? 250 : -1;

	GameOptions GO;
	GO.GT = get_config_int("server", "game_type", 0) ? GT_TEAM_DEATHMATCH : GT_DEATHMATCH;
	GO.FrendlyFire = get_config_int("server", "frendly_fire", 0);
	GO.PointsMax = get_config_int("server", "max_points", 10);

	int argc;
	char **argv = get_config_argv("server", "map_files", &argc);
	for (int x = 0; x < argc; ++x)
		GO.MapFileNameList.push_back(argv[x]);

	if (Server.Start(SO, GO)) {
		_SAY("Server is running now!");

		if (RunInBackground) {
			if (fork() != 0) {
				_SAY("Forking background process.");
				printf("Server is up and running in background!\n");
				printf("Use `killall sparklet_s-bin` to quit.\n");
				exit(EXIT_SUCCESS);
			}
		}
		else {
			printf("Server is up and running!\n");
			printf("Press Ctrl+C to quit.\n");
		}
	}
	else {
		_ERROR_;
		_SAY("Server failed to run! Check the configuration.");
		printf("Server failed to run! Check the configuration.\n");
		Finish = true;
	}

	const bool yield = get_config_int("server", "yield", false);

	double LastTickTime = get_time();

	while (!Finish) {
		Server.Process();

		double now = get_time();
		if ((now - LastTickTime) >= gTimeElapsed) {
			LastTickTime += gTimeElapsed;

			if ((now - LastTickTime) > gTimeElapsed * 10.0)
				LastTickTime = now;

			Server.Update();
		}

		if (Server.HighLoad()) {
			if (yield)
				sched_yield();
			else
				SP_Sleep(2);
		}
		else
			SP_Sleep(5);
	}

	_CHECKPOINT_;

	Server.Stop();

	/*
		Server is now gone. We can delete other objects.
	*/
	}

	delete gInitializer;
	zig_shutdown(gLC);
	delete gBmpSrv;
	delete gSmpSrv;
	delete gXMLSrv;

	return true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
void Game::OnDisplaySwitch(NetworkClient *cli, GameGUI *gui) {
	if (Showing == DT_GUI) {
		Showing = DT_GAME;

		glEnable(GL_TEXTURE_2D);

		cli->Show();
		gui->Hide();
	}
	else {
		Showing = DT_GUI;

		glColor4f(1, 1, 1, 1);
		glDisable(GL_TEXTURE_2D);

		gui->Show();
		cli->Hide();
	}
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
void IncreaseTickCounter() {
	++TickCounter;
}
END_OF_FUNCTION(IncreaseTickCounter)
#endif


#ifndef SERVER_ONLY
void AddFrames() {
	gFPS -= FrameQueue.front();
	gFPS += Frames;
	FrameQueue.pop();
	int lFrames = Frames;
	FrameQueue.push(lFrames);
	Frames = 0;
}
END_OF_FUNCTION(AddFrames)
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
bool Game::InitTimers() {
	if (install_timer() < 0)
		return false;

	//allegro timer stuff
	LOCK_VARIABLE(Frames)
	LOCK_VARIABLE(TickTime)
	LOCK_FUNCTION(AddFrames)

	//fill initial FPS values to 50 fps
	for (int x = 0; x < FramesInterval; ++x)
		FrameQueue.push(200 / FramesInterval);

	if (install_int_ex(AddFrames, BPS_TO_TIMER(FramesInterval)) == -1)
		return false;

	if (install_int_ex(IncreaseTickCounter, BPS_TO_TIMER(Interval)) == -1)
		return false;

	return true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
void WindowSwitchIn() {
	DoFullRedraw = true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
bool Game::InitGraphics() {
	//init alegrogl
	if (install_allegro_gl() != 0) {
		_ERROR_;
		_SAY("Couldn't init allegroGL!");
		return false;
	}

	pop_config_state();
	const Size ScreenDim = ParseDimensionString(get_config_string("graphics", "resolution", "1024x768"));
	const bool win = get_config_int("graphics", "windowed", false);
	const int freq = get_config_int("graphics", "try_frequency", 0);
	push_config_state();

	if (freq)
		request_refresh_rate(freq);

	//set up allegrogl
	allegro_gl_clear_settings();
	allegro_gl_set(AGL_COLOR_DEPTH, 32);
	allegro_gl_set(AGL_DOUBLEBUFFER, 1);
	allegro_gl_set(AGL_WINDOWED, win);
	allegro_gl_set(AGL_RENDERMETHOD, 1);
	allegro_gl_set(AGL_REQUIRE, AGL_WINDOWED);
	allegro_gl_set(AGL_SUGGEST, AGL_COLOR_DEPTH | AGL_DOUBLEBUFFER | AGL_RENDERMETHOD);

	//set graphics mode
	if (set_gfx_mode(GFX_OPENGL, ScreenDim.Width, ScreenDim.Height, 0, 0)) {
		_ERROR_;
		_SAY("Error setting graphics mode!");
		allegro_message ("Error setting OpenGL graphics mode:\n%s\n"
			"Allegro GL error : %s\n", allegro_error, allegro_gl_error);

		return false;
	}

	_DISPLAY(SCREEN_W);
	_DISPLAY(SCREEN_H);
	_DISPLAY(get_refresh_rate());

	//set MASkinG a bit here
	MAS::Settings::gfxMode = GFX_OPENGL;
	MAS::Settings::fullscreen = !win;
	MAS::Settings::useVideoMemory = false;

	set_window_title("sparklet");

	//if (!is_windowed_mode()) //why did I make it only for fullscreen in the first place?
		set_display_switch_callback(SWITCH_IN, WindowSwitchIn);

	//set color depth: pretty hacky but it seems to work
	set_color_depth(32);

	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glEnable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);

	glViewport(0, 0, SCREEN_W, SCREEN_H);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, SCREEN_W, SCREEN_H, 0, 1, -1);
	glMatrixMode(GL_MODELVIEW);

	return true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
bool Game::InitSound() {
	pop_config_state();
	const int DriverIndex = get_config_int("sound", "driver", 0);
	push_config_state();

	if (SoundDriver[DriverIndex] == DIGI_NONE) {
		_SAY("Not using any sound driver - no sounds will be available.");
		return true;
	}

	const int VoicesMax = detect_digi_driver(SoundDriver[DriverIndex]);
	char buff[255];
	sprintf(buff, "Sound drvier: %i voices available", VoicesMax);
	_SAY(buff);

	if (VoicesMax < 16)
		reserve_voices(VoicesMax, -1);
	else
		reserve_voices(16, -1);

	if (install_sound(SoundDriver[DriverIndex], MIDI_NONE, NULL) == -1) {
		const std::string Msg = "'" + std::string(SOUND_DRIVER_NAME[DriverIndex]) + "' sound driver initialization failed, using autodetect.";
		_SAY(Msg);
		if (is_windowed_mode())
			allegro_message(Msg.c_str());
		if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) == -1) {
			_ERROR_;
			_SAY("Couldn't init sound!");
			return false;
		}
	}

	_SAY("Using '" + string(digi_driver->name) + "' sound driver.");

	return true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
bool Game::PerformOGLTest() {
	const string Path = FixFilenameSlashes(string(DATA_PREFIX) + "bitmaps/gui/logo.tga");

	if (!exists(Path.c_str())) {
		_ERROR_;
		_SAY("File '" + Path + "' does not exist!!!");
		return false;
	}

	BITMAP *TestBmp = load_bitmap(Path.c_str(), NULL);
	_SPARKLET_ASSERT(TestBmp);
	_SPARKLET_ASSERT(bitmap_color_depth(TestBmp) == 32);

	const GLuint RetVal = allegro_gl_make_texture_ex(AGL_TEXTURE_FLIP |
		AGL_TEXTURE_HAS_ALPHA |
		AGL_TEXTURE_RESCALE,
		TestBmp, GL_RGBA8);

	destroy_bitmap(TestBmp);

	if (RetVal == 0) {
		_ERROR_;
		return false;
	}

	_SAY("OpenGL test check passed!");

	return true;
}
#endif //SERVER_ONLY


#ifndef SERVER_ONLY
bool Game::AreMicrosoftDrivers() {
	const UCHAR *Verdor = glGetString(GL_VENDOR);
	if (strstr((const char*)Verdor, "Microsoft"))
		return true;

	return false;
}
#endif //SERVER_ONLY
