/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "sprite.h"
#include "statical_sprite.h"

#include "engine1.h"
#include "sparklet_utils.h"

#include "network_server.h"
#include "sound.h"

#define POWER_STR std::string("power")
#define MAX_FORCE_STR std::string("max_force")

#define ROT_ENGINE_STR std::string("rot_engine")
#define RET_ENGINE_STR std::string("ret_engine")
#define STR_ENGINE_STR std::string("str_engine")
#define BACK_ENGINE_STR std::string("back_engine")

#define FLAME_ROT_UP_LEFT_STR std::string("flame_rot_up_left")
#define FLAME_ROT_UP_RIGHT_STR std::string("flame_rot_up_right")
#define FLAME_ROT_DOWN_LEFT_STR std::string("flame_rot_down_left")
#define FLAME_ROT_DOWN_RIGHT_STR std::string("flame_rot_down_right")

#define FLAME_STR_UP_STR std::string("flame_str_up")
#define FLAME_STR_DOWN_STR std::string("flame_str_down")

#define FLAME_BACK_STR std::string("flame_back")

#define FLAME_SND_SMALL_STR std::string("flame_snd_small")
#define FLAME_SND_BIG_STR std::string("flame_snd_big")


extern NetworkServer *gNetworkServer;


Engine1::Engine1(const TiXmlElement *Element, const Rect<float> &Parent, bool SrvMode) :
	Engine(Element, Parent, SrvMode) {

	const TiXmlElement *EL = XMLParser::GetChild(Element, ROT_ENGINE_STR);
	RotUpLeft.Power = RotUpRight.Power = RotDownLeft.Power =
	RotDownRight.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 100;
	RotUpLeft.MaxForce = RotUpRight.MaxForce = RotDownLeft.MaxForce =
	RotDownRight.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 100;

	EL = XMLParser::GetChild(Element, STR_ENGINE_STR);
	StrUp.Power = StrDown.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 1000;
	StrUp.MaxForce = StrDown.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 1000;

	EL = XMLParser::GetChild(Element, BACK_ENGINE_STR);
	Main.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 1000;
	Main.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 1000;

	EL = XMLParser::GetChild(Element, RET_ENGINE_STR);
	RetDown.Power = RetUp.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 1000;
	RetDown.MaxForce = RetUp.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 1000;

	if (ServerMode)
		goto skip_flame_loading;

	EL = XMLParser::GetChild(Element, FLAME_ROT_UP_LEFT_STR);
	RotUpLeft.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotUpLeft.Flame);

	EL = XMLParser::GetChild(Element, FLAME_ROT_UP_RIGHT_STR);
	RotUpRight.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotUpRight.Flame);

	EL = XMLParser::GetChild(Element, FLAME_ROT_DOWN_LEFT_STR);
	RotDownLeft.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotDownLeft.Flame);

	EL = XMLParser::GetChild(Element, FLAME_ROT_DOWN_RIGHT_STR);
	RotDownRight.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotDownRight.Flame);

	EL = XMLParser::GetChild(Element, FLAME_STR_UP_STR);
	StrUp.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(StrUp.Flame);

	EL = XMLParser::GetChild(Element, FLAME_STR_DOWN_STR);
	StrDown.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(StrDown.Flame);


	EL = XMLParser::GetChild(Element, FLAME_BACK_STR);
	Main.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(Main.Flame);

	RetUp.Flame = RotUpRight.Flame;
	RetDown.Flame = RotDownRight.Flame;


skip_flame_loading:

	if (ServerMode) {
		EL = XMLParser::GetChild(Element, FLAME_SND_SMALL_STR);
		RotUpLeft.FlameSnd = new Sound(EL, Parent);
		RotUpRight.FlameSnd = new Sound(EL, Parent);
		RotDownLeft.FlameSnd = new Sound(EL, Parent);
		RotDownRight.FlameSnd = new Sound(EL, Parent);
		StrDown.FlameSnd = new Sound(EL, Parent);
		StrUp.FlameSnd = new Sound(EL, Parent);

		EL = XMLParser::GetChild(Element, FLAME_SND_BIG_STR);
		Main.FlameSnd = new Sound(EL, Parent);
	}

	EngnList[0] = &Main;
	EngnList[1] = &StrDown;
	EngnList[2] = &StrUp;
	EngnList[3] = &RotUpLeft;
	EngnList[4] = &RotUpRight;
	EngnList[5] = &RotDownLeft;
	EngnList[6] = &RotDownRight;
	EngnList[7] = &RetUp;
	EngnList[8] = &RetDown;

	fill(OldEngnState, OldEngnState + 7, OFF);

	Reset();
}


Engine1::~Engine1() {
	if (ServerMode) {
		for (int x = 0; x < 7; ++x) {
			gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);
			delete EngnList[x]->FlameSnd;
		}
	}
}

float Engine1::GetForward() {
	return Main.CurrentForce - (RetUp.CurrentForce + RetDown.CurrentForce);
}


float Engine1::GetTorque() {
	return (RotUpLeft.CurrentForce - RotUpRight.CurrentForce) -
		(RotDownLeft.CurrentForce - RotDownRight.CurrentForce);
}


float Engine1::GetStrafe() {
	return (StrUp.CurrentForce - StrDown.CurrentForce);
}


bool Engine1::NeedsUpdate(float TimeElapsed) {
	SetEnergy(TimeElapsed);

	if (ServerMode) {
		for (int x = 0; x < 7; ++x) {
			if (OldEngnState[x] != EngnList[x]->State) {
				if (EngnList[x]->State)
					gNetworkServer->AddSound(EngnList[x]->FlameSnd, false);
				else
					gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);

				OldEngnState[x] = EngnList[x]->State;
			}
		}

		return false;
	}

	return Engine::NeedsUpdate(TimeElapsed);
}


void Engine1::SetEnergy(float TimeElapsed) {
	for (int x = 0; x < 7; ++x) {
		if (EngnList[x]->State == ON) {

			EngnList[x]->CurrentForce += EngnList[x]->Power * TimeElapsed;
			if (EngnList[x]->CurrentForce > EngnList[x]->MaxForce)
				EngnList[x]->CurrentForce = EngnList[x]->MaxForce;
		}
		else
			EngnList[x]->CurrentForce = 0.f;
	}

	if (RetUp.State == ON && RetDown.State == ON) {
		RotUpRight.State = ON;
		RotDownRight.State = ON;

		RetUp.CurrentForce += RetUp.Power * TimeElapsed;
		if (RetUp.CurrentForce > RetUp.MaxForce)
			RetUp.CurrentForce = RetUp.MaxForce;

		RetDown.CurrentForce += RetUp.Power * TimeElapsed;
		if (RetDown.CurrentForce > RetDown.MaxForce)
			RetDown.CurrentForce = RetDown.MaxForce;
	}
	else {
		RetUp.CurrentForce = 0.f;
		RetDown.CurrentForce = 0.f;
	}
}


void Engine1::Reset() {
	if (ServerMode) {
		for (int x = 0; x < 7; ++x)
			gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);
	}

	for (int x = 0; x < 9; ++x) {
		EngnList[x]->State = OFF;
		EngnList[x]->CurrentForce = 0;
	}
}


EngineInfo* Engine1::GetEngineInfo() {
	_SPARKLET_ASSERT(ServerMode);

	EngineInfo *EI = new EngineInfo;

	EI->Count = 7;
	EI->Engine = new bool[7];

	for (int x = 0; x < 7; ++x)
		EI->Engine[x] = EngnList[x]->State;

	return EI;
}


#ifndef SERVER_ONLY

void Engine1::SetEngineFlame(const EngineInfo* EI) {
	_SPARKLET_ASSERT(!ServerMode);

	for (int x = 0; x < 7; ++x) {
		if (EI->Engine[x])
			EngnList[x]->Flame->Play();
		else
			EngnList[x]->Flame->Stop();
	}
}
#endif //SERVER_ONLY
