/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SERVER_ONLY

#include "agl_bitmap.h"

#include "bitmap_server.h"

#define R2D 57.2957795056f


extern BitmapServer *gBmpSrv;


AGLBitmap::AGLBitmap(const std::string &Path) :
	tBmp(gBmpSrv->GetBitmap32(Path)),
	Texture(gBmpSrv->GetTexture(tBmp)),
	DoHFlip(false),
	DoVFlip(false),
	Dim(tBmp->w, tBmp->h) {

	//This bitmaps is used only in object initialization. We shouldn't use it in any other way.
	tBmp = NULL;
}


AGLBitmap::AGLBitmap(BITMAP *Bmp) :
	Texture(gBmpSrv->LoadTexture(Bmp)),
	DoHFlip(false),
	DoVFlip(false),
	Dim(Bmp->w, Bmp->h) {

	tBmp = NULL;
}


/*
Renders the bitmap.
SrcRect is the source rectangle to be rendered.
DstRect is the destaination rectangle where the bitmap will be rednered. The bitmap is scaled to
	destination's rectangle dimensions.
Angle is the rotation of destiantion bitmap in radians. It will be rotated around destination's
	center.*/
void AGLBitmap::Render(const Rect<float> &SrcRect, const Rect<float> &DstRect, const float &Angle) const {
	//We handle destination rectangle size equal to zero. It that case just return.
	if (DstRect.Width == 0 || DstRect.Height == 0)
		return;

	/*just to make the code look nicer will define some constants first*/

	const float HalfWidth = Dim.Width / 2.f;
	const float HalfHeight = Dim.Height / 2.f;

	//Calculete scale factor
	const float ScaleX = DstRect.Width / static_cast<float>(SrcRect.Width);
	const float ScaleY = DstRect.Height / static_cast<float>(SrcRect.Height);

	//These are used to set the rotation origin of the destination bitmap
	const float OffX = HalfWidth - (SrcRect.x + SrcRect.Width / 2.f);
	const float OffY = HalfHeight - (SrcRect.y + SrcRect.Height / 2.f);

	//calculate matrix translation values
	float TranslateX = (HalfWidth - SrcRect.x - OffX) * ScaleX + DstRect.x;
	float TranslateY = (HalfHeight - SrcRect.y - OffY) * ScaleY + DstRect.y;

	//transform the matrix the way we need it
	glPushMatrix();

	//to handle different resolutions properly
	TranslateX *= (float)SCREEN_W / UNITS_PER_SCREEN_W;
	TranslateY *= (float)SCREEN_H / UNITS_PER_SCREEN_H;
	glTranslatef(TranslateX, TranslateY, 0);

	glRotatef(Angle * R2D, 0, 0, 1);
	//to handle different resolutions properly
	glScalef((float)SCREEN_W / UNITS_PER_SCREEN_W, (float)SCREEN_H / UNITS_PER_SCREEN_H, 1);

	if (!DoHFlip && !DoVFlip)
		glScalef(ScaleX, ScaleY, 1);
	else if (DoHFlip && !DoVFlip)
		glScalef(ScaleX, -ScaleY, 1);
	else if (!DoHFlip && DoVFlip)
		glScalef(-ScaleX, ScaleY, 1);
	else
		glScalef(-ScaleX, -ScaleY, 1);

	//these are passed to ogl for partial texture displaying
	const float xt = SrcRect.x / static_cast<float>(Dim.Width);
	const float wt = 1.f - (xt + SrcRect.Width / static_cast<float>(Dim.Width));

	const float yt = SrcRect.y / static_cast<float>(Dim.Height);
	const float ht = 1.f - (yt + SrcRect.Height / static_cast<float>(Dim.Height));

	//do the ogl things
	glBindTexture(GL_TEXTURE_2D, Texture);
	glEnable(GL_BLEND);
	glBegin(GL_QUADS);
		glTexCoord2f(xt, ht);
		glVertex2f(Dim.Width * xt - HalfWidth + OffX, -Dim.Height * ht + HalfHeight + OffY);

		glTexCoord2f(1.f - wt, ht);
		glVertex2f(-Dim.Width * wt + HalfWidth + OffX, -Dim.Height * ht + HalfHeight + OffY);

		glTexCoord2f(1.f - wt, 1 - yt);
		glVertex2f(-Dim.Width * wt + HalfWidth + OffX, Dim.Height * yt - HalfHeight + OffY);

		glTexCoord2f(xt, 1 - yt);
		glVertex2f(Dim.Width * xt - HalfWidth + OffX, Dim.Height * yt - HalfHeight + OffY);
	glEnd();
	glDisable(GL_BLEND);

	glPopMatrix();
}


/*
Renders tiled the bitmap.
SrcRect is the source rectangle to be rendered.
DstRect is the destaination rectangle where the bitmap will be rednered. The bitmap is tiled to
	destination's rectangle dimensions.
*/
void AGLBitmap::Tile(const Rect<float> &SrcRect, const Rect<float> &DstRect) const {
	//just to make the code look nicer
	const float HalfWidth = Dim.Width / 2.f;
	const float HalfHeight = Dim.Height / 2.f;
	const float ScaleX = DstRect.Width / static_cast<float>(SrcRect.Width);
	const float ScaleY = DstRect.Height / static_cast<float>(SrcRect.Height);

	float TranslateX = HalfWidth * ScaleX + DstRect.x;
	float TranslateY = HalfHeight * ScaleY + DstRect.y;

	//transform the matrix the way we need it
	glPushMatrix();

	//to handle different resolutions properly
	TranslateX *= (float)SCREEN_W / UNITS_PER_SCREEN_W;
	TranslateY *= (float)SCREEN_H / UNITS_PER_SCREEN_H;
	glTranslatef(TranslateX, TranslateY, 0);

	glScalef(ScaleX, ScaleY, 1.f);
	//to handle different resolutions properly
	glScalef((float)SCREEN_W / UNITS_PER_SCREEN_W, (float)SCREEN_H / UNITS_PER_SCREEN_H, 1);

	//do the ogl things
	glBindTexture(GL_TEXTURE_2D, Texture);
	glBegin(GL_QUADS);
		glTexCoord2f(1.f - ScaleX, 0);
		glVertex2f(HalfWidth, HalfHeight);

		glTexCoord2f(1.f - ScaleX, ScaleY);
		glVertex2f(HalfWidth, - HalfHeight);

		glTexCoord2f(1.f, ScaleY);
		glVertex2f(- HalfWidth, - HalfHeight);

		glTexCoord2f(1.f, 0);
		glVertex2f(- HalfWidth, HalfHeight);
	glEnd();

	glPopMatrix();
}


#endif
