/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef WEAPON_H
#define WEAPON_H

#include "complex_object_extension.h"
#include "weapon_status.h"

#include <string>
#include <vector>

#define WEAPON_STR std::string("weapon")


class WeaponInfo;
class WeaponStatus;
class Sprite;
class Statusbar;
class Sound;
struct IDENTITY;


class Weapon : public ComplexObjectExtension {
public:
	Weapon(const TiXmlElement *Element, const Rect<float> &OwnerGeo, const IDENTITY &OwnerIdentity, Statusbar *S, bool SrvMode);
	~Weapon();

	virtual bool NeedsUpdate(float TimeElapsed);
	virtual void Reset();

	enum FIRE_TYPE {PRIMARY = 0, SECONDARY};

	/*Invoked when fire key is pressed. TF defines the key type (fire type should depend on it).
	Direction is the current direction of the weapon owner. Velocity is the owner's speed.
	Return the loop-back Impulse wich affects the owner.*/
	virtual Vector<float> KeyDown(FIRE_TYPE FT, float Direction, Vector<float> Speed);

	/*Invoked when fire key is released.*/
	virtual Vector<float> KeyUp(FIRE_TYPE FT, float Direction, Vector<float> Speed);

	/*Returns the pointer to the list of enpoints if the weapon is active, NULL otherwise.*/
	virtual const std::vector<Vector<float> >* GetEndpoints() const {
		if (WStatus != HIDDEN)
			return ComplexObjectExtension::GetEndpoints();
		else
			return NULL;
	}

	/*Invoked when weapon is reloaded. Returns true if loading actualy took place, false otherwise.*/
	virtual bool Load(USHORT HowMany);

	/*Invoked when weapon is going to be shown, due to changing the weapon. An animation is
	advisable*/
	void Show();

	/*Invoked when weapon is going to be hidden, due to changing the weapon. An animation is
	advisable*/
	void Hide();

	/*Returns true if the weapon is inactive.*/
	bool IsHidden() const {return WStatus == HIDDEN;}

	/*Invoked when the weapon is gathered by the owner.
	Pass -1 to set to weapon's default initial ammo count.*/
	void MakeGathered(int InitialAmmoCount);

	/*Returns true if the weapon is gathered.*/
	bool IsGathered() const {return BarStatus == WEAPON_GATHERED || BarStatus == WEAPON_SELECTED;}

	/*Invoked when the owner looses the weapon.*/
	void MakeLost() {BarStatus = WEAPON_INACTIVE;}

	/*Returns weapon index.*/
	USHORT GetIndex() const {return Index;}

	/*Returns true if there is no more ammo.*/
	bool IsEmpty() const {return Ammo == 0;}

	/*Creates an use-once-and-destroy ammo_pack object*/
	void LeavePacket(const Point<float>& Loc) const;

	/*Returns info struct ready to be sent over the network by server. The calling process is
	responsable of deleting the return value.*/
	WeaponInfo* GetWeaponInfo() const;

	/*Function used by client to set weapon state received from server.*/
	void SetWeaponInfo(const WeaponInfo* WI);


protected:
	USHORT Ammo;
	USHORT AmmoLimit;

	const Rect<float> &OwnerGeo;
	const IDENTITY &OwnerIdentity;

	XMLParser Parser;
	const Point<float> Offset;
	Statusbar *Stb;

	struct FM {
		FM() : TimeRemaning(0) {}
		float TimeRemaning;
		float Period;
		float PeriodOffset;
		float Fuzzynes;
		float Impulse;
		const TiXmlElement *Att;
		const Sound *Sample;
	};

	struct Arm{
		Sprite *Firearm;
		float FirePointRadius;
		float FirePointAngle;
	};

	std::vector<Arm> ArmList;
	std::vector<FM> FireMode;

	Vector<float> Fire(const Arm &Gun, const FM &Mode, float Direction, Vector<float> Velocity);

private:
	enum STATUS {SHOWN, HIDDEN, HIDING, SHOWING};
	STATUS WStatus;
	WEAPON_STATUS BarStatus;
	USHORT Index;

	WeaponStatus *WS;

	void DecraseRemaningTime(float TimeElapsed);
	void StopWeaponAnim();

	USHORT InitialAmmo;

	const TiXmlElement *WeaponPacketElement;
};

#endif
