/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SERVER_ONLY


#undef RootWindow

#include "server_selector.h"
#include "network_messages.h"
#include <string>


ServerSelector::ServerSelector() :
	TabIndexActive(0) {

	TP.Shape(0, 0, 470, 210);
	TP.Attach(&LSD, "local servers");
	TP.Attach(&ISD, "internet servers");
	TP.Attach(&SSD, "server IP");

	MAS::Dialog::Add(TP);

}


void ServersDialogGeneric::Process() {
	ServerScanner.process_nonblocking();
}


void ServerSelector::HandleEvent(Widget &obj, int msg, int arg1, int arg2) {
	Dialog::HandleEvent(obj, msg, arg1, arg2);

	switch (msg) {
		case MSG_SCROLL: //why the hell did he use MSG_SCROLL?
			if (obj == TP)
				TabIndexActive = arg1;
		break;
	}
}


std::string ServerSelector::GetSelectedAddress() {
	switch (TabIndexActive) {
		case 0: //"LAN servers" tab
			return LSD.GetIP() + ":" + LSD.GetPort();
		break;
		case 1: //"internet servers" tab
			return ISD.GetIP() + ":" + ISD.GetPort();
		break;
		case 2: //"server IP" tab
			return SSD.GetIP() + ":" + SSD.GetPort();
		break;
		default:
			_ERROR_;
	}

	return "";
}


ServersDialogGeneric::ServersDialogGeneric() {
	bRefresh.Setup(5, 5, 60, 20, 0, 0, "rescan");
	lbServerList.Shape(5, 35, 455, 150);

	lbServerList.InsertColumn("name", -1, 2, 90);
	lbServerList.InsertColumn("ping", -1, 2, 40);
	lbServerList.InsertColumn("players", -1, 2, 70);
	lbServerList.InsertColumn("map name", -1, 2, 115);
	lbServerList.InsertColumn("game type", -1, 2, 80);
	lbServerList.InsertColumn("net ratio", -1, 2, 80);
	lbServerList.InsertColumn("password", -1, 2, 75);
	lbServerList.InsertColumn("net ver.", -1, 2, 70);
	lbServerList.InsertColumn("hostname", -1, 2, 120);
	lbServerList.InsertColumn("port", -1, 2, 50);

	lbServerList.EnableAutosort(true);

	MAS::Dialog::Add(bRefresh);
	MAS::Dialog::Add(lbServerList);

	//to be changed for ziglite
	ServerScanner.init(this, gLC);
}


void ServersDialogGeneric::serverinfo_timeout(address_c &addr, int time_ms) {
	bRefresh.Enable();
}


void ServersDialogGeneric::serverinfo_local_timeout(int time_ms) {
	bRefresh.Enable();
}


void ServersDialogGeneric::HandleEvent(Widget &obj, int msg, int arg1, int arg2) {
	Dialog::HandleEvent(obj, msg, arg1, arg2);

	switch (msg) {
		case MSG_ACTIVATE:
			if (obj == bRefresh) {
				bRefresh.Disable();
				lbServerList.DeleteAllItems();
				RequestServerInfo();
			}
		break;
	}
}


void ServersDialogGeneric::InsertServer(const ServerInfo *SI, const std::string &IP, const int Ping) {
	lbServerList.InsertItem(new MAS::ListBoxEx::Item);

	const int LastRow = lbServerList.GetItemCount() - 1;

	lbServerList.SetItem(LastRow, 0, SI->Name.c_str());
	lbServerList.SetItem(LastRow, 1, itoa(Ping).c_str());
	lbServerList.SetItem(LastRow, 2, (itoa(SI->PlayersConnected) + "/" + itoa(SI->PlayerLimit)).c_str());
	lbServerList.SetItem(LastRow, 3, SI->MapName.c_str());

	std::string GT;
	if (SI->GameType == GT_DEATHMATCH)
		GT = "DM";
	else if (SI->GameType == GT_TEAM_DEATHMATCH)
		GT = "TDM";
	lbServerList.SetItem(LastRow, 4, GT.c_str());

	lbServerList.SetItem(LastRow, 5, itoa(SI->NetRatio).c_str());
	lbServerList.SetItem(LastRow, 6, SI->RequirePassword ? "yes" : "no");
	lbServerList.SetItem(LastRow, 7, itoa(SI->ProtocolVersion).c_str());
	lbServerList.SetItem(LastRow, 8, IP.c_str());
	lbServerList.SetItem(LastRow, 9, itoa(SI->Port).c_str());
}


std::string ServersDialogGeneric::GetIP() {
	const int Index = lbServerList.GetSelectedIndex();
	if (Index != -1)
		return lbServerList.GetItem(Index, 8);

	return "";
}


std::string ServersDialogGeneric::GetPort() {
	const int Index = lbServerList.GetSelectedIndex();
	if (Index != -1)
		return lbServerList.GetItem(Index, 9);

	return "";
}


LANServersDialog::LANServersDialog() {
	RequestServerInfo();
}


void LANServersDialog::RequestServerInfo() {
	ServerScanner.get_local_server_info();
}


void LANServersDialog::serverinfo_local_result(address_c &addr, buffer_c &info, int time_ms) {
	try {
		auto_ptr<const ServerInfo> SI(static_cast<const ServerInfo*>(info.getObject()));
		std::string IP = addr.get_address();
		IP = IP.substr(0, IP.rfind(':')); //strip port number
		InsertServer(SI.get(), IP, time_ms);
	}
	catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Illegal server info structure received!");
		return;
	}
}


bool InternetServersDialog::RetrieveServerList(const std::string &web_address, const std::string &web_file) {
	NLaddress addr;
	NLsocket sock = NL_INVALID;
	const std::string request = "GET " + web_file + " HTTP/1.0\r\nHost:" + web_address + "\r\n\r\n";
	NLbyte buffer[4096];
	NLint count;
	FILE *f = NULL;
	bool Failed = true;
	int lf_found = 0;
	int cr_found = 0;

	const int TIMEOUT = 2000;

	nlEnable(NL_BLOCKING_IO);
	nlGetAddrFromName(web_address.c_str(), &addr);
	nlSetAddrPort(&addr, 80);

	sock = nlOpen(0, NL_TCP);
	if(sock == NL_INVALID) {
		_ERROR_;
		goto cleanup;
	}

	if (nlConnect(sock, &addr) == NL_FALSE) {
		_ERROR_;
		goto cleanup;
	}

	if (nlWrite(sock, (NLvoid *)request.c_str(), request.length()) == NL_INVALID) {
		_ERROR_;
		goto cleanup;
	}

	f = fopen(SERVER_LIST_FILE_NAME, "w");
	if (!f) {
		_ERROR_;
		_SAY("Cound not open file for writing.");
		goto cleanup;
	}

	while (true) {
		if (nlPollSocket(sock, NL_READ_STATUS, TIMEOUT))
			count = nlRead(sock, (NLvoid *)buffer, (NLint)sizeof(buffer) - 1);
		else {
			_SAY("Timed out.");
			break;
		}

		if (count < 0) {
			NLint err = nlGetError();
			/* is the connection closed? */
			if(err == NL_MESSAGE_END) {
				Failed = false;
				break;
			}
			else {
				_ERROR_;
				break;
			}
		}
		else if (count > 0) {
			for (int i = 0; i < count; i++) {
				if (cr_found == 2 && lf_found == 2) {
					if (memchr(&buffer[i], '<', count - i)) {
						/* We probably got some HTML saying an error occured. */
						_ERROR_;
						goto cleanup;
					}
					/* The rest of the buffer is the file contents. */
					write(fileno(f), &buffer[i], count - i);
					break;
				}

				if (buffer[i] == '\r')
					cr_found++;
				else if (buffer[i] == '\n')
					lf_found++;
				else
					cr_found = lf_found = 0;
			}
		}
		else {
			Failed = false;
			break;
		}
	}

cleanup:
	if (sock != NL_INVALID)
		nlClose(sock);
	nlDisable(NL_BLOCKING_IO);
	if (f)
		fclose(f);

	if (Failed)
		remove(SERVER_LIST_FILE_NAME);

	return !Failed;
}


void InternetServersDialog::RequestServerInfo() {
	char line[128];

	if (!exists(SERVER_LIST_FILE_NAME)) {
		if (!RetrieveServerList("www.riteh.hr", "/~mmimica/srvlst.txt")) {
			if (!RetrieveServerList("sparklet.sourceforge.net", "/srvlst.txt")) {
				_ERROR_;
				bRefresh.Enable();
				return;
			}
		}
	}

	FILE *f = fopen(SERVER_LIST_FILE_NAME, "r");
	if (!f) {
		_ERROR_;
		return;
	}

	while (fgets(line, 128, f)) {
		address_c adr(line);
		buffer_c custom;
#ifdef ENABLE_SERVER_LOG_OBTAINING
		_CHECKPOINT_;
		custom.putByte(0);
#endif
		ServerScanner.get_server_info(adr, custom);
	}

	fclose(f);
}


void InternetServersDialog::serverinfo_result(address_c &addr, buffer_c &info, int time_ms) {
	try {
		auto_ptr<const ServerInfo> SI(static_cast<const ServerInfo*>(info.getObject()));
		std::string IP = addr.get_address();
		IP = IP.substr(0, IP.rfind(':')); //strip port number
		InsertServer(SI.get(), IP, time_ms);

#ifdef ENABLE_SERVER_LOG_OBTAINING
		if (info.size_left()) {
			buffer_c LogFile;
			info.unzip(LogFile);
			LogFile.putByte('\0');

			FILE *f = fopen((SI->Name + ".log").c_str(), "w");
			if (f) {
				fputs(LogFile.data(), f);
				fclose(f);
			}
		}
#endif
	}
	catch (zig_int_exceptions_t ex) {
		_ERROR_;
		_SAY("Illegal server info structure received!");
		return;
	}

	bRefresh.Enable();
}


SingleServerDialog::SingleServerDialog() {
	lIP.Setup(10, 15, 60, 20, 0, 0, "IP or hostname");
	lPort.Setup(360, 15, 10, 20, 0, 0, "port");
	eIP.Setup(7, 30, 340, 20, 0, 0, get_config_string("single_server", "ip", "0.0.0.0"));
	ePort.Setup(350, 30, 50, 20, 0, 0, get_config_string("single_server", "port", "2020"), 5);

	Add(lIP);
	Add(lPort);
	Add(eIP);
	Add(ePort);
}


SingleServerDialog::~SingleServerDialog() {
	set_config_string("single_server", "ip", eIP.GetText());
	set_config_string("single_server", "port", ePort.GetText());
}


#endif //SERVER_ONLY
