/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "sound.h"
#include "projectile.h"

#include "particle.h"
#include "invisible_particle.h"
#include "sparklet_utils.h"
#include <algorithm>

#define EXPLOSION_STR std::string("explosion")
#define COLOR_STR std::string("color")
#define PARTICLE_COUNT_STR std::string("particle_count")

#define LAYOUT_STR std::string("layout")
#define PARTICLES_STR std::string("particles")
#define FLAME_STR std::string("flame")

#define IMPULSE_STR std::string("impulse")
#define RND_IMPULSE_STR std::string("rnd_impulse")

#define INVISIBLE_PARTICLE_STR std::string("invisible_particle")

#define EXPLOSION_SOUND_STR std::string("explosion_sound")


using namespace std;

extern float gTimeElapsed;


Projectile::Projectile(const TiXmlElement* Att, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	RoundObject(Att, Scale),
	NonRotatedObject(Att, Scale) {

	const TiXmlElement *Root = Parser->GetRoot();

	ExplosionSound = new Sound(XMLParser::GetChild(Root, EXPLOSION_SOUND_STR), Geo);

	const TiXmlElement *Explosion = XMLParser::GetChild(Root, EXPLOSION_STR);

	ParticleCount = XMLParser::GetValueL(Explosion, PARTICLE_COUNT_STR, 0);
	_SPARKLET_ASSERT(ParticleCount > 0);
	Layout = XMLParser::GetValueS(Explosion, LAYOUT_STR, "circle");
	ExplosionImpulse = XMLParser::GetValueL(Explosion, IMPULSE_STR, 10);
	RndExplosionImpulse = XMLParser::GetValueL(Explosion, RND_IMPULSE_STR, 5);
	ExplosionColor = XMLParser::GetValueL(Explosion, COLOR_STR, 0);

	const TiXmlElementList* List1 = XMLParser::GetAllChildren(Explosion, EXPL_PARTICLE_STR);
	const TiXmlElementList* List2 = XMLParser::GetAllChildren(Explosion, INVISIBLE_PARTICLE_STR);
	DOMParticleList.insert(DOMParticleList.end(), List1->begin(), List1->end());
	DOMParticleList.insert(DOMParticleList.end(), List2->begin(), List2->end());
	delete List1;
	delete List2;

	if (DOMParticleList.size() == 0) {
		_ERROR_;
		_SAY("There should be at least one particle tag!");
	}

	const TiXmlElementList* Child = XMLParser::GetAllChildren(Explosion, FLAME_STR);
	if (Child->size() == 1)
		Flame = Child->at(0);
	else
		Flame = NULL;
	delete Child;
}


Projectile::~Projectile() {
	delete ExplosionSound;
}


void Projectile::OnUpdate() {
	Geo += Velocity * gTimeElapsed;
	PhysicalObject::NextFrame(gTimeElapsed);

	PhysicalObject::OnUpdate();
}


Particle* Projectile::Explode() {
	vector<Particle*> ParticleList;

	gNetworkServer->AddSound(ExplosionSound, true);

	ParticleList.reserve(ParticleCount);
	for (USHORT x = 0; x < ParticleCount; ++x) {
		const USHORT rnd = (int)((float)DOMParticleList.size() * rand() / (RAND_MAX + 1.f));
		const TiXmlElement *Par = DOMParticleList.at(rnd);
		if (XMLParser::GetName(Par) == INVISIBLE_PARTICLE_STR)
			ParticleList.push_back(new InvisibleParticle(Par, ExplosionColor));
		else
			ParticleList.push_back(new Particle(Par, ExplosionColor));
	}

	if (Layout == "circle") {
		for (USHORT x = 0; x < ParticleList.size(); ++x) {
			const float AngleOffset = (rand() % 100) / 100.f;
			const float Angle = BoundAngle(AngleOffset + (2 * M_PI) * (x / (float)ParticleList.size()));
			const float Force = ExplosionImpulse + rand() % RndExplosionImpulse;
			ParticleList[x]->SetImpulse(polar(Force, Angle));
			ParticleList[x]->SetLocC(GetCenter(Geo));
			ParticleList[x]->Identity = Identity;
		}
	}
	else if (Layout == "star") {
		for (USHORT x = 0; x < ParticleList.size(); ++x) {
			float Angle = 0;
			if (x % 4 == 0)
				Angle = (rand() % (int)(2.f * M_PI / 8.f * 1000.f)) / 1000.f;
			else if (x % 4 == 1)
				Angle = M_PI / 2.f + (rand() % (int)(2.f * M_PI / 8.f * 1000.f)) / 1000.f;
			else if (x % 4 == 2)
				Angle = M_PI + (rand() % (int)(2.f * M_PI / 8.f * 1000.f)) / 1000.f;
			else if (x % 4 == 3)
				Angle = 3.f * M_PI / 2.f + (rand() % (int)(2.f * M_PI / 8.f * 1000.f)) / 1000.f;

			const float Force = ExplosionImpulse + rand() % RndExplosionImpulse;
			ParticleList[x]->SetImpulse(polar(Force, Angle));
			ParticleList[x]->SetLocC(GetCenter(Geo));
			ParticleList[x]->Identity = Identity;
		}
	}

	if (Flame) {
		Particle *NewOne = new Particle(Flame, 0);
		NewOne->SetLocC(GetCenter(Geo));
		NewOne->Identity = Identity;

		return NewOne;
	}

	return NULL;
}
