/***************************************************************************
                          physical_object.cpp  -  description
                             -------------------
    begin                : Sun Oct 26 2003
    copyright            : (C) 2003 by Milan Mimica
    email                : milan.mimica@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "bitmap_server.h"
#include "physical_object.h"
#include "game_logic.h"
#include "network_server.h"
#include "object_extension.h"
#include "sparklet_utils.h"
#include "pmask.h"

#define ELASTIC_STR std::string("elastic")
#define INNER_ENERGY_STR std::string("inner_energy")


using namespace std;

extern NetworkServer *gNetworkServer;
extern GameLogic *gGameLogic;
extern BitmapServer *gBmpSrv;
extern GameOptions gGameOpt;


PhysicalObject::PhysicalObject(const TiXmlElement *Att, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	CollidingEnergy(0),
	CollidingImpulse(0, 0),
	AllowInaccurateCollisionDetection(false) {

	const TiXmlElement *Root = Parser->GetRoot();

	Elastic = XMLParser::GetValueF(Root, ELASTIC_STR, 1.f);
	InnerEnergy = XMLParser::GetValueF(Root, INNER_ENERGY_STR, 0.f);
	IsFixedShape = false;

	InnerEnergy = XMLParser::GetValueF(Att, INNER_ENERGY_STR, InnerEnergy);
	Elastic = XMLParser::GetValueF(Att, ELASTIC_STR, Elastic);

	Shape = create_bitmap_ex(16, Geo.Width, Geo.Height);
	tmp = gBmpSrv->GetBitmap16(Geo);

	gNetworkServer->Add(this, Object::ObjectUID);

	PMaskCache = NULL;

	Identity.Team = TEAM_NO_TEAM;
	Identity.OwnerObjUID = 0;

	LastObjectCollided = NULL;
	CollidingNow = false;
	CollidedLastTime = false;
	PreventMultipleCollide = false;
}


PhysicalObject::~PhysicalObject() {
	if (PMaskCache)
		destroy_pmask(PMaskCache);

	destroy_bitmap(Shape);
	destroy_bitmap(tmp);
}


bool PhysicalObject::NextFrame(const float TimeElapsed) {
	bool Update = false;

	for (USHORT x = 0; x < Extension.size(); ++x) {
		if (Extension[x]->NeedsUpdate(TimeElapsed)) {
			Update = true;
			ShapeChanged = true;
			Extension[x]->Update();
		}
	}

	return Update;
}


bool PhysicalObject::NextFrameSingleExtension(const float TimeElapsed) {
	if (Extension[0]->NeedsUpdate(TimeElapsed)) {
		ShapeChanged = true;
		Extension[0]->Update();

		return true;
	}

	return false;
}


bool PhysicalObject::OnCollide(const PhysicalObject *Obj) {
	//Prevent calculateing collision reaction twice with the same object.
	/*History:
		For many reasons it is possible (and it often happens) that the collision
		betwen two objects is determined a bit too late - when the objects are quite
		much within each other. It is possible to prevent such situaltions by doing
		some complex vector calculations to alter object's position but here is the
		simplest way. Since we don't do elastic bumping (to much math) we just prevent
		multiple collision reaction calculation, thus acting like the objects just
		touch each other.
	*/
	if (PreventMultipleCollide && LastObjectCollided == Obj)
		return false;

	LastObjectCollided = Obj;

	const Vector<float> SpeedDiff = Obj->Velocity - Velocity;
	const Vector<float> BumpAngle = polar(1.f, Obj->GetSurfaceNormal(GetCenter(Geo)));
	const float ImpactMagnitude = abs(dot_product(BumpAngle, SpeedDiff))
								* (Elastic + Obj->Elastic)
								* (Mass * Obj->Mass) / (Mass + Obj->Mass);

	const Vector<float> ResultImpulse(ImpactMagnitude * BumpAngle);

	CollidingImpulse += ResultImpulse;
	Damage(Obj);

	return true;
}


void PhysicalObject::Damage(const PhysicalObject *Obj) {
	CollidingEnergy += Obj->InnerEnergy;
}


void PhysicalObject::DamageFrendly(const PhysicalObject *Obj, const float &FrendlyFire) {
	CollidingEnergy += Obj->InnerEnergy * FrendlyFire / 100.f;
}


void PhysicalObject::OnUpdate() {
	CollidingImpulse = Vector<float>(0, 0);
	CollidingEnergy = 0;

	CollidedLastTime = CollidingNow;
	CollidingNow = false;

	if (!CollidedLastTime)
		LastObjectCollided = NULL;

	//PhysicalObject should really contain at least one extension to make the pmask cacheing
	//algorithm work. Otherwise RenderShape() will always return 'false' and GetPmask() 'NULL'
	//and that's bad.
	_SPARKLET_ASSERT(Extension.size());

	MobileObject::OnUpdate();
}


PMASK const* PhysicalObject::GetPMask() {
	if (RenderShape()) {
		if (!PMaskCache)
			PMaskCache = create_pmask(Shape->w, Shape->h);

		pmask_load_func(PMaskCache, 0, 0, Shape, bitmap_mask_color(Shape), (int (*)(void*,int,int))getpixel);
	}

	return PMaskCache;
}


bool PhysicalObject::AreColliding(PhysicalObject *Obj) {
	if (!gGameLogic->CanCollide(GetTypeID(), Obj->GetTypeID()))
		return false;

	if (!CheckRectCollision(Geo.x, Geo.y, Geo.Width, Geo.Height, Obj->Geo.x, Obj->Geo.y, Obj->Geo.Width, Obj->Geo.Height))
		return false;

	if (AllowInaccurateCollisionDetection && Obj->AllowInaccurateCollisionDetection)
		return true;

	const PMASK *const Mask1 = GetPMask();
	const PMASK *const Mask2 = Obj->GetPMask();

	bool Colliding = check_pmask_collision(Mask1, Mask2, (int) Geo.x, (int) Geo.y, (int) Obj->Geo.x, (int) Obj->Geo.y);
	CollidingNow |= Colliding;
	return Colliding;
}


std::vector<float> PhysicalObject::GetExplosionFragmentsAngles(const USHORT FragmentsCount) {
	_SPARKLET_ASSERT(FragmentsCount > 0);

	std::vector<float> Angle;
	Angle.reserve(FragmentsCount);

	if (FragmentsCount == 1) {
		Angle.push_back(M_PI);
		return Angle;
	}

	const float Offset = (2*M_PI - M_PI/1.5f) / (FragmentsCount-1);
	for (float x = 0; x < FragmentsCount; ++x) {
		Angle.push_back(x * Offset + M_PI/3);
	}

	return Angle;
}


void PhysicalObject::Unlink() {
	gNetworkServer->RemovePhysicalObj(Object::ObjectUID);
	MobileObject::Unlink();
}
