/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "sprite.h"
#include "particle.h"
#include "game_logic.h"
#include "network_server.h"

#define DECREMENT_PER_SECOND_STR std::string("decrement_per_second")
#define DURATION_STR std::string("duration")


extern float gTimeElapsed;
extern GameLogic *gGameLogic;
extern NetworkServer *gNetworkServer;


const ULONG Particle::ID(GEN_ID('e', 'x', 'p', 'a'));


Particle::Particle(const TiXmlElement *Att, int lColor, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	RoundObject(Att, Scale),
	NonRotatedObject(Att, Scale),
	Color(lColor),
	TookCamera(false) {

	const TiXmlElement *Root = Parser->GetRoot();

	BaseSprite = GetBaseSprite(Root);
	BaseSprite->PlayOnce();

	EnergyDecrementPerSecond = XMLParser::GetValueF(Root, DECREMENT_PER_SECOND_STR, 0.f);
	EnergyDecrementPerSecond = XMLParser::GetValueF(Att, DECREMENT_PER_SECOND_STR, EnergyDecrementPerSecond);

	Duration = XMLParser::GetValueF(Root, DURATION_STR, 1.f);
	Duration = XMLParser::GetValueF(Att, DURATION_STR, Duration);

	BaseSprite->SetFPS(BaseSprite->GetFrameCount() / Duration);

	IsFixedShape = true;
	AllowInaccurateCollisionDetection = true;
}


void Particle::OnUpdate() {
	if (InnerEnergy > 0)
		InnerEnergy -= EnergyDecrementPerSecond * gTimeElapsed;
	else
		InnerEnergy = 0;

	Velocity += (Impulse + CollidingImpulse) / Mass;
	LimitSpeed();
	Geo += Velocity * gTimeElapsed;

	PhysicalObject::NextFrameSingleExtension(gTimeElapsed);

	if (BaseSprite->IsOver())
		Unlink();

	PhysicalObject::OnUpdate();
}


void Particle::GetCompleteData(MessageCoder &MC) const {
	FullObjInfo *FOI = new FullObjInfo();
	*FOI = FOIPartial;
	if (Color)
		FOI->XMLFilePathCode = Color;
	FOI->Dim = Geo;
	MC.Put(FOI);

	ParticleInfo *PI = new ParticleInfo();
	PI->LifetimeLeft = (BaseSprite->GetFrameCount() - BaseSprite->GetCurrentFrame()) * (Duration / BaseSprite->GetFrameCount());
	PI->VI.Velocity = Velocity;
	PI->VI.Loc = Geo;
	MC.Put(PI);
}


void Particle::TakeCamera(int ClientID) {
	_SPARKLET_ASSERT(!TookCamera);
	CameraClientID = ClientID;
	gNetworkServer->SetCamera(CameraClientID, ObjectUID);
	TookCamera = true;
}


void Particle::Unlink() {
	if (TookCamera)
		gNetworkServer->ReleaseCamera(CameraClientID, ObjectUID);
	PhysicalObject::Unlink();
}
