/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef NETWORK_MESSAGES_H
#define NETWORK_MESSAGES_H

#include "game_input.h"
#include "structures.h"

#include <zig/typereg.h>
#include <zig/buffer.h>

#undef writeString
#undef readString

#include <vector>
#include <set>


//Define the sparklet network protocol version number. This is used to prevent client connecting to
//server using different network protocol (older or newer) and crashing later. Different game data
//(new maps, weapons etc.) also issues new version number.
#define NSB_NET_PROTOCOL_VER 12


//Define a reason for client disconnection
// - a game has just ended and clients should reconnect again soon
#define NSB_CLI_DISCONNECTED_GAME_ENDED 5
// - other players voted to kick this client
#define NSB_CLI_DISCONNECTED_KICK_VOTED 6


//Sound PlayMode modes.
#define PM_PLAY		0x01
#define PM_LOOP		0x02
#define PM_FORWARD	0x04
#define PM_BACKWARD	0x08
#define PM_BIDIR	0x10



class KeySet : public std::set<INPUT_COMMAND>, public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(KeySet);

	KeySet() {};
	KeySet(const std::set<INPUT_COMMAND>& k) {std::set<INPUT_COMMAND>::insert(k.begin(), k.end());}

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class OUIDInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(OUIDInfo);

	OUIDInfo() {}
	OUIDInfo(ULONG n) : OUID(n) {}

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const OUIDInfo &n) const {return OUID != n.OUID;}
	bool operator==(const OUIDInfo &n) const {return OUID == n.OUID;}
	bool operator<(const OUIDInfo &n) const {return OUID < n.OUID;}
	bool operator>(const OUIDInfo &n) const {return OUID > n.OUID;}
	OUIDInfo operator-(const OUIDInfo &n) const {return OUIDInfo(OUID - n.OUID);}
	OUIDInfo operator+(const OUIDInfo &n) const {return OUIDInfo(OUID + n.OUID);}
	OUIDInfo& operator++();

private:
	ULONG OUID;
};


class ClientInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ClientInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	std::string ShipPath;
	std::string Name;
	TEAM Team;
};


class VelocityInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(VelocityInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	Vector<float> Velocity;
	Point<float> Loc;
};


class FullVelocityInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(FullVelocityInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	Vector<float> Velocity;
	Vector<float> Impulse;
	Point<float> Loc;
};


class RotationInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(RotationInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	float Direction;
	float Rotation;
};


class FullRotationInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(FullRotationInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	float Direction;
	float Rotation;
	float Torque;
};


class ParticleInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ParticleInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	VelocityInfo VI;
	float LifetimeLeft;
};


class FullObjInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(FullObjInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	USHORT XMLFilePathCode;
	USHORT Surface;
	Size Dim;
	float Scale;
};


class EngineInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(EngineInfo);

	virtual ~EngineInfo() {delete [] Engine;}
	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const EngineInfo &EI);

	USHORT Count;
	bool *Engine;
};


class WeaponInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(WeaponInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const WeaponInfo &WI);

	USHORT Index;
	USHORT Ammo;
	int BarStatus;
	int WStatus;
};


class ShipState : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ShipState);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const ShipState &SS);

	UCHAR Energy;
	UCHAR ChassisFrame;
};


class ShipInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ShipInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const ShipInfo &SI);

	TEAM Team;
	std::string Name;
};


class AspectInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(AspectInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	bool operator!=(const AspectInfo &AI);

	bool Visible;
};


class CodedPaths : public std::map<USHORT, std::string>, public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(CodedPaths);

	CodedPaths() {};
	CodedPaths(std::map<USHORT, std::string> m) {insert(m.begin(), m.end());}

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class NewPlayerInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(NewPlayerInfo);

	std::string Name;
	USHORT UID;
	TEAM Team;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class PlayerGoneInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(PlayerGoneInfo);

	USHORT UID;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class ScoreInfo : public std::vector<char>, public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ScoreInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class PingInfo : public std::vector<USHORT>, public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(PingInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	void push_back(USHORT p);
};


class WinnerInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(WinnerInfo);

	USHORT ClientID;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class StateInfoGeneric1 : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(StateInfoGeneric1);

	UCHAR State;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class StateInfoGeneric2 : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(StateInfoGeneric2);

	UCHAR State;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class PlayerMessage : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(PlayerMessage);

	std::string Text;
	std::string To;
	USHORT From;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class GameMessage : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(GameMessage);

	USHORT WhoKilled;
	USHORT WhoWasKilled;
	bool KilledWithWeapon;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


enum GAME_TYPE {GT_DEATHMATCH, GT_TEAM_DEATHMATCH};


class ServerInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ServerInfo);

	std::string Name;
	USHORT ProtocolVersion;
	USHORT PlayersConnected;
	USHORT PlayerLimit;
	USHORT Port;
	UCHAR NetRatio;
	GAME_TYPE GameType;
	std::string MapName;
	bool RequirePassword;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};


class GameOptions : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(GameOptions);

	GameOptions() : CurrentMap(0) {}

	GAME_TYPE GT;
	UCHAR PointsMax;
	float MediumViscosity;

	//transistent
	float FrendlyFire;
	std::string NextMap;
	USHORT CurrentMap;
	std::vector<std::string> MapFileNameList;

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);
};



class NewSoundInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(NewSoundInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	OUIDInfo SoundUID;
	USHORT PathCode;
	UCHAR Priority;
	UCHAR PlayMode;
};


class NewSoundInfoUntunable : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(NewSoundInfoUntunable);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	OUIDInfo SoundUID;
	USHORT PathCode;
	UCHAR Volume;
	UCHAR Pan;
	UCHAR Priority;
	UCHAR PlayMode;
};


class AdjustSoundInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(AdjustSoundInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	OUIDInfo SoundUID;
	UCHAR Volume;
	UCHAR Pan;
};


class NewKickPollInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(NewKickPollInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	std::string KickWho;
	std::string Reason;
};


enum SERVER_REQUEST {SR_MAP_LIST = 0};

class ServerRequestInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ServerRequestInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	SERVER_REQUEST RequestCode;
};


class ChangeMapCommandInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ChangeMapCommandInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	std::string MapName;
};


class ServerMessageInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ServerMessageInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	std::string Text;
};


class ChangeTeamCommandInfo : public serializable_c {
public:
	ZIG_SERIALIZABLE_CLASS_H(ChangeTeamCommandInfo);

	virtual void write(buffer_c &out) const;
	virtual void read(buffer_c &in);

	TEAM Team;
	int Who;
};

#endif
