/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "sprite.h"
#include "statical_sprite.h"

#include "engine2.h"
#include "sparklet_utils.h"

#include "network_server.h"
#include "sound.h"

#define POWER_STR std::string("power")
#define MAX_FORCE_STR std::string("max_force")

#define ROT_ENGINE_STR std::string("rot_engine")
#define BACK_ENGINE_STR std::string("back_engine")
#define RET_ENGINE_STR std::string("ret_engine")

#define FLAME_UP_LEFT_STR std::string("flame_up_left")
#define FLAME_UP_RIGHT_STR std::string("flame_up_right")
#define FLAME_DOWN_LEFT_STR std::string("flame_down_left")
#define FLAME_DOWN_RIGHT_STR std::string("flame_down_right")

#define FLAME_SND_STR std::string("flame_snd")


extern NetworkServer *gNetworkServer;


Engine2::Engine2(const TiXmlElement *Element, const Rect<float> &Parent, bool SrvMode) :
	Engine(Element, Parent, SrvMode) {

	const TiXmlElement *EL  = XMLParser::GetChild(Element, ROT_ENGINE_STR);
	RotUpLeft.Power = RotUpRight.Power = RotDownLeft.Power =
	RotDownRight.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 100;
	RotUpLeft.MaxForce = RotUpRight.MaxForce = RotDownLeft.MaxForce =
	RotDownRight.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 100;

	EL = XMLParser::GetChild(Element, BACK_ENGINE_STR);
	FwdUp.Power = FwdDown.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 1000;
	FwdUp.MaxForce = FwdDown.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 1000;

	EL = XMLParser::GetChild(Element, RET_ENGINE_STR);
	RetDown.Power = RetUp.Power = XMLParser::GetValueL(EL, POWER_STR, 0) * 1000;
	RetDown.MaxForce = RetUp.MaxForce = XMLParser::GetValueL(EL, MAX_FORCE_STR, 0) * 1000;

	if (ServerMode)
		goto skip_flame_loading;

	EL = XMLParser::GetChild(Element, FLAME_UP_LEFT_STR);
	RotUpLeft.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotUpLeft.Flame);

	EL = XMLParser::GetChild(Element, FLAME_UP_RIGHT_STR);
	RotUpRight.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotUpRight.Flame);

	EL = XMLParser::GetChild(Element, FLAME_DOWN_LEFT_STR);
	RotDownLeft.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotDownLeft.Flame);

	EL = XMLParser::GetChild(Element, FLAME_DOWN_RIGHT_STR);
	RotDownRight.Flame = new Sprite(EL, Parent, ServerMode);
	SpriteLayer.push_back(RotDownRight.Flame);

	FwdUp.Flame = RotUpLeft.Flame;
	FwdDown.Flame = RotDownLeft.Flame;
	RetUp.Flame = RotUpRight.Flame;
	RetDown.Flame = RotDownRight.Flame;


skip_flame_loading:

	if (ServerMode) {
		EL = XMLParser::GetChild(Element, FLAME_SND_STR);
		RotUpLeft.FlameSnd = new Sound(EL, Parent);
		RotUpRight.FlameSnd = new Sound(EL, Parent);
		RotDownLeft.FlameSnd = new Sound(EL, Parent);
		RotDownRight.FlameSnd = new Sound(EL, Parent);
	}

	EngnList[0] = &RotDownLeft;
	EngnList[1] = &RotDownRight;
	EngnList[2] = &RotUpLeft;
	EngnList[3] = &RotUpRight;
	EngnList[4] = &FwdUp;
	EngnList[5] = &FwdDown;
	EngnList[6] = &RetUp;
	EngnList[7] = &RetDown;

	fill(OldEngnState, OldEngnState + 4, OFF);

	Reset();
}


Engine2::~Engine2() {
	if (ServerMode) {
		for (int x = 0; x < 4; ++x) {
			gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);
			delete EngnList[x]->FlameSnd;
		}
	}
}


float Engine2::GetForward() {
	return (FwdUp.CurrentForce + FwdDown.CurrentForce) - (RetUp.CurrentForce + RetDown.CurrentForce);
}


float Engine2::GetTorque() {
	return (RotUpLeft.CurrentForce - RotUpRight.CurrentForce) -
		(RotDownLeft.CurrentForce - RotDownRight.CurrentForce);
}


bool Engine2::NeedsUpdate(float TimeElapsed) {
	SetEnergy(TimeElapsed);

	if (ServerMode) {
		for (int x = 0; x < 4; ++x) {
			if (OldEngnState[x] != EngnList[x]->State) {
				if (EngnList[x]->State)
					gNetworkServer->AddSound(EngnList[x]->FlameSnd, false);
				else
					gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);

				OldEngnState[x] = EngnList[x]->State;
			}
		}

		return false;
	}

	return Engine::NeedsUpdate(TimeElapsed);
}


void Engine2::SetEnergy(float TimeElapsed) {
	for (int x = 0; x < 4; ++x) {
		if (EngnList[x]->State == ON) {

			EngnList[x]->CurrentForce += EngnList[x]->Power * TimeElapsed;
			if (EngnList[x]->CurrentForce > EngnList[x]->MaxForce)
				EngnList[x]->CurrentForce = EngnList[x]->MaxForce;
		}
		else
			EngnList[x]->CurrentForce = 0.f;
	}

	if (RetUp.State == ON && RetDown.State == ON) {
		RotUpRight.State = ON;
		RotDownRight.State = ON;

		RetUp.CurrentForce += RetUp.Power * TimeElapsed;
		if (RetUp.CurrentForce > RetUp.MaxForce)
			RetUp.CurrentForce = RetUp.MaxForce;

		RetDown.CurrentForce += RetUp.Power * TimeElapsed;
		if (RetDown.CurrentForce > RetDown.MaxForce)
			RetDown.CurrentForce = RetDown.MaxForce;
	}
	else {
		RetUp.CurrentForce = 0.f;
		RetDown.CurrentForce = 0.f;
	}


	if (FwdUp.State == ON && FwdDown.State == ON) {
		RotUpLeft.State = ON;
		RotDownLeft.State = ON;

		FwdUp.CurrentForce += FwdUp.Power * TimeElapsed;
		if (FwdUp.CurrentForce > FwdUp.MaxForce)
			FwdUp.CurrentForce = FwdUp.MaxForce;

		FwdDown.CurrentForce += FwdUp.Power * TimeElapsed;
		if (FwdDown.CurrentForce > FwdDown.MaxForce)
			FwdDown.CurrentForce = FwdDown.MaxForce;
	}
	else {
		FwdUp.CurrentForce = 0.f;
		FwdDown.CurrentForce = 0.f;
	}
}


void Engine2::Reset() {
	if (ServerMode) {
		for (int x = 0; x < 4; ++x)
			gNetworkServer->RemoveSound(EngnList[x]->FlameSnd->SoundUID);
	}

	for (int x = 0; x < 7; ++x) {
		EngnList[x]->State = OFF;
		EngnList[x]->CurrentForce = 0;
	}
}


EngineInfo* Engine2::GetEngineInfo() {
	_SPARKLET_ASSERT(ServerMode);

	EngineInfo *EI = new EngineInfo;

	EI->Count = 4;
	EI->Engine = new bool[4];
	fill(EI->Engine, EI->Engine + 4, false);

	for (int x = 0; x < 4; ++x)
		EI->Engine[x] = EngnList[x]->State;

	return EI;
}


#ifndef SERVER_ONLY

void Engine2::SetEngineFlame(const EngineInfo* EI) {
	_SPARKLET_ASSERT(!ServerMode);

	for (int x = 0; x < 4; ++x) {
		if (EI->Engine[x])
			EngnList[x]->Flame->Play();
		else
			EngnList[x]->Flame->Stop();
	}
}
#endif //SERVER_ONLY

