/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SERVER_ONLY


#include "sprite.h"
#include "cli_player_object.h"
#include "text.h"

#include "sparklet_utils.h"

#include "weapon.h"
#include "engine1.h"
#include "engine2.h"
#include "statusbar.h"

#include "network_client.h"

#define SHAPE_RES_ROT_STR std::string("shape_res_rot")
#define MOMENT_OF_INERTIA_STR std::string("moment_of_inertia")

#define SHAPE_RES_STR std::string("shape_res")
#define MASS_STR std::string("mass")

#define EXPLOSION_STR std::string("explosion")
#define NAME_STR std::string("name")
#define ENGINE_CHASSIS_STR std::string("engine_chassis")


extern NetworkClient *gNetworkClient;
extern float gTimeElapsed;


CliPlayerObject::CliPlayerObject(const TiXmlElement *Root, const Size &Dim, float Scale, Statusbar *lStb) :
	CliRotatedObject(Root, Dim, Scale),
	Torque(0),
	Impulse(0, 0),
	Stb(lStb) {

	ShapeResRot = XMLParser::GetValueF(Root, SHAPE_RES_ROT_STR, 1.f) * Scale;
	MomentOfInertia = XMLParser::GetValueF(Root, MOMENT_OF_INERTIA_STR, 1.f) * Scale;

	ShapeRes = XMLParser::GetValueF(Root, SHAPE_RES_STR, 0.1f) * Scale;
	Mass = XMLParser::GetValueF(Root, MASS_STR, 10.f) * Scale;

	const TiXmlElement* EL = XMLParser::GetChild(Root, ENGINE_STR);
	const UCHAR EngineType = XMLParser::GetValueL(EL, ENGINE_TYPE_STR, 0);
	switch (EngineType) {
		case 1: Machinery = new Engine1(EL, Geo, false); break;
		case 2: Machinery = new Engine2(EL, Geo, false); break;
		default: _ERROR_;
	}
	Extension.push_back(Machinery);

	EL = XMLParser::GetChild(Root, EXPLOSION_STR);
	Explosion = new Sprite(EL, Geo, false);
	Explosion->Stop();
	Extension.push_back(Explosion);

	EL = XMLParser::GetChild(Root, ENGINE_CHASSIS_STR);
	EngineChassis = new Sprite(EL, Geo, false);
	EngineChassis->Pause();
	Extension.push_back(EngineChassis);

	EL = XMLParser::GetChild(Root, NAME_STR);
	Name = new Text(EL, Geo);
	Extension.push_back(Name);

	Chassis = GetBaseSprite();
	Chassis->Pause();

	ArrangeWeapons(Root);

	ResetExtensions();
}


void CliPlayerObject::ArrangeWeapons(const TiXmlElement *Root) {
	const TiXmlElementList* Child = XMLParser::GetChildren(Root, WEAPON_STR);

	for (USHORT x = 0; x < Child->size(); ++x) {
		Weapon *NewOne = new Weapon(Child->at(x), Geo, Identity, Stb, false);
		Extension.push_back(NewOne);
		WeaponList.insert(make_pair(NewOne->GetIndex(), NewOne));
	}
	delete Child;
}


void CliPlayerObject::SetMsg(const FullRotationInfo *FRI) {
	Direction = FRI->Direction;
	Rotation = FRI->Rotation;
	Torque = FRI->Torque;
}


void CliPlayerObject::SetMsg(const FullVelocityInfo *VI) {
	Impulse = VI->Impulse;
	Velocity = VI->Velocity;
	Geo = VI->Loc;
}


void CliPlayerObject::SetMsg(const EngineInfo *EI) {
	Machinery->SetEngineFlame(EI);
}


void CliPlayerObject::SetMsg(const WeaponInfo *WI) {
	WeaponList[WI->Index]->SetWeaponInfo(WI);
}


void CliPlayerObject::SetMsg(const ShipState *SS) {
	if (Stb)
		Stb->SetEnergy(SS->Energy);
	Chassis->JumpToFrame(SS->ChassisFrame);
	EngineChassis->JumpToFrame(SS->ChassisFrame);

	if (SS->Energy == 0 && Explosion->GetCurrentFrame() == 0)
		Explosion->PlayOnce();
}


void CliPlayerObject::SetMsg(const ShipInfo *SI) {
	Name->SetText(SI->Name);
	Identity.Team = SI->Team;
	Chassis->SetTintColor(GetTeamTintColor(Identity.Team));
}


void CliPlayerObject::OnUpdate() {
	if (Explosion->IsOver()) {
		Explosion->JumpToStart();
		Explosion->Stop();
	}

	UpdateVelocity();
	UpdateRotation();

	//We calculated only Rotation and Velocity, CliRotatedObject and CliMobileObject will apply them
	//to Direction and Geo values.
	CliRotatedObject::OnUpdate();
}


void CliPlayerObject::UpdateVelocity() {
	Velocity += Impulse / Mass;
	Velocity -= CalcResistance(Velocity, ShapeRes * gNetworkClient->GetViscosity() / Mass * gTimeElapsed);
}


void CliPlayerObject::UpdateRotation() {
	float AngularMomentum = Torque * gTimeElapsed;

	Rotation += AngularMomentum / MomentOfInertia;
	Rotation -= CalcResistance(Rotation, ShapeResRot * gNetworkClient->GetViscosity() / MomentOfInertia * gTimeElapsed);
}


#endif //SERVER_ONLY
