/***************************************************************************
 *   Copyright (C) 2003 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "statical_sprite.h"
#include "bullet.h"
#include "sparklet_utils.h"


#define VELOCITY_MODIFIER_STR std::string("velocity_modifier")
#define MOTION_TYPE_STR std::string("motion_type")
#define ABS_MAX_STR std::string("abs_max")
#define ROT_PER_SEC_STR std::string("rot_per_sec")

#define BOUNCER_STR std::string("bouncer")
#define BOUNCE_MAX_STR std::string("bounce_max")


extern float gTimeElapsed;


const ULONG Bullet::ID(GEN_ID('b', 'l', 'l', 't'));


Bullet::Bullet(const TiXmlElement* Att, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	Projectile(Att, Scale),
	VelocityModifierAng(0),
	InitialAngle(0) {

	const TiXmlElement *Root = Parser->GetRoot();

	const TiXmlElementList* Nl = XMLParser::GetAllChildren(Root, VELOCITY_MODIFIER_STR);
	if (Nl->size() == 1) {
		const TiXmlElement *El = Nl->at(0);
		MotionType = XMLParser::GetValueS(El, MOTION_TYPE_STR, "");
		VelocityModifierAbsMax = XMLParser::GetValueF(El, ABS_MAX_STR, 0.f);
		VelocityModifierRotPerSec = XMLParser::GetValueF(El, ROT_PER_SEC_STR, 0.f);
	}
	delete Nl;

	BaseBitmap = Object::GetBaseBitmap(Root);
	IsFixedShape = true;
	AllowInaccurateCollisionDetection = true;

	Bouncer = XMLParser::GetValueL(Root, BOUNCER_STR, 0);
	BounceMax = XMLParser::GetValueL(Root, BOUNCE_MAX_STR, 5);
	BounceCount = 0;

	//LastObjCollided.Team = TEAM_NO_TEAM;
	//IsLastObjectCollidedBouncer = false;

	//don't support bounce and velocity modifier attributes at the same time
	if (Bouncer && !MotionType.empty()) {
		_ERROR_;
		_SAY("bouncer and velocity modifier attributes not supported at the same time");
		//just disable bouncer, leave vel. modifier on
		Bouncer = false;
	}

	PreventMultipleCollide = true;
}


void Bullet::CheckData() {
	if (Bouncer)
		Projectile::CheckData();

	//nothing to do in CheckData() if not bouncing
	//no data can cahnge
	//if appling velocity modifier the client is capable to do it itself
}


void Bullet::GetChangedData(MessageCoder &MC) const {
	if (Bouncer)
		Projectile::GetChangedData(MC);

	//nothing to do in GetChangedData() if not bouncing
	//no data can cahnge
	//if appling velocity modifier the client is capable to do it itself
}


void Bullet::GetCompleteData(MessageCoder &MC) const {
	if (!MotionType.empty()) {
		VelocityInfo *VI = new VelocityInfo();
		//undo the last modification to get the unmodified velocity that the client can use to do it's own modification
		VI->Velocity = Velocity - polar<float>(sinf(VelocityModifierAng) * VelocityModifierAbsMax, InitialAngle + M_PI / 2.f);
		VI->Loc = Geo;
		MC.Put(VI);

		Object::GetCompleteData(MC);
	}
	else
		Projectile::GetCompleteData(MC);
}


void Bullet::Damage(const PhysicalObject *Obj) {
	if (Bouncer) {
		bool IsLastObjectCollidedBouncer;
		//get the last object collided
		IDENTITY LastObjCollided = Obj->Identity;
		//increase bounce count because there is a limit
		BounceCount++;

		//determine if the object we collided with is also a bouncer bullet
		if (Obj->GetTypeID() == GetTypeID())
			IsLastObjectCollidedBouncer = dynamic_cast<const Bullet*>(Obj)->Bouncer;
		else
			IsLastObjectCollidedBouncer = false;

		//if the bouncing bullet collided with an "independent" object, like a wall or stone,
		//or it collided with another bouncer bullet and bounce limit is not reached, then
		//apply CollidingImpulse and do not explode.
		if ((LastObjCollided.Team == TEAM_NO_TEAM || IsLastObjectCollidedBouncer) && BounceCount <= BounceMax) {
			Impulse += CollidingImpulse;
		}
		else {
			Explode();
			Unlink();
		}
	}
	else {
		Explode();
		Unlink();
	}

	Projectile::Damage(Obj);
}


void Bullet::OnUpdate() {
	Velocity += Impulse / Mass;
	LimitSpeed();

	if (!MotionType.empty()) {
		//do the velicity modulation
		//currently only sinusoidal modulation supported
		if (MotionType == "sinusoidal") {
			//undo last Velocity modification
			Velocity -= polar<float>(sinf(VelocityModifierAng) * VelocityModifierAbsMax, InitialAngle + M_PI / 2.f);
			VelocityModifierAng += gTimeElapsed * (M_PI * 2.f * VelocityModifierRotPerSec);
			Velocity += polar<float>(sinf(BoundAngle(VelocityModifierAng)) * VelocityModifierAbsMax, InitialAngle + M_PI / 2.f);
		}
	}

	BaseBitmap->SetDirection(arg(Velocity));

	Projectile::OnUpdate();
}
