/***************************************************************************
 *   Copyright (C) 2004 by Milan Mimica                                    *
 *   milan.mimica@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "sprite.h"
#include "bouncing.h"
#include "invisible_particle.h"
#include "sparklet_utils.h"
#include "sound.h"

#define BASE_SPRITE_STR std::string("base_sprite")
#define EXPLOSION_SPRITE_STR std::string("explosion_sprite")
#define STRENGTH_STR std::string("strength")

#define CAN_EXPLODE_STR std::string("can_explode")
#define RND_START_FRAME_STR std::string("rnd_start_frame")

#define INVISIBLE_EXPLOSION_STR std::string("invisible_explosion")
#define INVISIBLE_PARTICLE_STR std::string("invisible_particle")
#define PARTICLE_COUNT_STR std::string("particle_count")

#define IMPULSE_STR std::string("impulse")
#define RND_IMPULSE_STR std::string("rnd_impulse")

#define START_ROLING_VELOCITY 40.f

#define EXPLOSION_SOUND_STR std::string("explosion_sound")


extern float gTimeElapsed;
extern NetworkServer *gNetworkServer;


Bouncing::Bouncing(const TiXmlElement* Att, float Scale) :
	Object(Att, Scale),
	MobileObject(Att, Scale),
	PhysicalObject(Att, Scale),
	RoundObject(Att, Scale),
	NonRotatedObject(Att, Scale) {

	const TiXmlElement *Root = Parser->GetRoot();

	BaseSprite = GetBaseSprite(Root);

	CanExplode = XMLParser::GetValueL(Root, CAN_EXPLODE_STR, 1);
	RndStartFrame = XMLParser::GetValueL(Root, RND_START_FRAME_STR, 0);

	if (CanExplode) {
		const TiXmlElement *EL = Parser->GetChild(Root, EXPLOSION_SPRITE_STR);
		Explosion = new Sprite(EL, Geo, true);
		Explosion->Stop();
		Extension.push_back(Explosion);

		Strength = XMLParser::GetValueL(Root, STRENGTH_STR, 1) * 200;

		const TiXmlElement *DOMExplosion = XMLParser::GetChild(Root, INVISIBLE_EXPLOSION_STR);
		ParticleCount = XMLParser::GetValueL(DOMExplosion, PARTICLE_COUNT_STR, 0);
		_SPARKLET_ASSERT(ParticleCount > 0);
		ExplosionImpulse = XMLParser::GetValueL(DOMExplosion, IMPULSE_STR, 10);
		RndExplosionImpulse = XMLParser::GetValueL(DOMExplosion, RND_IMPULSE_STR, 5);

		DOMParticle = XMLParser::GetChild(DOMExplosion, INVISIBLE_PARTICLE_STR);

		ExplosionSound = new Sound(XMLParser::GetChild(Root, EXPLOSION_SOUND_STR), Geo);
	}

	if (RndStartFrame)
		BaseSprite->JumpToFrame((int)((float)BaseSprite->GetFrameCount() * rand() / (RAND_MAX + 1.0)));
	BaseSprite->SetFPS(0);
	BaseSprite->Pause();

	OldSpriteFPS = 0;
	DataChanged = false;

	Stage = STAGE_NORMAL;
	OldStage = STAGE_NORMAL;

	PreventMultipleCollide = true;
}


Bouncing::~Bouncing() {
	if (CanExplode)
		delete ExplosionSound;
}


void Bouncing::Damage(const PhysicalObject *Obj) {
	PhysicalObject::Damage(Obj);

	if (CanExplode) {
		if (Strength > 0)
			Strength -= abs(CollidingImpulse) + CollidingEnergy;
		if (Strength <= 0 && Stage != STAGE_EXPLODING) {
			OldStage = Stage;
			Stage = STAGE_EXPLODING;
			Explode();
		}
	}
}


void Bouncing::OnUpdate() {
	Impulse += CollidingImpulse;
	Velocity += Impulse / Mass;

	LimitSpeed();

	Geo += Velocity * gTimeElapsed;

	switch (Stage) {
		case STAGE_NORMAL:
			if (abs(Impulse) && abs(Velocity) > START_ROLING_VELOCITY) {
				const float CurrentFPS = BaseSprite->GetFPS();
				if (CurrentFPS < 5.f) {
					BaseSprite->SetFPS((7.f * rand() / (RAND_MAX + 1.0)) + 13);
				}
				else {
					const float RndFPSFac = (30.f * rand() / (RAND_MAX + 1.0)) / 100.f + 0.85f;
					const float FPS = CurrentFPS * RndFPSFac;
					BaseSprite->SetFPS(FPS > 50.f ? 20.f : FPS);
					if (!BaseSprite->IsPlaying())
						BaseSprite->Play();
				}
			}
		break;
		case STAGE_EXPLODING:
			if (CanExplode)
				if (Explosion->IsOver())
					Unlink();
		break;
	}

	LimitSpeed();

	NextFrame(gTimeElapsed);
	PhysicalObject::OnUpdate();
}


void Bouncing::CheckData() {
	DataChanged = false;

	if (CanExplode) {
		if (Stage != OldStage) {
			DataChanged = true;
			OldStage = Stage;
		}
	}

	if (OldSpriteFPS != BaseSprite->GetFPS()) {
		OldSpriteFPS = BaseSprite->GetFPS();
		DataChanged = true;
	}

	MobileObject::CheckData();
}


void Bouncing::GetChangedData(MessageCoder &MC) const {
	if (DataChanged) {
		StateInfoGeneric1 *SIG1 = new StateInfoGeneric1;
		SIG1->State = 0;
		SIG1->State = (int)BaseSprite->GetCurrentFrame();
		SIG1->State |= ((Stage == STAGE_EXPLODING) ? 0x80 : 0);
		MC.Put(SIG1);

		StateInfoGeneric2 *SIG2 = new StateInfoGeneric2;
		SIG2->State = (int)(10.f * BaseSprite->GetFPS() / 2.f);
		MC.Put(SIG2);
	}

	MobileObject::GetChangedData(MC);
}


void Bouncing::GetCompleteData(MessageCoder &MC) const {
	StateInfoGeneric1 *SIG1 = new StateInfoGeneric1;
	SIG1->State = 0;
	SIG1->State = (int)BaseSprite->GetCurrentFrame();
	SIG1->State |= ((Stage == STAGE_EXPLODING) ? 0x80 : 0);
	MC.Put(SIG1);

	StateInfoGeneric2 *SIG2 = new StateInfoGeneric2;
	SIG2->State = (int)(10.f * BaseSprite->GetFPS() / 2.f);
	MC.Put(SIG2);

	MobileObject::GetCompleteData(MC);
}


void Bouncing::Explode() {
	BaseSprite->Stop();
	Explosion->PlayOnce();
	Velocity = Vector<float>(0, 0);

	gNetworkServer->AddSound(ExplosionSound, true);

	for (USHORT x = 0; x < ParticleCount; ++x) {
		const float AngleOffset = (rand() % 100) / 100.f;
		const float Angle = BoundAngle(AngleOffset + (2 * M_PI) * (x / (float)ParticleCount));
		const float Force = ExplosionImpulse + rand() % RndExplosionImpulse;
		InvisibleParticle *Par = new InvisibleParticle(DOMParticle, 0, 1);
		Par->SetImpulse(polar(Force, Angle));
		Par->SetLocC(GetCenter(Geo));
		Par->Identity = Identity;
	}
}
