/***************************************************************************
                          bitmap_server.cpp  -  description
                             -------------------
    begin                : Mon Mar 31 2003
    copyright            : (C) 2003 by Milan Mimica
    email                : milan.mimica@gmail.com   
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "bitmap_server.h"

#include "sparklet_utils.h"

#define STUB_BITMAP_PATH "bitmaps/stub.tga"

using namespace std;


BitmapServer::~BitmapServer() {
	while(!SharedMem.empty()) {
		destroy_bitmap(SharedMem.back());
		SharedMem.pop_back();
	}

	for (map<const string, BITMAP*>::iterator x = BitmapList32.begin(); x != BitmapList32.end(); ++x)  {
		destroy_bitmap(x->second);
	}

	for (map<const string, BITMAP*>::iterator x = BitmapList16.begin(); x != BitmapList16.end(); ++x)  {
		destroy_bitmap(x->second);
	}
}


BITMAP *BitmapServer::LoadBitmap32(const string &Path) {
	//Check wether the file exists.
	if (!exists(Path.c_str())) {
		_ERROR_;
		_SAY("Bitmap '" + Path + "' does not exist!!!");

		const string FullStubPath = FixFilenameSlashes(string(DATA_PREFIX) + STUB_BITMAP_PATH);
		if (Path == FullStubPath) {
			_SAY("Bitmap '" + FullStubPath + "' does not exist!!!");
			_SAY("No stub found!");
			return NULL;
		}

		_SAY("Using stub.");
		return LoadBitmap32(FullStubPath);
	}

	//Load the bitmap.
	//We don't know bitmap's color depth before we load it. That's why we'll just load it as it is
	//and eventually convert it to 32-bit
	set_color_conversion(COLORCONV_NONE);
	BITMAP *Bmp = load_bitmap(Path.c_str(), NULL);
	_SPARKLET_ASSERT(Bmp);

	/*
	We can use 24-bit bitmap too. They have no alpha channel, instead they use magenta.
	We'll create a brand new 32-bit bitmap, clear it, and put in just the non-magenta pixels.
	*/

	//Do not handle depths other than 24 and 32.
	if (bitmap_color_depth(Bmp) != 24 && bitmap_color_depth(Bmp) != 32) {
		_ERROR_;
		_SAY("Can't handle bitmaps other than 24 or 32 bit color depth!");
		_SAY("The bitmap is: '" + Path + "'");
		destroy_bitmap(Bmp);
		return NULL;
	}

	BITMAP *RetVal = Bmp;

	//If 24-bit proceed with converting.
	if (bitmap_color_depth(Bmp) == 24) {
		RetVal = create_bitmap_ex(32, Bmp->w, Bmp->h);
		_SPARKLET_ASSERT(RetVal);
		clear_bitmap(RetVal);

		//Replace every non-magenta pixel with the one from the original bitmap and alpha 255.
		for (int y = 0; y < Bmp->h; ++y) {
			for (int x = 0; x < Bmp->w; ++x) {
				const int c = _getpixel24(Bmp, x, y);
				if (c != makecol24(255, 0, 255))
					_putpixel32(RetVal, x, y, makeacol32(getr32(c), getg32(c), getb32(c), 255));
			}
		}

		//Destroy the 24-bit bitmap.
		destroy_bitmap(Bmp);
	}

	return RetVal;
}


BITMAP *BitmapServer::LoadBitmap16(const string &Path) {
	//Check wether the file exists.
	if (!exists(Path.c_str())) {
		_ERROR_;
		_SAY("Bitmap '" + Path + "' does not exist!!!");

		const string FullStubPath = FixFilenameSlashes(string(DATA_PREFIX) + STUB_BITMAP_PATH);
		if (Path == FullStubPath) {
			_SAY("Bitmap '" + FullStubPath + "' does not exist!!!");
			_SAY("No stub found!");
			return NULL;
		}

		_SAY("Using stub.");
		return LoadBitmap32(FullStubPath);
	}

	//Load the bitmap.
	//We don't know bitmaps color depth before we load it. That's why we'll just load it as it is
	//and eventually convert it to 16-bit
	set_color_conversion(COLORCONV_NONE);
	BITMAP *Bmp = load_bitmap(Path.c_str(), NULL);
	_SPARKLET_ASSERT(Bmp);

	//Do not handle depths other than 24 and 32.
	if (bitmap_color_depth(Bmp) != 24 && bitmap_color_depth(Bmp) != 32) {
		_ERROR_;
		_SAY("Can't handle bitmaps other than 24 or 32 bit color depth!");
		_SAY("The bitmap is: '" + Path + "'");
		destroy_bitmap(Bmp);
		return NULL;
	}

	/*
	Bitmaps saved to disk can be 32-bit or 24-bit. We'll have to convert them to 16-bit. 32-bit
	bitmaps use alpha channels instead of magenta so we'll have to convert that too.
	*/

	//Create the bitmap we will return.
	BITMAP *RetVal = create_bitmap_ex(16, Bmp->w, Bmp->h);
	_SPARKLET_ASSERT(RetVal);


	//Convert 24-bit bitmap to 16-bit.
	if (bitmap_color_depth(Bmp) == 24) {
		blit(Bmp, RetVal, 0, 0, 0, 0, Bmp->w, Bmp->h);

		destroy_bitmap(Bmp);
	}
	//convert 32-bit to 16-bit
	else if (bitmap_color_depth(Bmp) == 32) {
		/*
		We'll fill the 16-bit bitmap with magenta and then put every visible pixel from 32-bit
		bitmap into it.
		*/
		clear_to_color(RetVal, makecol16(255, 0, 255));

		for (int y = 0; y < Bmp->h; ++y) {
			for (int x = 0; x < Bmp->w; ++x) {
				const int c = _getpixel32(Bmp, x, y);
				//Because this bitmaps will be used mostly by colision detection routines let's
				//pretend that the hardly visible pixels are invisible
				if (geta32(c) > 64)
					_putpixel16(RetVal, x, y, makecol16(getr32(c), getg32(c), getb32(c)));
			}
		}

		destroy_bitmap(Bmp);
	}

	return RetVal;
}


#ifndef SERVER_ONLY
GLuint BitmapServer::LoadTexture(BITMAP *bmp) {
	GLuint RetVal = 0;

	if (bitmap_color_depth(bmp) != 32)
		_ERROR_;

	RetVal = allegro_gl_make_texture_ex(AGL_TEXTURE_FLIP |
		AGL_TEXTURE_HAS_ALPHA |
		AGL_TEXTURE_RESCALE /*|
		AGL_TEXTURE_MIPMAP*/,
		bmp, GL_RGBA8);
	_SPARKLET_ASSERT(RetVal);

	return RetVal;
}
#endif //SERVER_ONLY


BITMAP *BitmapServer::GetBitmap32(const string &RelPath) {
	const string FullPath = FixFilenameSlashes(string(DATA_PREFIX) + RelPath);

	if (BitmapList32.find(FullPath) == BitmapList32.end()) {
		BITMAP *b = LoadBitmap32(FullPath);
		_SPARKLET_ASSERT(b);

		BitmapList32.insert(make_pair(FullPath, b));
	}

	BITMAP *Bmp = BitmapList32.find(FullPath)->second;

	return Bmp;
}


BITMAP *BitmapServer::GetBitmap16(const string &RelPath) {
	const string FullPath = FixFilenameSlashes(string(DATA_PREFIX) + RelPath);

	if (BitmapList16.find(FullPath) == BitmapList16.end()) {
		BITMAP *b = LoadBitmap16(FullPath);
		_SPARKLET_ASSERT(b);

		BitmapList16.insert(make_pair(FullPath, b));
	}

	BITMAP *Bmp = BitmapList16.find(FullPath)->second;

	return Bmp;
}


#ifndef SERVER_ONLY
GLuint BitmapServer::GetTexture(BITMAP *bmp) {
	if (TextureList.find(bmp) == TextureList.end()) {
		const GLuint t = LoadTexture(bmp);
		_SPARKLET_ASSERT(t);

		TextureList.insert(make_pair(bmp, t));
	}

	GLuint t = TextureList.find(bmp)->second;

	return t;
}
#endif //SERVER_ONLY


//Returns a pointer to a memory bitmap sized Dim. The returned bitmap pointers share memory
//space so you can't use them to store data permanently nor use them at the same time.
BITMAP* BitmapServer::GetBitmap16(const Size &Dim) {
	BITMAP *RetVal = 0;

	//Iterate through alocated memory bitmaps seeking for same-sized or larger bitmap. If we found
	//it create a subbitmap.
	for (USHORT x = 0; x < SharedMem.size() && !RetVal; ++x) {
		if (SharedMem[x]->w >= Dim.Width && SharedMem[x]->h >= Dim.Height)
			RetVal = create_sub_bitmap(SharedMem[x], 0, 0, Dim.Width, Dim.Height);
	}

	//If no stored bitmap is large enough alocate a new one, and create a subbitmap of it.
	if (!RetVal) {
		SharedMem.push_back(create_bitmap_ex(16, Dim.Width, Dim.Height));
		RetVal = create_sub_bitmap(SharedMem.back(), 0, 0, Dim.Width, Dim.Height);
	}

	_SPARKLET_ASSERT(RetVal);
	return RetVal;
}

