#include "map.h"

//! \brief Carrega os recursos.
h8or::Map::Map()
{
	screen2 = create_bitmap(SCREEN_W + (TILE_SIZE * 2), SCREEN_H + (TILE_SIZE * 2) );
    for ( int i = 0; i < 5; i++)
    {
        autotileCSE[i] = create_bitmap(16, 16);
        autotileCSD[i] = create_bitmap(16, 16);
        autotileCIE[i] = create_bitmap(16, 16);
        autotileCID[i] = create_bitmap(16, 16);
    }
    floor = create_bitmap(32, 32);
}



//! \brief Libera os recursos requisitados. Apaga o autotile
h8or::Map::~Map()
{
	destroy_bitmap( screen2 );
	destroy_bitmap( floor );
	destroy_bitmap( tileset );
	for ( int i = 0; i < 5; i++ )
	{
		destroy_bitmap(autotileCSE[i]);
        destroy_bitmap(autotileCSD[i]);
        destroy_bitmap(autotileCIE[i]);
        destroy_bitmap(autotileCID[i]);
	}
}

//! Faz a lgica do mapa.
void h8or::Map::Update()
{
	if (state == STATE_RUN)
	{
		switch (direcao)
		{
			case DIR_UP:
				if ( visibleY1 - 1 >= 0 && visibleY2 - 1 <= heightPixels - 1 )
				{
					visibleY1 -= 1;
					visibleY2 -= 1;
				}
				break;

			case DIR_DOWN:
				if ( visibleY1 + 1 >= 0 && visibleY2 + 1 <= heightPixels - 1 )
				{
					visibleY1 += 1;
					visibleY2 += 1;
				}
				break;

			case DIR_LEFT:
				if ( visibleX1 - 1 >= 0 && visibleX2 - 1 <= widthPixels - 1 )
				{
					visibleX1 -= 1;
					visibleX2 -= 1;
				}
				break;

			case DIR_RIGHT:
				if ( visibleX1 + 1 >= 0 && visibleX2 + 1 <= widthPixels - 1 )
				{
					visibleX1 += 1;
					visibleX2 += 1;
				}
				break;
		}
		if (++offset == 32)
		{
			offset = 0;
			state = STATE_STAND;
		}
	}

}


/*! \brief Calcula os ndices dos autotiles.
 *  \param a Vertical.
 *  \param b Horizontal.
 *  \param c Diagonal.
 */
int h8or::Map::CalculaIndice(int a, int b, int c)
{
	int s0 = a & ~(c & b);
	int s1 = b & ~(a & c);
	int s2 = a & b & c;
	int result = (s2 << 2) | (s1 << 1) | s0;
	return result;
}

//! Carrega as informaes do mapa.
int h8or::Map::LoadMap(const char* fileName)
{
	offset = 0;
	state = STATE_STAND;
	SetVisible(0, 0);
	TiXmlDocument *doc = new TiXmlDocument(fileName);
	doc->LoadFile();
	TiXmlElement *root = doc->RootElement();
	TiXmlElement *mapa = root->FirstChildElement("map");

	const char *autotileName = mapa->Attribute("autotile");
	LoadAutoTile(autotileName);

	mapa->QueryIntAttribute("width", &widthTiles);
	mapa->QueryIntAttribute("height", &heightTiles);
	widthPixels = widthTiles * TILE_SIZE;
	heightPixels = heightTiles * TILE_SIZE;
	matriz = new int*[heightTiles];
	for (int i = 0; i < heightTiles; i++)
		matriz[i] = new int[widthTiles];

	// preenche a matriz
	TiXmlElement *iterator = mapa->FirstChildElement("data");
    for (int linha = 0; linha < heightTiles; linha++)
    {	string linhaData = iterator->GetText();
		for ( int coluna = 0; coluna < widthTiles; coluna++)
			matriz[linha][coluna] = linhaData[coluna] - 48; // 48  o codigo ASCII do caracter '0'?
		iterator = iterator->NextSiblingElement("data");
    }

    mapa = root->FirstChildElement("layer1");
    const char *tilesetName = mapa->Attribute("tileset");
    tileset = load_bitmap(tilesetName, NULL);
    tilesetWidth = tileset->w / TILE_SIZE;
    if (!tileset)
    {	allegro_message("No foi possvel carregar o tileset.");
		exit(1);
    }
    iterator = mapa->FirstChildElement("data");
	for (int linha = 0; linha < heightTiles; linha++)
    {	string linhaData = iterator->GetText();
		for ( int coluna = 0; coluna < widthTiles; coluna++)
			matriz[linha][coluna] |= (linhaData[coluna]-65) << 1; // 65  o codigo ASCII do caracter 'A'?
		iterator = iterator->NextSiblingElement("data");
    }

	return 0;
}

//! Carrega as texturas no formato do RPG Maker XP Autotile.
int h8or::Map::LoadAutoTile(const char* autotileName)
{
	BITMAP *image = load_bitmap(autotileName, NULL);
	if (!image)
	{
		allegro_message("Unable to open file '%s'", autotileName);
		return 1;
	}
	blit(image, floor, 32, 0, 0, 0, 32, 32);

	blit(image, autotileCSE[0],    0,    0, 0, 0, 16, 16);
	blit(image, autotileCSD[0], 0+16,    0, 0, 0, 16, 16);
	blit(image, autotileCIE[0],    0, 0+16, 0, 0, 16, 16);
	blit(image, autotileCID[0], 0+16, 0+16, 0, 0, 16, 16);

	blit(image, autotileCSE[1],     0,    64, 0, 0, 16, 16);
	blit(image, autotileCSD[1], 64+16,    64, 0, 0, 16, 16);
	blit(image, autotileCIE[1],     0, 64+16, 0, 0, 16, 16);
	blit(image, autotileCID[1], 64+16, 64+16, 0, 0, 16, 16);

	blit(image, autotileCSE[2],    32,    32, 0, 0, 16, 16);
	blit(image, autotileCSD[2], 32+16,    32, 0, 0, 16, 16);
	blit(image, autotileCIE[2],    32, 96+16, 0, 0, 16, 16);
	blit(image, autotileCID[2], 32+16, 96+16, 0, 0, 16, 16);

	blit(image, autotileCSE[3],    64,    0, 0, 0, 16, 16);
	blit(image, autotileCSD[3], 64+16,    0, 0, 0, 16, 16);
	blit(image, autotileCIE[3],    64, 0+16, 0, 0, 16, 16);
	blit(image, autotileCID[3], 64+16, 0+16, 0, 0, 16, 16);

	blit(image, autotileCSE[4],    32,    64, 0, 0, 16, 16);
	blit(image, autotileCSD[4], 32+16,    64, 0, 0, 16, 16);
	blit(image, autotileCIE[4],    32, 64+16, 0, 0, 16, 16);
	blit(image, autotileCID[4], 32+16, 64+16, 0, 0, 16, 16);

	destroy_bitmap(image);
	return 0;
}

//! Desenha o mapa na tela.
void h8or::Map::Draw(BITMAP* surface)
{
    int visible_tile_x1;     /* upper-left tile of the viewable area */
    int visible_offset_x1;   /* offset into the upper-left tile */
    int visible_tile_y1;     /* upper-left tile of the viewable area */
    int visible_offset_y1;   /* offset into the upper-left tile */
    int visible_tile_x2;     /* lower-right tile of the viewable area */
    int visible_tile_y2;     /* lower-right tile of the viewable area */

    /* find exact location of the current viewable section of the map using */
    /* 4 numbers for the current tile and how offset into that tile you are */
    visible_tile_x1 = visibleX1 / TILE_SIZE;
    visible_offset_x1 = visibleX1 % TILE_SIZE;
    visible_tile_y1 = visibleY1 / TILE_SIZE;
    visible_offset_y1 = visibleY1 % TILE_SIZE;

    visible_tile_x2 = visibleX2 / TILE_SIZE;
    if (visibleX2 % TILE_SIZE)
        visible_tile_x2++;

    visible_tile_y2 = visibleY2 / TILE_SIZE;
    if (visibleY2 % TILE_SIZE)
        visible_tile_y2++;

    /* construct screen2 so that all viewable and partially viewable tiles are */
    /* in it */
    for (int coluna = visible_tile_x1; coluna < visible_tile_x2; coluna++)
    {
        for (int linha = visible_tile_y1; linha < visible_tile_y2; linha++)
        {
        	if ( !(matriz[linha][coluna] & 1) )
				blit(floor, screen2, 0, 0,
                        (coluna - visible_tile_x1) * TILE_SIZE,
						(linha - visible_tile_y1) * TILE_SIZE,
						TILE_SIZE, TILE_SIZE);
			else
			{
				int a, b, c;

				a = linha > 0 ? matriz[linha-1][coluna] & 1 : 1;
				b = coluna > 0 ? matriz[linha][coluna-1]& 1 : 1;
				c = linha > 0 && coluna > 0 ? matriz[linha-1][coluna-1]& 1 : 1;
				blit(autotileCSE[CalculaIndice(a, b, c)], screen2, 0, 0, (coluna - visible_tile_x1) * TILE_SIZE, (linha - visible_tile_y1) * TILE_SIZE, 16, 16);

				a = linha > 0 ? matriz[linha-1][coluna]& 1 : 1;
				b = coluna < widthTiles-1 ? matriz[linha][coluna+1]& 1 : 1;
				c = linha > 0 && coluna < widthTiles-1 ? matriz[linha-1][coluna+1]& 1 : 1;
				blit(autotileCSD[CalculaIndice(a, b, c)], screen2, 0, 0, (coluna - visible_tile_x1) * TILE_SIZE + 16, (linha - visible_tile_y1) * TILE_SIZE, 16, 16);

				a = linha < heightTiles-1 ? matriz[linha+1][coluna]& 1 : 1;
				b = coluna > 0 ? matriz[linha][coluna-1]& 1 : 1;
				c = linha < heightTiles-1 && coluna > 0 ? matriz[linha+1][coluna-1]& 1 : 1;
				blit(autotileCIE[CalculaIndice(a, b, c)], screen2, 0, 0, (coluna - visible_tile_x1) * TILE_SIZE, (linha - visible_tile_y1) * TILE_SIZE + 16, 16, 16);

				a = linha < heightTiles-1 ? matriz[linha+1][coluna]& 1 : 1;
				b = coluna < widthTiles-1 ? matriz[linha][coluna+1]& 1 : 1;
				c = linha < heightTiles-1 && coluna < widthTiles-1 ? matriz[linha+1][coluna+1]& 1 : 1;

				blit(autotileCID[CalculaIndice(a, b, c)], screen2, 0, 0, (coluna - visible_tile_x1) * TILE_SIZE + 16, (linha - visible_tile_y1) * TILE_SIZE + 16, 16, 16);
			}

			// agora  a vez de desenhar os layers
			if (matriz[linha][coluna] >> 1)
			{
				int valor = (matriz[linha][coluna] >> 1)-1;
				int deslocamentoX = (valor % tilesetWidth) * TILE_SIZE;
				int deslocamentoY = (valor / tilesetWidth) * TILE_SIZE;
				masked_blit(tileset, screen2, deslocamentoX, deslocamentoY, (coluna - visible_tile_x1) * TILE_SIZE, (linha - visible_tile_y1) * TILE_SIZE, TILE_SIZE, TILE_SIZE);
			}
        }
    }

    /* copy viewable part of screen2 to screen1 */
    blit(screen2, surface, visible_offset_x1, visible_offset_y1, 0, 0, SCREEN_W, SCREEN_H);
    //textprintf_ex(surface, font, 0, 0, makecol(255, 255, 255), 0, "x2 = %3d   y2 = %3d", visibleX2, visibleY2);
}



void h8or::Map::SetVisible(int x, int y)
{
//	if (x >= 0 && y >= 0 && x+SCREEN_W <= widthPixels && y+SCREEN_H <= heightPixels)
	visibleX1 = x;
	visibleY1 = y;
	visibleX2 = visibleX1 + SCREEN_W - 1;
	visibleY2 = visibleY1 + SCREEN_H - 1;
}

int h8or::Map::Move(Direcao dir)
{
	if (dir == DIR_UP && visibleY1 == 0 ||
		dir == DIR_DOWN && visibleY2 == TILE_SIZE * heightTiles - 1 ||
		dir == DIR_LEFT && visibleX1 == 0 ||
		dir == DIR_RIGHT && visibleX2 == TILE_SIZE * widthTiles - 1)
		return 1;

	state = STATE_RUN;
	direcao = dir;
	return 0;
}

State h8or::Map::GetCurrentState()
{
	return state;
}
