#include <math.h>
#include "core.h"

//memory locks
extern BITMAP* gbuffer;
extern BITMAP* mbuffer;
extern BITMAP* terrain;
extern int font_height,colour_black,colour_light,colour_red,colour_dark,colour_white;
extern volatile int frame_count;
extern volatile int fps;
extern int gbufw_c,gbufh_c;

volatile int game_time;
extern volatile int frame_count; //updated every frame to keep fps in sync

const int MAX_ANGLE=255;

//MATRIX FOR ROTATION
//old method using calculated matrixes
//(new one uses allegro matrices and transformations)
struct TDPoint
{
	long x;
	long y;
	long z;
};
double rotmatrix[3][3];

MATRIX_f terrainmatrix;

int viewx=500;
int viewy=255;

int polycount=0;
int levelsmoothness=3,leveldetail=42,levelwater=96,levelwaterdepth=10,levelbeach=10;
bool showwork=false;
bool refresh3D=false;
bool show3D=false;
double posx=0,posy=0,posz=128;
int luminance=4;
int threshold=8;
int resolution=1;
int zoom=-3;
const int pi = 3.14159265358979;
int maparray[256][256];
int heightarray[256][256];
int pointlist[8];	//unused for non-allegro matrix functions
int t_pointlist[8];
int Min=0;
int Max=0;
bool quit=false;
long seed;
bool hires=false;
bool neednormalise=true;
bool needsmooth=true;

void game_timer()
{
	game_time++;
}
END_OF_FUNCTION(game_timer);

void resetarrays()
{
	int x,y;
	for(x=0;x<256;x++)
	{
		for(y=0;y<256;y++)
		{
			maparray[x][y]=0;
			heightarray[x][y]=0;
		}
	}
}

void setup_game()
{
	//start a new game
	srand( (unsigned)time( NULL ) );
	seed=rand()*3;
	clear(screen);
	resetarrays();
	//nice easy timer to keep the refresh ticking over nicely
	install_int(game_timer,30);
}

void play_game()
{
	//reset control flags
	int done=false;
	int draw=true;
	clear_to_color(gbuffer,colour_black);										
	setup_game();
	buildmap();

	clear_keybuf();
	do 
	{
		//loop to keep speed consistent
		//while(game_time) 
		{
			keyInput(); 
			game_time--;
		}
		menuDraw(); //background and platforms

		//now update the screen
		//this is the simple double buffer technique
		acquire_screen();
			blit(terrain,mbuffer ,0,0,32,16,terrain->w,terrain->h);
			blit(mbuffer,gbuffer,0,0,0,0,mbuffer->w,mbuffer->h);
			textprintf(gbuffer,font,10,gbuffer->h-20,colour_light,"FPS: %5d",fps);
			if(neednormalise && show3D)
				textprintf(gbuffer,font,200,0,colour_light,"You should [n] Re-Normalise Now");
			if(needsmooth && show3D)
				textprintf(gbuffer,font,450,0,colour_light,"You should [v] Smooth");
			blit(gbuffer, screen, 0, 0, 0, 0,gbuffer->w,gbuffer->h);
		release_screen();
		frame_count++; //update every drawn screen - should match fps which is required FPS or less, cannot be higher
	} while(!quit);
}

//draw crap menu
void menuDraw()
{
	//do backdrop first
	int yoffset=font_height;
	int ytop=200;
	int xoffset=1;

	clear_to_color(mbuffer,colour_red);

	textout(mbuffer,font,"TERRAIN Details",xoffset,ytop,colour_white);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*1,colour_light,	"M: New Map");
	if(show3D)
		textprintf(mbuffer,font,xoffset,ytop+yoffset*2,colour_light,"G: Toggle 3D     - On");
	else
		textprintf(mbuffer,font,xoffset,ytop+yoffset*2,colour_light,"G: toggle 3D     - Off");

	textprintf(mbuffer,font,xoffset,ytop+yoffset*3,colour_light,	"W: Water Level   - %d",levelwater);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*4,colour_light,	"S: Shallow Water - %d",levelwaterdepth);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*5,colour_light,	"B: Beach Height  - %d",levelbeach);

	textprintf(mbuffer,font,xoffset,ytop+yoffset*6,colour_light,	"N: Re-Normalise");
	textprintf(mbuffer,font,xoffset,ytop+yoffset*7,colour_light,	"V: Smoother");
	textprintf(mbuffer,font,xoffset,ytop+yoffset*8,colour_light,	"R: Roughen");
	
	textout(mbuffer,font,"Options",xoffset,ytop+yoffset*10,colour_white);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*11,colour_light,	"1: Reset Defaults");
	textprintf(mbuffer,font,xoffset,ytop+yoffset*12,colour_light,	"2: Reset 3D Levels");
	if(hires)
	textprintf(mbuffer,font,xoffset,ytop+yoffset*13,colour_light,	"3: Hi-Res 3D Movement On");
	else
	textprintf(mbuffer,font,xoffset,ytop+yoffset*13,colour_light,	"3: Hi-Res 3D Movement Off");
	if(showwork)
	textprintf(mbuffer,font,xoffset,ytop+yoffset*14,colour_light,	"X: Show Work     - %s","Yes");
	else
	textprintf(mbuffer,font,xoffset,ytop+yoffset*14,colour_light,	"X: Show Work     - %s","No");
	textprintf(mbuffer,font,xoffset,ytop+yoffset*16,colour_light,	"ESC Exits");

	textout(mbuffer,font,"3D View",xoffset,ytop+yoffset*18,colour_white);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*19,colour_light,	"F: Flatness      - %d",threshold);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*20,colour_light,	"L: Luminance     - %d",luminance);
	textout(mbuffer,font,"3D Positions",xoffset,ytop+yoffset*22,colour_white);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*23,colour_light,	"X                - %f",posx);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*24,colour_light,	"Y                - %f",posy);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*25,colour_light,	"Z                - %f",posz);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*26,colour_light,	"Zoom             - %d",zoom);
	textprintf(mbuffer,font,xoffset,ytop+yoffset*27,colour_light,	"Polygon Count    - %d",polycount);
	
	textout(mbuffer,font,"Keys",xoffset,ytop+yoffset*29,colour_white);
	textout(mbuffer,font,"2D: SHIFT reverses",xoffset,ytop+yoffset*30,colour_light);
	textout(mbuffer,font,"3D - ",xoffset,ytop+yoffset*32,colour_light);
	textout(mbuffer,font," Cursors:       Rotation",xoffset,ytop+yoffset*33,colour_light);
	textout(mbuffer,font," A/Z:           Rotation",xoffset,ytop+yoffset*34,colour_light);
	textout(mbuffer,font," SHIFT Cursors: Placement",xoffset,ytop+yoffset*35,colour_light);
	textout(mbuffer,font," SHIFT A/Z:     Zoom",xoffset,ytop+yoffset*36,colour_light);

	textout(mbuffer,font,"Neil Walker 2002",xoffset,ytop+yoffset*42,colour_white);
	textout(mbuffer,font,"neil@retrospec.org",xoffset,ytop+yoffset*43,colour_white);
}


//get keys to control map
bool keyInput()
{
	static int idle=0;
	bool redraw2D=true;
	bool redraw3D=true;
	refresh3D=true;

	if(keypressed())
	{
		switch(readkey() >> 8)
		{
			case KEY_ESC: 
				quit=true;
				redraw2D=false;
				redraw3D=false;
				break;
			case KEY_W: 
				if(key_shifts & KB_SHIFT_FLAG) levelwater-=2;
				else levelwater+=2; 
				break;
			case KEY_S: 
				if(key_shifts & KB_SHIFT_FLAG) levelwaterdepth-=2;
				else levelwaterdepth+=2; 
				break;
			case KEY_M: 
				//rebuild - this calls redraw
				redraw2D=false;
				redraw3D=false;
				neednormalise=true;
				needsmooth=true;
				buildmap();
				buildmap3D();
				//toggle standard/detailed
				if(leveldetail==42) leveldetail=63;
				else leveldetail=42;
				break;
			case KEY_B: 
				if(key_shifts & KB_SHIFT_FLAG) levelbeach-=2;
				else levelbeach+=2; 
				break;
			case KEY_X: 
				showwork=(!showwork); clear_keybuf(); redraw2D=false;redraw3D=false;break;
			case KEY_V:
				smoothmap(127,levelsmoothness);
				redraw2D=false;
				needsmooth=false;
				buildmap3D();
				break;
			case KEY_R:
				roughenmap();
				redraw2D=false;
				break;
			case KEY_1: 
				levelsmoothness=3;leveldetail=42;levelwater=96;
				levelwaterdepth=10;levelbeach=10;
				redraw2D=false;
				redraw3D=false;
				neednormalise=true;
				needsmooth=true;
				buildmap();
				buildmap3D();
				break;
			case KEY_N:
				smoothmap(127,0);
				normalisemap(127);
				redraw3D=false;
				neednormalise=false;
				buildmap3D();
				break;
			//3d
			case KEY_UP: 
				idle=0;
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) viewy-=4;
				else posx--; 
				redraw2D=false;
				break;
			case KEY_DOWN: 
				idle=0;
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) viewy+=4;
				else posx++;  
				redraw2D=false;
				break;
			case KEY_LEFT: 
				idle=0;
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) viewx-=4;
				else posy++;
				redraw2D=false;
				break;
			case KEY_RIGHT: 
				idle=0;
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) viewx+=4;
				else posy--;
				redraw2D=false;
				break;
			case KEY_3:
				redraw3D=false;
				redraw2D=false;
				hires=!hires;
				break;
			case KEY_A: 
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) zoom++;
				else posz++;
				redraw2D=false;
				break;
			case KEY_Z:
				if(!hires) resolution=2;
				if(key_shifts & KB_SHIFT_FLAG) zoom--;
				else posz--;  
				redraw2D=false;
				break;
			//3dview detail
			case KEY_L: 
				if(key_shifts & KB_SHIFT_FLAG) luminance--;
				else luminance++;
				redraw2D=false;
				break;
			case KEY_F: 
				if(key_shifts & KB_SHIFT_FLAG) {threshold--; }
				else {threshold++; }
				redraw2D=false;
				break;
			case KEY_2: luminance=4;threshold=10;posx=posy=0;posz=128;zoom=-3;
				redraw2D=false;
				break;
			case KEY_G:
				redraw3D=false;
				redraw2D=false;
				show3D=!show3D;
				if(show3D)
					buildmap3D();
				else
					clear_to_color(gbuffer,colour_black);
				break;
			default:
				redraw2D=redraw3D=false;
				break;
		}
		//clear buffer
		clear_keybuf();

		//update variables
		if(zoom>=0) zoom=-1;
		if(zoom<-30) zoom=-30;
		if(posy<0) posy=MAX_ANGLE;
		if(posx<0) posx=MAX_ANGLE;
		if(posx>MAX_ANGLE) posx=0;
		if(posy>MAX_ANGLE) posy=0;
		if(posz<0) posz=MAX_ANGLE;
		if(posz>MAX_ANGLE) posz=0;
		if(threshold>60) threshold=1;
		if(threshold<1) threshold=60;
		if(luminance>15) luminance=1;
		if(luminance<1) luminance=15;
		if(levelwater>255) levelwater=0;
		if(levelwater<0) levelwater=255;
		if(levelwaterdepth>255) levelwaterdepth=0;
		if(levelwaterdepth<0) levelwaterdepth=255;
		if(levelbeach>255) levelbeach=0;
		if(levelbeach<0) levelbeach=255;
		//redraw
		if(redraw2D) mapdraw();
		if(redraw3D) drawmap3D();

	}
	else
	{
		//if nothing pressed then increase idle time
		//when reach a number, redraw the map
		//if we are in low res movement mode (default)
		idle++;
		if(resolution!=1 && idle>30) 
		{
			resolution=1;
			idle=0;
			drawmap3D();
		}
	}

	return true;
}

//generate small terrain map
void buildmap(bool reseed)
{
	bool ret=false;
	int x;
	int y;

	//level of detail toggles between two values - one highish, one lowish
	for(y=0;y<=leveldetail;y++)
	{
		for(x=0;x<=leveldetail;x++)
		{
			//store random number in terrain
			//keep the same values
			seed = (seed * 10421 + 1) % 65536;
			maparray[x][y]=seed/256;
		}
	}

	//once random numbers generated, draw it
	clear_to_color(terrain,colour_black);
	smoothmap(leveldetail,levelsmoothness);
	normalisemap(leveldetail);
	stretchmap();
	mapdraw();
}

void roughenmap()
{
	//add or takeaway one from each value
	int x,y;
	for(y=0;y<=127;y++)
	{
		for(x=0;x<=127;x++)
		{
			maparray[x][y]=maparray[x][y]+(2*(rand()%2)-1);
			if(maparray[x][y]>255) maparray[x][y]=255;
			if(maparray[x][y]<0) maparray[x][y]=0;
		}
	}
	mapdraw();
}


void smoothmap(int ldet, int lsmoo)
{
	//smooth map - remove jaggedness
	//used fixed smoothness
	//repeated calls smoothens the smooth - more so
	int i,x,y,temp,temp2;

	Min=255;
	Max=0;
	for(i=0;i<=lsmoo;i++)
	{
		Min=255;
		Max=0;
		//loop each map item
		//get the surrounding values
		//use next value if more than the threshold passed in
		//i.e. blend the pixels
		for(y=0;y<=ldet;y++)
		{
			for(x=0;x<=ldet;x++)
			{
				temp=temp2=0;
				if(x>0)
				{
					temp+=maparray[x-1][y];
					temp2++;
				}
				if(x<ldet)
				{
					temp+=maparray[x+1][y];
					temp2++;
				}
				if(y>0)
				{
					temp+=maparray[x][y-1];
					temp2++;
				}
				if(y<ldet)
				{
					temp+=maparray[x][y+1];
					temp2++;
				}
				if(temp2==0) temp2=1;
				temp/=temp2;
				maparray[x][y]=temp;
				if(temp>Max) Max=temp;
				if(temp<Min) Min=temp;

				if(showwork)
				{
					putpixel(terrain, x,y,makecol(temp,temp,temp));
					blit(terrain,screen,0,0,32,64,terrain->w,terrain->h);
				}
			}
		}
	}
}

void normalisemap(int ldet)
{
	//normalize values to 0 and 255 (for full range of height)
	//build evenly distributed highs, etc.
	int x,y,temp;
	for(y=0;y<=ldet;y++)
	{
		for(x=0;x<=ldet;x++)
		{
			temp=maparray[x][y];
			temp-=Min;
			if(Max==Min) Max++;
			if(Max<Min) Max+=2;
			if(Max>255) Max=255;
			temp=(temp*255)/(Max-Min);
			maparray[x][y]=temp;

			//update small terrain model if set to show
			if(showwork)
			{
				putpixel(terrain, x,y,makecol(temp,temp,temp));
				blit(terrain,screen,0,0,32,64,terrain->w,terrain->h);
			}
		}
	}
}

void stretchmap()
{
	//make the map full size
	int temparray[128][128];
	int blocksize,cx,cy,x,y;
	int temp;
	for(y=0;y<=127;y++)
	{
		for(x=0;x<=127;x++)
		{
			temparray[x][y]=maparray[x][y];
			if(showwork)
			{
				temp=temparray[x][y];
				temp=makecol(temp,temp,temp);
				putpixel(terrain, x,y,makecol(temp,temp,temp));
				blit(terrain,screen,0,0,32,64,terrain->w,terrain->h);
			}
		}
	}


	//stretch to 128 x 128
	blocksize = 128 / leveldetail; //get blocksize from detail
	for(y=0;y<=leveldetail;y++)
	{
		for(x=0;x<=leveldetail;x++)
		{
			for(cy=0;cy<=blocksize;cy++)
			{
				for(cx=0;cx<=blocksize;cx++)
					maparray[x*blocksize+cx][y*blocksize+cy]=temparray[x][y];
			}
		}
	}
}

void mapdraw()
{
	//draw the map
	int x,y,temp;

	for(y=0;y<128;y++)
	{
		for(x=0;x<128;x++)
		{
			//land as higher than water and beach
			temp=maparray[x][y];
			if(temp>(levelwater+levelbeach)) 
				putpixel(terrain, x,y, makecol(0,temp,0));
			else
			{
				//water
				if(temp<=levelwater)
				{
					//deep water or in the shallows
					if((levelwater-maparray[x][y])<levelwaterdepth)
						putpixel(terrain, x,y, makecol(0,64,127));	//light blue
					//else((levelwater-maparray[x][y])>levelwaterdepth)
					else
						putpixel(terrain, x,y, makecol(0,0,127));	//blue
				}
				else
					//must be beach
					putpixel(terrain, x,y, makecol(temp,temp,temp/2));	//brown
			}
		}
	}
}


void buildmap3D()
{
	int x,y;
	for(y=0;y<=127;y++)
	{
		for(x=0;x<=127;x++)
		{
			//build array from the terrain map
			heightarray[x][y] = maparray[x][y];
			if(heightarray[x][y]<=levelwater) heightarray[x][y]=levelwater;
		}
	}
	drawmap3D();
}


void drawmap3D()
{
	//draw the 3D model
	float i;
	float j;
	float k;
	//TDPoint XPoint;
	int light,resblock;
	int lmin=-64;
	int lmax=63;
	int x,y;
	int overflowlight;
	int currentcolour;
	polycount=0;
	if(!refresh3D || !show3D) return;
	clear_to_color(gbuffer,colour_black);

	//size of polygon - smaller more polygons
	if(resolution==1) resblock=1;
	if(resolution==2) resblock=4;


//build the 3D rotation matrix
	//matrixbuild(posx*pi/180,posy*pi/180,posz*pi/180);		//original pre-calculated
	get_rotation_matrix_f(&terrainmatrix, posx,posy,posz);	//allegro
	set_projection_viewport(mbuffer->w, 0, gbuffer->w-mbuffer->w, gbuffer->h);

//we start at -64 so that the middle of the map is at 0,0
//this way the map rotates around the middle and not the upper-left corner
	for(y=lmin;y<=(lmax-resblock);y+=resblock)
	{
		for(x=lmin;x<=(lmax-resblock);x+=resblock)
		{
			//rotate each vertex
			//x and y just simply the locations
			//z is the height of each item
			//threshold limits the height - set by user
			//rotate
			apply_matrix_f(&terrainmatrix,x,y,(-heightarray[x+lmax+1][y+lmax+1])/threshold, &i,&j,&k);
			//persp_project_f(i,j,k, &t_pointlist[0],&t_pointlist[1]);
			t_pointlist[0]=i*zoom+viewx;
			t_pointlist[1]=j*zoom+viewy;

			//old way
			//XPoint=rotatepoint(x,y,(-heightarray[x+lmax+1][y+lmax+1])/threshold);
			//pointlist[0]=XPoint.x*zoom+viewx;
			//pointlist[1]=XPoint.y*zoom+viewy;

			apply_matrix_f(&terrainmatrix,x+resblock,y,(-heightarray[x+lmax+1+resblock][y+lmax+1])/threshold, &i,&j,&k);
			//persp_project_f(i,j,k, &t_pointlist[2],&t_pointlist[3]);
			t_pointlist[2]=i*zoom+viewx;
			t_pointlist[3]=j*zoom+viewy;
			//old way
			//XPoint=rotatepoint(x+resblock,y,(-heightarray[x+lmax+1+resblock][y+lmax+1])/threshold);
			//pointlist[2]=XPoint.x*zoom+viewx;
			//pointlist[3]=XPoint.y*zoom+viewy;

			apply_matrix_f(&terrainmatrix,x+resblock,y+resblock,(-heightarray[x+lmax+1+resblock][y+lmax+1+resblock])/threshold, &i,&j,&k);
			//persp_project_f(i,j,k, &t_pointlist[4],&t_pointlist[5]);
			t_pointlist[4]=i*zoom+viewx;
			t_pointlist[5]=j*zoom+viewy;
			//old way
			//XPoint=rotatepoint(x+resblock,y+resblock,(-heightarray[x+lmax+1+resblock][y+lmax+1+resblock])/threshold);
			//pointlist[4]=XPoint.x*zoom+viewx;
			//pointlist[5]=XPoint.y*zoom+viewy;

			apply_matrix_f(&terrainmatrix,x,y+resblock,(-heightarray[x+lmax+1][y+lmax+1+resblock])/threshold, &i,&j,&k);
			//persp_project_f(i,j,k, &t_pointlist[6],&t_pointlist[7]);
			t_pointlist[6]=i*zoom+viewx;
			t_pointlist[7]=j*zoom+viewy;
			//old way
			//XPoint=rotatepoint(x,y+resblock,(-heightarray[x+lmax+1][y+lmax+1+resblock])/threshold);
			//pointlist[6]=XPoint.x*zoom+viewx;
			//pointlist[7]=XPoint.y*zoom+viewy;
			polycount++;

			//a crude shading technique
			//basically it's saying that if the left side is higher than the right side of
			//the polygon then the polygon is tilted away from the source, and make it a negative
			//correction factor.  Otherwise, it's positive, so the polygon is brighter
			overflowlight = 0;
			light = (heightarray[x + lmax+1][y+lmax+1] - heightarray[x + lmax+2][y+lmax+2]) * luminance + 127;
			if(light<0) light=0;
			if(light>255) 
			{
				light=0;
				overflowlight = light - 255; //catch extra light for fading to white
				light = 255;
			}

			//draw for land
			if(heightarray[x+lmax+1][y+lmax+1]>levelwater+levelbeach)
			{
				//land
				currentcolour=makecol(overflowlight,light,overflowlight);	//above beach
			}
			else 
			{
				if(heightarray[x+lmax+1][y+lmax+1]<=levelwater)	//under water
				{
					//deep water
					if(levelwater-maparray[x+lmax+1][y+lmax+1]>levelwaterdepth)
					{
						currentcolour=makecol(0,0,127);
					}
					else
					{
						//shallows - multiple shades
						currentcolour=makecol(0,levelwaterdepth-(levelwater-maparray[x+lmax+1][y+lmax+1]),127);
						//simpler shallows - just one shade
						//currentcolour=makecol(0,64,127);
					}
				}
				else
				{
					//beach
					currentcolour=makecol(light,light,light/2);
				}
			}

			polygon(gbuffer, 4, t_pointlist, currentcolour);

			//clipping - unused
			/*if(XPoint.z*zoom > -500)
			{
				polygon(gbuffer, 4, pointlist, currentcolour);
			}*/
		}
	}
}


//old matrix function - not used as allegro does this
void matrixbuild(double x, double y, double z)
{
	double sinx;
	double cosx;
	double siny;
	double cosy;
	double sinz;
	double cosz;

	sinx=sin(x);
	cosx=cos(x);
	siny=sin(y);
	cosy=cos(y);
	sinz=sin(z);
	cosz=cos(z);

	//fill out the rotation matrix.  Now we can multiply any point by this matrix
	rotmatrix[0][0]=(cosz*cosy);
	rotmatrix[0][1]=(cosz * (-siny) * -sinx + sinz*cosx);
	rotmatrix[0][2]=(cosz * (-siny) * cosx + sinz*sinx);
	rotmatrix[1][0]=((-sinz)*cosy);
	rotmatrix[1][1]=((-sinz)*(-siny)*(-sinx)+cosz*cosx);
	rotmatrix[1][2]=((-sinz)*(-siny)*cosx+cosz*sinx);
	rotmatrix[2][0]=(siny);
	rotmatrix[2][1]=(cosy*(-sinx));
	rotmatrix[2][2]=(cosy*cosx);
}

//old matrix function - not used as allegro does this
TDPoint rotatepoint(double x,double y, double z)
{
	//rotate using the trig matrix created
	TDPoint ZPoint;
	ZPoint.x=(x*rotmatrix[0][0]) + (y*rotmatrix[0][1]) + (z*rotmatrix[0][2]);
	ZPoint.y=(x*rotmatrix[1][0]) + (y*rotmatrix[1][1]) + (z*rotmatrix[1][2]);
	ZPoint.z=(x*rotmatrix[2][0]) + (y*rotmatrix[2][1]) + (z*rotmatrix[2][2]);

	return ZPoint;
}

