#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

//----------------------------------------------------------------------------

Train::Train( int train_id, bool is_freight_train, int max_speed_10th_tiles_per_hour, int station_time_10th_of_hour, int cur_game_time )
: train_id_(train_id)
, is_stopped_at_station_(false)
, n_boxes_(0)
, n_passengers_(0)
, last_visited_station_id_(-1)
, is_freight_train_(is_freight_train)
, cur_speed_10th_tiles_per_hour_(max_speed_10th_tiles_per_hour)
, max_speed_10th_tiles_per_hour_(max_speed_10th_tiles_per_hour)
, station_time_10th_of_hour_(station_time_10th_of_hour)
, last_moved_(cur_game_time)
{
  pos_[0].clear();
  pos_[1].clear();
  pos_[2].clear();
}

void Train::reset_game_time()
{
  pos_[0].clear();
  pos_[1].clear();
  pos_[2].clear();
  cost_rev_.clear();
  is_stopped_at_station_=false;
  n_boxes_ = 0;
  n_passengers_ = 0;
  last_visited_station_id_ = -1;

  cur_speed_10th_tiles_per_hour_=0;
  max_speed_10th_tiles_per_hour_=0;
  last_moved_=0;
}

inline speed_to_duration( int speed_10th_tiles_per_hour )
{
  if( speed_10th_tiles_per_hour <= 0 )
    return 999;
  return (GameClock::ticks_per_game_hour*10)/speed_10th_tiles_per_hour;
}

inline station_time_to_ticks( int station_time )
{
  return (GameClock::ticks_per_game_hour*station_time)/10;
}

bool Train::needs_update( int cur_game_time ) const
{
  if( is_stopped_at_station_ )
    return cur_game_time - last_moved_ >= station_time_to_ticks( station_time_10th_of_hour_ );
  else
    return cur_game_time - last_moved_ >= speed_to_duration( cur_speed_10th_tiles_per_hour_ );
}

void Train::reduce_speed_entering_station( int parts_of_train_in_station )
{
  switch( parts_of_train_in_station )
  {
  case 1:
    cur_speed_10th_tiles_per_hour_ = max_speed_10th_tiles_per_hour_ * 0.66;
    break;
  case 2:
    cur_speed_10th_tiles_per_hour_ = max_speed_10th_tiles_per_hour_ * 0.33;
    break;
  }
}

void Train::update_time( int cur_game_time )
{
  if( is_stopped_at_station_ )
    last_moved_ += station_time_to_ticks( station_time_10th_of_hour_ );
  else
    last_moved_ += speed_to_duration( cur_speed_10th_tiles_per_hour_ );
}

void Train::increase_speed_leaving_station()
{
  if( is_stopped_at_station_ )
  {
    is_stopped_at_station_ = false;
    cur_speed_10th_tiles_per_hour_ = max_speed_10th_tiles_per_hour_ / 5;
  }
  else if( cur_speed_10th_tiles_per_hour_ < max_speed_10th_tiles_per_hour_ )
  {
    cur_speed_10th_tiles_per_hour_ += max_speed_10th_tiles_per_hour_ / 5;
    if( cur_speed_10th_tiles_per_hour_ > max_speed_10th_tiles_per_hour_ )
      cur_speed_10th_tiles_per_hour_ = max_speed_10th_tiles_per_hour_;
  }
}

void Train::reverse()
{
  const TileKOO tmp = pos_[0];
  pos_[0] = pos_[2];
  pos_[2] = tmp;
}

TrainShow Train::get_train_show( int i ) const
{
  if( is_freight_train_ )
  {
    switch(i)
    {
    case 0:
      return FreightLok;
      break;
    case 1:
      return n_boxes_>=2 ? Freight2 : n_boxes_==1 ? Freight1 : Freight0;
      break;
    case 2:
      return n_boxes_>=4 ? Freight2 : n_boxes_==3 ? Freight1 : Freight0;
      break;
    }
  }
  else
  {
    switch(i)
    {
    case 0:
      return PersLok;
      break;
    case 1:
    case 2:
      return PersWaggon;
      break;
    }
  }
  return NoTrain;
}











//----------------------------------------------------------------------------

ExtTrain::ExtTrain( int p_max_speed, int p_station_time, int f_max_speed, int f_station_time )
: Train( -1, true, f_max_speed, f_station_time, 0 )
, is_on_map_(false)
, is_importing_(true)
, p_max_speed_10th_tiles_per_hour_(p_max_speed)
, f_max_speed_10th_tiles_per_hour_(f_max_speed)
, p_station_time_10th_of_hour_(p_station_time)
, f_station_time_10th_of_hour_(f_station_time)
{
}

void ExtTrain::switch_train()
{
  is_freight_train_ = !is_freight_train_;
  n_boxes_ = is_importing_ ? 4 : 0;
  max_speed_10th_tiles_per_hour_ = is_freight_train_ ? f_max_speed_10th_tiles_per_hour_ : p_max_speed_10th_tiles_per_hour_;
  cur_speed_10th_tiles_per_hour_ = max_speed_10th_tiles_per_hour_;
  station_time_10th_of_hour_ = is_freight_train_ ? f_station_time_10th_of_hour_ : p_station_time_10th_of_hour_;
}

//----------------------------------------------------------------------------

void ExtTrain::reset_game_time()
{
  Train::reset_game_time();
  is_freight_train_ = true;
  is_on_map_=false;
  is_importing_ = true;
  cur_speed_10th_tiles_per_hour_ = 0;
  max_speed_10th_tiles_per_hour_ = 0;
}














//----------------------------------------------------------------------------

int WorldMap::place_train_find_pos( const TileKOO&t1, bool train_reversed, bool is_freight_train, Outline&train_shape ) const
{
  if( !is_inside_map(t1) )
    return 0;
  train_shape.clear();
  const MapTile&mt1 = map_tiles_[ index(t1) ];
  if( !mt1.player_is_owner_ || !mt1.is_rail() )
    return 0;
  assertCondition( mt1.player_is_owner_ && mt1.is_rail() );
  std::vector<TileKOO> nb_tiles;
  const int n_nb = find_connecting_rail_neighbours( t1, nb_tiles );
  if( n_nb!=2 )
    return 0;
  const TileKOO&t0 = nb_tiles[0];
  const MapTile&mt0 = map_tiles_[ index(t0) ];
  assertCondition( mt0.player_is_owner_ && mt0.is_rail() );
  const TileKOO&t2 = nb_tiles[1];
  const MapTile&mt2 = map_tiles_[ index(t2) ];
  assertCondition( mt2.player_is_owner_ && mt2.is_rail() );
  const TrainShow lok_show = is_freight_train ? FreightLok : PersLok;
  const TrainShow wag_show = is_freight_train ? Freight0 : PersWaggon;
  train_shape.push_back( OutlineDesc( train_reversed ? t2 : t0, &get_train_overlay( mt0.tcl_idx_, lok_show ) ) );
  train_shape.push_back( OutlineDesc( t1,                       &get_train_overlay( mt1.tcl_idx_, wag_show ) ) );
  train_shape.push_back( OutlineDesc( train_reversed ? t0 : t2, &get_train_overlay( mt2.tcl_idx_, wag_show ) ) );
  if( cash_.player_total_ < def_values_.cost_place_train )
    return 0;
  return def_values_.cost_place_train;
}

void WorldMap::place_freight_train( const Outline&train_shape )
{
  const int train_id = find_empty_slot( v_trains_ );
  Train train( train_id, true, def_values_.f_speed_tenth_tiles_per_hour, def_values_.f_station_time_tenth_of_hour, game_clock_.get_time_since0() );
  train.pos_[0] = train_shape[0].t_;
  train.pos_[1] = train_shape[1].t_;
  train.pos_[2] = train_shape[2].t_;
  show_train( train.pos_[0], train.get_train_show(0), train_id, true );
  show_train( train.pos_[1], train.get_train_show(1), train_id, true );
  show_train( train.pos_[2], train.get_train_show(2), train_id, true );
  if( train_id >= v_trains_.size() )
    v_trains_.push_back( train );
  else
    v_trains_[train_id] = train;
  cash_.trains_re_.push_cost( def_values_.cost_place_train );
  cash_.total_.push_cost( def_values_.cost_place_train );
  cash_.player_total_ -= def_values_.cost_place_train;
}

void WorldMap::place_pers_train( const Outline&train_shape )
{
  const int train_id = find_empty_slot( v_trains_ );
  Train train( train_id, false, def_values_.p_speed_tenth_tiles_per_hour, def_values_.p_station_time_tenth_of_hour, game_clock_.get_time_since0() );
  train.pos_[0] = train_shape[0].t_;
  train.pos_[1] = train_shape[1].t_;
  train.pos_[2] = train_shape[2].t_;
  show_train( train.pos_[0], train.get_train_show(0), train_id, false );
  show_train( train.pos_[1], train.get_train_show(1), train_id, false );
  show_train( train.pos_[2], train.get_train_show(2), train_id, false );
  if( train_id >= v_trains_.size() )
    v_trains_.push_back( train );
  else
    v_trains_[train_id] = train;
  cash_.trains_re_.push_cost( def_values_.cost_place_train );
  cash_.total_.push_cost( def_values_.cost_place_train );
  cash_.player_total_ -= def_values_.cost_place_train;
}

//----------------------------------------------------------------------------

void WorldMap::remove_train( size_t train_id )
{
  assertCondition( train_id < v_trains_.size() );
  Train&train = v_trains_[train_id];
  assertCondition( train.is_valid() );
  for( int ip=0; ip<3; ++ip )
  {
    show_train( train.pos_[ip], NoTrain, -1, false );
  }
  cash_.trains_re_.push_revenue( def_values_.revenue_remove_train );
  cash_.total_.push_revenue( def_values_.revenue_remove_train );
  cash_.player_total_ += def_values_.revenue_remove_train;
  train.reset_game_time();
  train.clear();
}






















//----------------------------------------------------------------------------

void WorldMap::show_train( const TileKOO&t, TrainShow show, int train_id, bool is_f_train )
{
  MapTile&mt0 = map_tiles_[ index(t) ];
  assertCondition( mt0.is_rail() );
  mt0.train_show_ = show;
  mt0.train_id_ = train_id;
  mt0.is_train_f_train_ = is_f_train;
  if( show == NoTrain )
  {
    assertCondition( train_id < 0 );
  }
}

bool WorldMap::is_train_complete_off_map( const Train&train ) const
{
  return !is_on_map(train.pos_[0])
    &&   !is_on_map(train.pos_[1])
    &&   !is_on_map(train.pos_[2])
    ;
}

bool WorldMap::is_train_complete_on_map( const Train&train ) const
{
  return is_on_map(train.pos_[0])
    &&   is_on_map(train.pos_[1])
    &&   is_on_map(train.pos_[2])
    ;
}

int WorldMap::is_train_complete_in_station( const Train&train ) const
{
  const int station_id = map_tiles_[ index(train.pos_[0]) ].station_id_;
  if(  station_id >= 0
    && map_tiles_[ index(train.pos_[1]) ].station_id_ == station_id
    && map_tiles_[ index(train.pos_[2]) ].station_id_ == station_id
    )
    return station_id;
  return -1;
}

int WorldMap::count_train_parts_in_station( const Train&train ) const
{
  const int station_id = map_tiles_[ index(train.pos_[0]) ].station_id_;
  if( station_id < 0 )
    return 0;
  if( map_tiles_[ index(train.pos_[2]) ].station_id_ == station_id )
    return 3;
  if( map_tiles_[ index(train.pos_[1]) ].station_id_ == station_id )
    return 2;
  return 1;
}

//----------------------------------------------------------------------------

void WorldMap::advance_ext_train_to_next_tile( ExtTrain&train )
{
  const TileKOO new_pos = find_next_rail_tile( train.pos_ );
  if( is_on_map(train.pos_[2]) )
    show_train( train.pos_[2], NoTrain, -1, false );
  train.pos_[2] = train.pos_[1];
  train.pos_[1] = train.pos_[0];
  train.pos_[0] = new_pos;
}

void WorldMap::advance_train_to_next_tile( Train&train, bool allow_reverse )
{
  const TileKOO new_pos = find_next_rail_tile( train.pos_ );
  if( is_on_map(new_pos) )
  {
    const MapTile&mt = map_tiles_[ index(new_pos) ];
    if( mt.is_rail() && mt.train_show_==NoTrain )
    {
      if( is_on_map(train.pos_[2]) )
        show_train( train.pos_[2], NoTrain, -1, false );
      train.pos_[2] = train.pos_[1];
      train.pos_[1] = train.pos_[0];
      train.pos_[0] = new_pos;
      return;
    }
    if( mt.is_rail() && mt.train_show_!=PersLok && mt.train_show_!=FreightLok )
      return;
  }
  // couldn't advance, have to reverse
  if( allow_reverse )
  {
    train.reverse();
    advance_train_to_next_tile( train, false );
  }
}

//----------------------------------------------------------------------------

void WorldMap::reverse_train( bool is_freight_train, int train_id )
{
  Train&train = v_trains_[train_id];
  train.reverse();
}

//----------------------------------------------------------------------------

int WorldMap::advance_ext_train( int cur_game_time )
{
  if( ext_train_.needs_update(cur_game_time) )
  {
    if( is_on_map( ext_train_.pos_[0] ) )
    {
      const MapTile&mt0 = map_tiles_[ index( ext_train_.pos_[0] ) ];
      if( mt0.station_id_ >= 0 )
        ext_train_.reduce_speed_entering_station( count_train_parts_in_station(ext_train_) );
    }
    ext_train_.update_time( cur_game_time );
    if( !ext_train_.is_on_map_ )
    {
      ext_train_.switch_train();
      const TileKOO&entry_koo = ext_entry_;
      MapTile&entry_tile = map_tiles_[ index(entry_koo) ];
      assertCondition( entry_tile.is_rail() );
      entry_tile.train_show_ = ext_train_.is_freight_train() ? FreightLok : PersLok;
      ext_train_.pos_[0] = entry_koo;
      ext_train_.is_on_map_ = true;
    }
    else
    {
      ext_train_.increase_speed_leaving_station();
      advance_ext_train_to_next_tile(ext_train_);
      if( is_train_complete_off_map(ext_train_) )
      {
        ext_train_.is_on_map_ = false;
        if( ext_train_.is_freight_train() )
          ext_train_.is_importing_ = ext_train_.n_boxes_==0;
      }
      if( is_train_complete_on_map(ext_train_) && is_train_complete_in_station(ext_train_) >= 0 )
      {
        ext_train_.is_stopped_at_station_ = true;
        if( ext_train_.is_freight_train() )
        {
          if( ext_train_.n_boxes_ > 0 )
            unload_all_boxes_from_train( ext_train_.pos_[1], ext_train_ );
          else
            load_all_boxes_to_train( ext_train_.pos_[1], ext_train_ );
        }
      }
    }
    if( is_on_map( ext_train_.pos_[0] ) )
      show_train( ext_train_.pos_[0], ext_train_.get_train_show(0), -1, ext_train_.is_freight_train() );
    if( is_on_map( ext_train_.pos_[1] ) )
      show_train( ext_train_.pos_[1], ext_train_.get_train_show(1), -1, ext_train_.is_freight_train() );
    if( is_on_map( ext_train_.pos_[2] ) )
      show_train( ext_train_.pos_[2], ext_train_.get_train_show(2), -1, ext_train_.is_freight_train() );
    return 1;
  }
  return 0;
}

int WorldMap::advance_f_train( Train&train, int cur_game_time )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_freight_train() );
  if( train.needs_update( cur_game_time ) )
  {
    const MapTile&mt0 = map_tiles_[ index( train.pos_[0] ) ];
    if( mt0.station_id_ > 0 && does_train_stop_at_station(train.train_id_,mt0.station_id_) )
      train.reduce_speed_entering_station( count_train_parts_in_station(train) );
    train.update_time( cur_game_time );
    train.increase_speed_leaving_station();
    advance_train_to_next_tile( train, true );
    const int station_id = is_train_complete_in_station(train);
    if( station_id >= 0 && does_train_stop_at_station(train.train_id_,station_id) )
    {
      train.is_stopped_at_station_ = true;
      if( train.n_boxes_ > 0 )
        unload_all_boxes_from_train( train.pos_[1], train );
      else
        load_all_boxes_to_train( train.pos_[1], train );
    }
    show_train( train.pos_[0], train.get_train_show(0), train.train_id_, true );
    show_train( train.pos_[1], train.get_train_show(1), train.train_id_, true );
    show_train( train.pos_[2], train.get_train_show(2), train.train_id_, true );
    return 1;
  }
  return 0;
}

int WorldMap::advance_p_train( Train&train, int cur_game_time )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_passenger_train() );
  if( train.needs_update( cur_game_time ) )
  {
    const MapTile&mt0 = map_tiles_[ index( train.pos_[0] ) ];
    if( mt0.station_id_ > 0 && does_train_stop_at_station(train.train_id_,mt0.station_id_) )
      train.reduce_speed_entering_station( count_train_parts_in_station(train) );
    train.update_time( cur_game_time );
    train.increase_speed_leaving_station();
    if( 2*train.cur_speed_10th_tiles_per_hour_ > train.max_speed_10th_tiles_per_hour_ )
    {
      const MapTile&mt1 = map_tiles_[ index( train.pos_[1] ) ];
      if(  mt0.tcl_idx_ > IDXrail_straight
        || mt1.tcl_idx_ > IDXrail_straight
        )
        train.cur_speed_10th_tiles_per_hour_ = train.max_speed_10th_tiles_per_hour_ / 2;
    }
    advance_train_to_next_tile( train, true );
    const int station_id = is_train_complete_in_station(train);
    if( station_id >= 0 && does_train_stop_at_station(train.train_id_,station_id) )
    {
      train.is_stopped_at_station_ = true;
      const MapTile&mt0 = map_tiles_[ index(train.pos_[0]) ];
      assertCondition( mt0.station_id_ >= 0 );
      assertCondition( train.n_passengers_ <= def_values_.max_passengers_train_ );
      Station&station = v_stations_[ mt0.station_id_ ];
      if( train.last_visited_station_id_ != station.station_id_ )
      {
        const int n_pass_waiting = station.passenger_queue_.peek_sum();
        const int pass_counted_from_train = MIN( n_pass_waiting, train.n_passengers_ );
        // count arriving passengers
        station.n_good_passengers_ += pass_counted_from_train;
        train.n_passengers_ -= pass_counted_from_train;
        train.n_passengers_ /= 5;
        const int new_passengers = station.passenger_queue_.pop_value_distributed( def_values_.max_passengers_train_ - train.n_passengers_ );
        train.n_passengers_ += new_passengers;
        if( train.n_passengers_ > def_values_.max_passengers_train_ )
          train.n_passengers_ = def_values_.max_passengers_train_;
        // revenues
        if( train.last_visited_station_id_ >= 0 )
        {
          const Station&last_visited_station = v_stations_[ train.last_visited_station_id_ ];
          const double train_rev_per_10passengers = def_values_.revenue_train_fix_per_10passengers_
            + ( def_values_.revenue_train_var_per_10passengers_per_10tiles_ * distance( station.center_tile_, last_visited_station.center_tile_ ) ) / 10;
          const int train_revenue = ( new_passengers * train_rev_per_10passengers ) / 10;
          train.cost_rev_.push_revenue( train_revenue );
          cash_.trains_op_.push_revenue( train_revenue );
          cash_.total_.push_revenue( train_revenue );
          cash_.player_total_ += train_revenue;
        }
      }
      train.last_visited_station_id_ = station.station_id_;
    }
    show_train( train.pos_[0], train.get_train_show(0), train.train_id_, false );
    show_train( train.pos_[1], train.get_train_show(1), train.train_id_, false );
    show_train( train.pos_[2], train.get_train_show(2), train.train_id_, false );
    return 1;
  }
  return 0;
}

//----------------------------------------------------------------------------
