#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

#ifndef INC_MTRAND
#include "../gfwk/src/mtrand.h"
#endif

//----------------------------------------------------------------------------

namespace
{

char s_buffer[20] = "123";
char w_buffer[20] = "40";
char h_buffer[20] = "40";

DIALOG map_dlg[] =
{
  /* (proc)        (x)  (y)  (w)  (h)  (fg) (bg)   (key) (flags) (d1) (d2) (dp)      (dp2) (dp3) */
  { d_box_proc,    20,  20,  224, 152, 0,   65504, 0,    0,      0,   0,   NULL,     NULL, NULL },
  { d_rtext_proc,  72,  56,  56,  16,  0,   65504, 0,    0,      4,   0,   "Seed",   NULL, NULL },
  { d_edit_proc,   136, 56,  56,  16,  0,   18400, 0,    0,      4,   0,   s_buffer, NULL, NULL },
  { d_rtext_proc,  72,  80,  56,  16,  0,   65504, 0,    0,      4,   0,   "Width",  NULL, NULL },
  { d_edit_proc,   136, 80,  56,  16,  0,   18400, 0,    0,      4,   0,   w_buffer, NULL, NULL },
  { d_rtext_proc,  72,  104, 56,  16,  0,   65504, 0,    0,      4,   0,   "Height", NULL, NULL },
  { d_edit_proc,   136, 104, 56,  16,  0,   18400, 0,    0,      4,   0,   h_buffer, NULL, NULL },
  { d_button_proc, 48,  120, 80,  24,  0,   18400, 0,    D_EXIT, 0,   0,   "OK",     NULL, NULL },
  { d_button_proc, 136, 120, 80,  24,  0,   18400, 0,    D_EXIT, 0,   0,   "Cancel", NULL, NULL },
  { d_yield_proc,  40,  40,  8,   8,   0,   0,     0,    0,      0,   0,   NULL,     NULL, NULL },
  { NULL,          0,   0,   0,   0,   0,   0,     0,    0,      0,   0,   NULL,     NULL, NULL }
};

#define default_savefilename "tntrain.tnt"
const char*const autosave_filename = "autosave.tnt";

} // namespace

//----------------------------------------------------------------------------

void WorldMap::generate_random_map( unsigned int seed, size_t aw, size_t ah )
{
  map_tiles_.clear();
  map_tiles_.resize( ah*aw );
  map_x_ext_ = aw;
  map_y_ext_ = ah;
  rng.Seed(seed);
  for( size_t iy=0; iy<ah; ++iy )
    for( size_t ix=0; ix<aw; ++ix )
    {
      const TileKOO tij( ix, iy );
      const size_t i = index( tij );
      map_tiles_[i] = MapTile( 0, IDXempty, 0, 0 );
      const int rd = rng.IRand(100);
      const TileClassIDX tcl_idx = rd<96 ? IDXgras : rd<99 ? IDXsmall_house : IDXwater;
      set_general_tile( tij, tcl_idx, SUB_IDX_RANDOM, true, false );
    }
    ext_entry_ = TileKOO(0,5);
    {
      for( int i=0; i<13; ++i )
      {
        set_general_tile( TileKOO(i,5), IDXrail_nwso, SUB_IDX_RANDOM, false, false );
      }
    }
    {
      set_general_tile( TileKOO(13,5), IDXrail_oonw, SUB_IDX_RANDOM, false, false );
    }
    {
      for( int i=4; i>=2; --i )
      {
        set_general_tile( TileKOO(18-i,i), IDXrail_wwoo, SUB_IDX_RANDOM, false, false );
      }
    }
    {
      set_general_tile( TileKOO(17,1), IDXrail_noww, SUB_IDX_RANDOM, false, false );
      set_general_tile( TileKOO(17,0), IDXrail_nosw, SUB_IDX_RANDOM, false, false );
    }
    {
      for( int i=10; i<14; ++i )
      {
        set_box_tile( TileKOO(i,6), false, 0 );
        set_box_tile( TileKOO(i,7), false, 0 );
      }
      for( int i=14; i<18; ++i )
      {
        set_box_tile( TileKOO(i,5), false, 0 );
        set_box_tile( TileKOO(i,6), false, 0 );
        set_box_tile( TileKOO(i,7), false, 0 );
      }
    }
    build_station( TileKOO(10, 6), MTIDXstation_dn_nwso, false );
}

//----------------------------------------------------------------------------

void WorldMap::new_game()
{
  game_clock_.reset_game_time();
  ext_train_.reset_game_time();
  map_dlg[1].dp = (void*)str_mgr_.get_string("new_game_seed");
  map_dlg[3].dp = (void*)str_mgr_.get_string("new_game_width");
  map_dlg[5].dp = (void*)str_mgr_.get_string("new_game_height");
  map_dlg[7].dp = (void*)str_mgr_.get_string("new_game_ok");
  map_dlg[8].dp = (void*)str_mgr_.get_string("new_game_cancel");
  centre_dialog( map_dlg );
  const int rc = do_dialog( map_dlg, 7 );
  if( rc == 7 )
  {
    const int seed   = atoi( s_buffer );
    const int width  = atoi( w_buffer );
    const int height = atoi( h_buffer );
    if(  seed > 0 && seed < 65000
      && width >= 30 && width < 300
      && height >= 30 && height < 300
      )
    {
      reset_selection();
      change_menu_state( false, false );
      v_stations_.clear();
      v_trains_.clear();
      v_apartments_.clear();
      v_offices_.clear();
      train_station_set_.clear();
      current_hour_since0_ = -1;
      current_day_since0_ = -1;
      current_month_ = -1;
      current_year_since0_ = -1;
      last_value_update_time_ = 0;
      last_build_decision_time_ = 0;
      last_update_buildings_time_ = 0;
      cash_ = Cash( def_values_.initial_total_cash_ );
      generate_random_map( seed, width, height );
      p_main_view_->notify_chg_size();
    }
  }
}

void WorldMap::save_game()
{
  char filename[FILENAME_MAX+1] = default_savefilename;
  if( ! file_select_ex( str_mgr_.get_string("save_game"), filename, "tnt", FILENAME_MAX, SCREEN_W/2, SCREEN_H/2 ) )
    return;
  PACKFILE*fp = pack_fopen( filename, F_WRITE );
  save_to_file( fp );
  pack_fclose( fp );
}

void WorldMap::save_autosave()
{
  PACKFILE*fp = pack_fopen( autosave_filename, F_WRITE );
  save_to_file( fp );
  pack_fclose( fp );
}

void WorldMap::load_autosave()
{
  PACKFILE*fp = pack_fopen( autosave_filename, F_READ );
  const int rc = load_from_file( fp );
  if( rc>0 )
    allegro_message( "outdated file format" );
  if( rc<0 )
    fatalError( "error reading tnt-file" );
  pack_fclose( fp );
}

bool WorldMap::exists_autosave()
{
  PACKFILE*fp = pack_fopen( autosave_filename, F_READ );
  if( fp )
  {
    pack_fclose( fp );
    return true;
  }
  return false;
}

void WorldMap::load_game()
{
  char filename[FILENAME_MAX+1] = default_savefilename;
  if( ! file_select_ex( str_mgr_.get_string("load_game"), filename, "tnt", FILENAME_MAX, SCREEN_W/2, SCREEN_H/2 ) )
    return;
  PACKFILE*fp = pack_fopen( filename, F_READ );

  const int rc = load_from_file( fp );
  if( rc>0 )
    allegro_message( "outdated file format" );
  if( rc<0 )
    fatalError( "error reading tnt-file" );

  pack_fclose( fp );
  p_main_view_->notify_chg_size();
}

//----------------------------------------------------------------------------

template<typename T>
void write_pack( PACKFILE*fp, const T&t )
{
  pack_fwrite( &t, sizeof(T), fp );
}

template<typename T>
bool read_pack( PACKFILE*fp, T&t )
{
  return sizeof(T) == pack_fread( &t, sizeof(T), fp );
}

template<typename T>
void write_sizeof( PACKFILE*fp )
{
  const int sz_T = sizeof(T);
  write_pack( fp, sz_T );
}

template<typename T>
bool read_and_check_sizeof( PACKFILE*fp )
{
  int sz_T = 0;
  if( sizeof(int) != pack_fread( &sz_T, sizeof(int), fp ) )
    return false;
  return sz_T == sizeof(T);
}

template<typename T>
void write_pack( PACKFILE*fp, const std::vector<T>&v  )
{
  write_pack( fp, v.size() );
  for( size_t i=0; i<v.size(); ++i )
  {
    const T&t = v[i];
    write_pack( fp, t );
  }
}

template<typename T>
bool read_pack( PACKFILE*fp, std::vector<T>&v )
{
  v.clear();
  size_t v_size = 0;
  if( !read_pack( fp, v_size ) ) return false;
  v.resize( v_size );
  for( size_t i=0; i<v.size(); ++i )
  {
    T&t = v[i];
    if( !read_pack( fp, t ) ) return false;
  }
  return true;
}

template<typename T>
void write_pack( PACKFILE*fp, const std::set<T>&m )
{
  write_pack( fp, m.size() );
  for( std::set<T>::const_iterator it = m.begin(); it != m.end(); ++it )
  {
    write_pack( fp, *it );
  }
}

template<typename T>
bool read_pack( PACKFILE*fp, std::set<T>&m )
{
  m.clear();
  size_t m_size = 0;
  if( !read_pack( fp, m_size ) ) return false;

  for( int i=0; i<m_size; ++i )
  {
    T t;
    if( !read_pack( fp, t ) ) return false;
    m.insert(t);
  }
  return true;
}

template<typename T,typename U>
void write_pack( PACKFILE*fp, const std::map<T,U>&m )
{
  write_pack( fp, m.size() );
  for( std::map<T,U>::const_iterator it = m.begin(); it != m.end(); ++it )
  {
    write_pack( fp, it->first );
    write_pack( fp, it->second );
  }
}

template<typename T,typename U>
bool read_pack( PACKFILE*fp, std::map<T,U>&m )
{
  m.clear();
  size_t m_size = 0;
  if( !read_pack( fp, m_size ) ) return false;

  for( int i=0; i<m_size; ++i )
  {
    T t;
    U u;
    if( !read_pack( fp, t ) ) return false;
    if( !read_pack( fp, u ) ) return false;
    m[t] = u;
  }
  return true;
}

//----------------------------------------------------------------------------

int WorldMap::load_from_file( PACKFILE*fp )
{
  if( !read_and_check_sizeof<Cash>( fp ) )              return 1;
  if( !read_and_check_sizeof<ExtTrain>( fp ) )          return 1;
  if( !read_and_check_sizeof<MapTile>( fp ) )           return 1;
  if( !read_and_check_sizeof<Station>( fp ) )           return 1;
  if( !read_and_check_sizeof<Train>( fp ) )             return 1;
  if( !read_and_check_sizeof<Apartment>( fp ) )         return 1;
  if( !read_and_check_sizeof<Office>( fp ) )            return 1;
  if( !read_and_check_sizeof<TrainStationPair>( fp ) )  return 1;

  int cur_game_time = 0;
  if( !read_pack( fp, cur_game_time ) )   return 1;
  int cur_speed_value = 0;
  if( !read_pack( fp, cur_speed_value ) ) return 1;
  game_clock_.reset_game_time( cur_game_time, cur_speed_value );

  if( !read_pack( fp, map_x_ext_ ) )    return -1;
  if( !read_pack( fp, map_y_ext_ ) )    return -1;
  if( !read_pack( fp, ext_entry_ ) )    return -1;
  if( !read_pack( fp, ext_train_ ) )    return -1;
  if( !read_pack( fp, cash_ ) )         return -1;
  if( !read_pack( fp, current_hour_since0_ ) )  return -1;
  if( !read_pack( fp, current_day_since0_ ) )   return -1;
  if( !read_pack( fp, current_month_ ) )        return -1;
  if( !read_pack( fp, current_year_since0_ ) )  return -1;
  if( !read_pack( fp, last_build_decision_time_ ) )   return -1;
  if( !read_pack( fp, last_value_update_time_ ) )     return -1;
  if( !read_pack( fp, last_update_buildings_time_ ) ) return -1;

  if( !read_pack( fp, map_tiles_ ) )            return -1;
  if( map_tiles_.size() != map_x_ext_*map_y_ext_ ) return -1;
  for( size_t it=0; it<map_tiles_.size(); ++it )
  {
    MapTile&mt = map_tiles_[it];
    mt.tile_ = 0;
    mt.tile_ = &tile_mgr_.tile_by_tile_idx( mt.tile_idx_ );
  }
  if( !read_pack( fp, v_stations_ ) )       return -1;
  if( !read_pack( fp, v_trains_ ) )         return -1;
  if( !read_pack( fp, v_apartments_ ) )     return -1;
  if( !read_pack( fp, v_offices_ ) )        return -1;

  if( !read_pack( fp, train_station_set_ ) ) return -1;

  return 0;
}

//----------------------------------------------------------------------------

void WorldMap::save_to_file( PACKFILE*fp ) const
{
  write_sizeof<Cash>( fp );
  write_sizeof<ExtTrain>( fp );
  write_sizeof<MapTile>( fp );
  write_sizeof<Station>( fp );
  write_sizeof<Train>( fp );
  write_sizeof<Apartment>( fp );
  write_sizeof<Office>( fp );
  write_sizeof<TrainStationPair>( fp );

  write_pack( fp, game_clock_.get_time_since0() );
  write_pack( fp, game_clock_.get_speed_value() );
  write_pack( fp, map_x_ext_ );
  write_pack( fp, map_y_ext_ );
  write_pack( fp, ext_entry_ );
  write_pack( fp, ext_train_ );
  write_pack( fp, cash_ );
  write_pack( fp, current_hour_since0_ );
  write_pack( fp, current_day_since0_ );
  write_pack( fp, current_month_ );
  write_pack( fp, current_year_since0_ );
  write_pack( fp, last_build_decision_time_ );
  write_pack( fp, last_value_update_time_ );
  write_pack( fp, last_update_buildings_time_ );

  write_pack( fp, map_tiles_ );
  write_pack( fp, v_stations_ );
  write_pack( fp, v_trains_ );
  write_pack( fp, v_apartments_ );
  write_pack( fp, v_offices_ );

  write_pack( fp, train_station_set_ );
}

//----------------------------------------------------------------------------
