#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

namespace
{

//----------------------------------------------------------------------------

#define IDXrail_____ (TileClassIDX)(-1)
const TileClassIDX rail_by_2_directions[8][8] =
{
  //        nn            no            oo            so            ss            sw            ww            nw
  { IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_sonn, IDXrail_nnss, IDXrail_nnsw, IDXrail_____, IDXrail_____ } // nn
, { IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_ssno, IDXrail_nosw, IDXrail_noww, IDXrail_____ } // no
, { IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_swoo, IDXrail_wwoo, IDXrail_oonw } // oo
, { IDXrail_sonn, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_wwso, IDXrail_nwso } // so
, { IDXrail_nnss, IDXrail_ssno, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_nwss } // ss
, { IDXrail_nnsw, IDXrail_nosw, IDXrail_swoo, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____ } // sw
, { IDXrail_____, IDXrail_noww, IDXrail_wwoo, IDXrail_wwso, IDXrail_____, IDXrail_____, IDXrail_____, IDXrail_____ } // ww
, { IDXrail_____, IDXrail_____, IDXrail_oonw, IDXrail_nwso, IDXrail_nwss, IDXrail_____, IDXrail_____, IDXrail_____ } // nw
};

//----------------------------------------------------------------------------

const TileKOO tilekoo_by_direction[] =
{
  TileKOO(-1,-1) // nn
, TileKOO( 0,-1) // no
, TileKOO( 1,-1) // oo
, TileKOO( 1, 0) // so
, TileKOO( 1, 1) // ss
, TileKOO( 0, 1) // sw
, TileKOO(-1, 1) // ww
, TileKOO(-1, 0) // nw
// ein zweites Mal, damit man mit Direction rechnen kann
, TileKOO(-1,-1) // nn
, TileKOO( 0,-1) // no
, TileKOO( 1,-1) // oo
, TileKOO( 1, 0) // so
, TileKOO( 1, 1) // ss
, TileKOO( 0, 1) // sw
, TileKOO(-1, 1) // ww
, TileKOO(-1, 0) // nw
};

//----------------------------------------------------------------------------

const Direction dir_part_table_by_rail[][3] =
{
  { DIR_nn, DIR_ss, DIR_none } // nnss
, { DIR_ww, DIR_oo, DIR_none } // wwoo
, { DIR_nw, DIR_so, DIR_none } // nwso
, { DIR_no, DIR_sw, DIR_none } // nosw

, { DIR_ww, DIR_so, DIR_none } // wwso
, { DIR_sw, DIR_oo, DIR_none } // swoo
, { DIR_no, DIR_ww, DIR_none } // noww
, { DIR_oo, DIR_nw, DIR_none } // oonw
, { DIR_so, DIR_nn, DIR_none } // sonn
, { DIR_nn, DIR_sw, DIR_none } // nnsw
, { DIR_ss, DIR_no, DIR_none } // ssno
, { DIR_nw, DIR_ss, DIR_none } // nwss
};

inline bool has_dir( Direction dir, TileClassIDX tcl )
{
  return dir_part_table_by_rail[tcl][0] == dir
    ||   dir_part_table_by_rail[tcl][1] == dir
    ;
}

} // namespace








//----------------------------------------------------------------------------

inline int sign( int x )
{
  return x>0 ? 1 : x<0 ? -1 : 0;
}

inline int dist_max( const TileKOO&t, const TileKOO&u )
{
  const TileKOO diff = t - u;
  return MAX( abs(diff.x), abs(diff.y) );
}

inline Direction get_direction_from_tilekoo( const TileKOO&t )
{
  if( t.x == 0 )
  {
    if( t.y == 0 )
      return DIR_none;
    else if( t.y > 0 )
      return DIR_sw;
    else
      return DIR_no;
  }
  if( t.y == 0 )
  {
    if( t.x > 0 )
      return DIR_so;
    else
      return DIR_nw;
  }
  if( t.x == t.y )
  {
    if( t.x > 0 )
      return DIR_ss;
    else
      return DIR_nn;
  }
  if( t.x == -t.y )
  {
    if( t.x > 0 )
      return DIR_oo;
    else
      return DIR_ww;
  }
  return DIR_none;
}

inline TileKOO normalize_tilekoo( TileKOO t )
{
  if( t.x == 0 )
    t.y = sign(t.y);
  else if( t.y == 0 )
    t.x = sign(t.x);
  else
  {
    assertCondition( abs(t.x) == abs(t.y) );
    t.x = sign(t.x);
    t.y = sign(t.y);
  }
  return t;
}

//----------------------------------------------------------------------------

Direction WorldMap::lay_rail_find_straight_line( const TileKOO&t_start, const TileKOO&t_end, RailPath&rail_path, bool include_first ) const
{
  const TileKOO diff = t_end - t_start;
  const Direction dir = get_direction_from_tilekoo( diff );
  if( dir == DIR_none )
    return dir;
  const TileKOO norm_diff = normalize_tilekoo(diff);
  const TileClassIDX new_tcl = rail_by_2_directions[dir][opposite_dir(dir)];
  assertCondition( new_tcl != IDXrail_____ );

  TileKOO t(t_start);
  if( !include_first )
    t += norm_diff;
  while(true)
  {
    assertCondition( is_on_map(t) );
    rail_path.push_back( RailPathDesc( t, new_tcl ) );
    if( t == t_end )
      break;
    t += norm_diff;
  }

  return dir;
}

int WorldMap::lay_rail_find_path( const TileKOO&t_start, const TileKOO&t_end, RailPath&rail_path ) const
{
  rail_path.clear();
  if( t_start == t_end )
    return 0;
  const TileKOO diff = t_end - t_start;

  TileKOO t_mid1(t_start);
  TileKOO t_mid2(t_start);
  if( abs(diff.y) <= abs(diff.x) )
  {
    const int d = abs(diff.y)*sign(diff.x);
    t_mid1 += TileKOO( diff.x-d, 0 );
    t_mid2 += TileKOO( d, diff.y );
  }
  else
  {
    const int d = abs(diff.x)*sign(diff.y);
    t_mid1 += TileKOO( 0, diff.y-d );
    t_mid2 += TileKOO( diff.x, d );
  }
  const TileKOO t_mid = dist_max(t_start,t_mid1) > dist_max(t_start,t_mid2) ? t_mid1 : t_mid2;

  Direction dir_first=DIR_none;
  Direction dir_second=DIR_none;
  if( t_mid == t_start || t_mid == t_end )
  {
    dir_first = lay_rail_find_straight_line( t_start, t_end, rail_path, true );
    dir_second = dir_first;
  }
  else
  {
    dir_first = lay_rail_find_straight_line( t_start, t_mid, rail_path, true );
    const int idx_mid_tile = rail_path.size()-1;
    dir_second = lay_rail_find_straight_line( t_mid, t_end, rail_path, false );
    const TileClassIDX new_tcl = rail_by_2_directions[opposite_dir(dir_first)][dir_second];
    assertCondition( new_tcl != IDXrail_____ );
    rail_path[idx_mid_tile].tcl_ = new_tcl;
  }

  if( dir_first==DIR_none )
    return 0;

  if( rail_path.size() > 0 )
  {
    // modify first rail tile
    {
      const RailPathDesc&rpd0 = rail_path.front();
      const MapTile mt0 = map_tiles_[ index(rpd0.t_) ];
      if( !mt0.is_rail() )
      {
        std::vector<Direction> nb_directions;
        const int n_nb = find_connecting_rail_neighbours( rpd0.t_, nb_directions );
        assertCondition( n_nb == nb_directions.size() );
        if( n_nb==1 )
        {
          const Direction nb_dir = nb_directions[0];
          const TileClassIDX tcl_rail = rail_by_2_directions[dir_first][nb_dir];
          if( tcl_rail != IDXrail_____ )
          {
            rail_path.front().tcl_ = tcl_rail;
          }
        }
      }
    }
    // modify last rail tile
    {
      const RailPathDesc&rpd1 = rail_path.back();
      const MapTile mt1 = map_tiles_[ index(rpd1.t_) ];
      if( !mt1.is_rail() )
      {
        std::vector<Direction> nb_directions;
        const int n_nb = find_connecting_rail_neighbours( rpd1.t_, nb_directions );
        assertCondition( n_nb == nb_directions.size() );
        if( n_nb==1 )
        {
          const Direction nb_dir = nb_directions[0];
          const TileClassIDX tcl_rail = rail_by_2_directions[opposite_dir(dir_second)][nb_dir];
          if( tcl_rail != IDXrail_____ )
          {
            rail_path.back().tcl_ = tcl_rail;
          }
        }
      }
    }
    int cost = 0;
    for( RailPath::const_iterator it = rail_path.begin(); it != rail_path.end(); ++it )
    {
      const TileKOO t = it->t_;
      const MapTile&mt = map_tiles_[ index(t) ];
      if( mt.is_building_ )
        return 0;
      if( mt.n_boxes_ > 0 )
        return 0;
      if( !mt.is_rail() && !mt.can_build_rail() )
        return 0;
      if( !is_rail_valid_here( t, it->tcl_ ) )
        return 0;
      if( !mt.player_is_owner_ )
        cost += mt.value();
      if( !mt.is_rail() )
        cost += def_values_.cost_lay_track;
    }
    if( cash_.player_total_ < cost )
      return 0;
    return cost;
  }
  return 0;
}

//----------------------------------------------------------------------------

int WorldMap::count_connecting_rail_neighbours( const TileKOO&t ) const
{
  int count=0;
  for( int idir=0; idir<DIR_none; ++idir )
  {
    const TileKOO&du = tilekoo_by_direction[idir];
    const TileKOO u = t+du;
    if( !is_on_map(u) )
      continue;
    const MapTile&mt = map_tiles_[ index(u) ];
    if( mt.is_rail() )
    {
      assertCondition( mt.tcl_idx_ <= IDXraillast );
      const Direction dir0 = dir_part_table_by_rail[mt.tcl_idx_][0];
      const Direction dir1 = dir_part_table_by_rail[mt.tcl_idx_][1];
      if( idir == dir0 || idir == dir1 )
        ++count;
    }
  }
  return count;
}

int WorldMap::find_connecting_rail_neighbours( const TileKOO&t, std::vector<Direction>&nb_directions ) const
{
  nb_directions.clear();
  for( int idir=0; idir<DIR_none; ++idir )
  {
    const TileKOO&du = tilekoo_by_direction[idir];
    const TileKOO u = t+du;
    if( !is_on_map(u) )
      continue;
    const MapTile&mt = map_tiles_[ index(u) ];
    if( mt.is_rail() )
    {
      assertCondition( mt.tcl_idx_ <= IDXraillast );
      const Direction dir0 = dir_part_table_by_rail[mt.tcl_idx_][0];
      const Direction dir1 = dir_part_table_by_rail[mt.tcl_idx_][1];
      if( idir == dir0 || idir == dir1 )
        nb_directions.push_back( (Direction)(idir) );
    }
  }
  return nb_directions.size();
}

int WorldMap::find_connecting_rail_neighbours( const TileKOO&t, std::vector<TileKOO>&nb_tiles ) const
{
  nb_tiles.clear();
  for( int idir=0; idir<DIR_none; ++idir )
  {
    const TileKOO&du = tilekoo_by_direction[idir];
    const TileKOO u = t+du;
    if( !is_on_map(u) )
      continue;
    const MapTile&mt = map_tiles_[ index(u) ];
    if( mt.is_rail() )
    {
      assertCondition( mt.tcl_idx_ <= IDXraillast );
      const Direction dir0 = dir_part_table_by_rail[mt.tcl_idx_][0];
      const Direction dir1 = dir_part_table_by_rail[mt.tcl_idx_][1];
      if( idir == dir0 || idir == dir1 )
        nb_tiles.push_back( u );
    }
  }
  return nb_tiles.size();
}

//----------------------------------------------------------------------------

bool WorldMap::is_rail_compatible( TileClassIDX t, TileClassIDX u, Direction dir ) const
{
  const Direction opp = opposite_dir(dir);

  if(  has_dir(dir,t)
    && has_dir(opp,u)
    )
    return true;
  if(  !has_dir(dir,t)
    && !has_dir(opp,u)
    )
  {
    if(  ( dir==DIR_no && has_dir(DIR_nn,t) && has_dir(DIR_ww,u) )
      || ( dir==DIR_no && has_dir(DIR_oo,t) && has_dir(DIR_ss,u) )
      || ( dir==DIR_nw && has_dir(DIR_nn,t) && has_dir(DIR_oo,u) )
      || ( dir==DIR_nw && has_dir(DIR_ww,t) && has_dir(DIR_ss,u) )
      || ( dir==DIR_so && has_dir(DIR_ss,t) && has_dir(DIR_ww,u) )
      || ( dir==DIR_so && has_dir(DIR_oo,t) && has_dir(DIR_nn,u) )
      || ( dir==DIR_sw && has_dir(DIR_ss,t) && has_dir(DIR_oo,u) )
      || ( dir==DIR_sw && has_dir(DIR_ww,t) && has_dir(DIR_nn,u) )
      )
      return false;
    return true;
  }
  return false;
}

bool WorldMap::is_rail_valid_here( const TileKOO&t, TileClassIDX tcl ) const
{
  assertCondition( is_on_map(t) );
  assertCondition( tcl<=IDXraillast );
  if( !is_inside_map(t) )
    return false;
  for( int idir=0; idir<8; ++idir )
  {
    const TileKOO&du = tilekoo_by_direction[idir];
    const TileKOO u = t+du;
    if( !is_inside_map(u) )
      continue;
    const MapTile&mt = map_tiles_[ index(u) ];
    if( mt.is_rail() )
    {
      if( !is_rail_compatible( tcl, mt.tcl_idx_, (Direction)idir ) )
        return false;
    }
  }
  return true;
}

//----------------------------------------------------------------------------

void WorldMap::lay_rail( const RailPath&rail_path )
{
  for( RailPath::const_iterator it=rail_path.begin(); it!=rail_path.end(); ++it )
  {
    const TileKOO t = it->t_;
    const MapTile&mt = map_tiles_[ index(t) ];
    if( !mt.player_is_owner_ && mt.can_build_rail() )
    {
      buy_land(t,true);
    }
    assertCondition( mt.is_rail() || mt.can_build_rail() );
    assertCondition( mt.player_is_owner_ );
    set_general_tile( t, it->tcl_, SUB_IDX_RANDOM, false, true );
    cash_.trains_re_.push_cost( def_values_.cost_lay_track );
    cash_.total_.push_cost( def_values_.cost_lay_track );
    cash_.player_total_ -= def_values_.cost_lay_track;
  }
}

int WorldMap::delete_rail_find_cost( const TileKOO&t ) const
{
  const MapTile&mt = map_tiles_[ index(t) ];
  if( !mt.is_rail() || !mt.player_is_owner_ || mt.train_show_!=NoTrain )
    return 0;
  if( cash_.player_total_ < def_values_.cost_delete_track )
    return 0;
  return def_values_.cost_delete_track;
}

//----------------------------------------------------------------------------

void WorldMap::delete_rail( const TileKOO&t )
{
  const MapTile&mt = map_tiles_[ index(t) ];
  assertCondition( mt.is_rail() );
  assertCondition( mt.player_is_owner_ );
  assertCondition( mt.train_show_ == NoTrain );
  set_box_tile( t, true, 0 );
  cash_.trains_re_.push_cost( def_values_.cost_delete_track );
  cash_.total_.push_cost( def_values_.cost_delete_track );
  cash_.player_total_ -= def_values_.cost_delete_track;
}























//----------------------------------------------------------------------------

const TileKOO WorldMap::find_next_rail_tile( Train::Positions&pos ) const
{
  TileKOO result;
  const TileKOO t0 = pos[0];
  if( !is_on_map(t0) )
    return result;
  const TileKOO t1 = pos[1];
  const MapTile&mt0 = map_tiles_[ index(t0) ];
  const TileClassIDX tcl_idx = mt0.tcl_idx_;
  assertCondition( tcl_idx <= IDXraillast );
  const Direction dir0 = dir_part_table_by_rail[tcl_idx][0];
  const Direction dir1 = dir_part_table_by_rail[tcl_idx][1];
  const TileKOO d0 = tilekoo_by_direction[dir0];
  const TileKOO d1 = tilekoo_by_direction[dir1];
  const TileKOO n00 = t0 + d0; // ( t0.x+d0.x, t0.y+d0.y );
  const TileKOO n01 = t0 + d1; // ( t0.x+d1.x, t0.y+d1.y );
  if( !is_on_map(n00) )
  {
    if( is_on_map(n01) && n01 != t1 )
      result = n01;
  }
  else
  {
    if( n00!=t1 )
      result = n00;
    else if( is_on_map(n01) )
      result = n01;
  }
  return result;
}

//----------------------------------------------------------------------------
