#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

//----------------------------------------------------------------------------

Station::Station( const TileKOO&ref_tile, const TileKOO&center_tile, MultiTileIDX mt_idx, int station_id )
: ref_tile_(ref_tile)
, center_tile_(center_tile)
, mt_idx_(mt_idx)
, station_id_(station_id)
, n_persons_env_(0)
, n_jobs_env_(0)
, n_good_passengers_(0)
, n_miss_passengers_(0)
{
}

//----------------------------------------------------------------------------

bool WorldMap::eval_station_building( const TileKOO&ti, int&cost ) const
{
  if( !is_inside_map(ti) )
    return false;
  const MapTile&mt = map_tiles_[ index(ti) ];
  if( !mt.player_can_buy_ && !mt.player_is_owner_ )
    return false;
  if( mt.is_building_ || mt.is_rail() || mt.n_boxes_ > 0 )
    return false;
  if( !mt.player_is_owner_ )
    cost += mt.value();
  return true;
}

int WorldMap::build_station_find_pos( const TileKOO&t, MultiTileIDX mt_idx, Outline&station_shape ) const
{
  const Tile*const p_empty_tile = &get_tile( IDXempty );
  int cost = def_values_.cost_build_station;
  const MultiTile&mult = tile_mgr_.multi_tile_by_idx( mt_idx );
  station_shape.clear();
  // check if everything is inside the map
  {
    for( int iy=0; iy<3; ++iy )
      for( int ix=0; ix<3; ++ix )
      {
        const TileKOO tij( t.x+ix, t.y-iy );
        if( !is_inside_map(tij) )
          return 0;
        const MapTile&mt = map_tiles_[ index(tij) ];
        if( mt.station_id_ >= 0 )
          return 0;
      }
  }
  // check for other stations
#if 0
  {
    const TileKOO center_t = center_tile_of_station( t, mt_idx );
    const int min_distance = MAX( 5, def_values_.radius_add_value / 2 );
    for( size_t is=0; is<v_stations_.size(); ++is )
    {
      const Station&station = v_stations_[is];
      const TileKOO diff = station.center_tile_ - center_t;
      if( diff.length() < min_distance )
        return 0;
    }
  }
#endif

  switch(mt_idx)
  {
  case MTIDXstation_up_nwso:
    {
      for( int ix=0; ix<3; ++ix )
      {
        TileKOO ti( t.x+ix, t.y-2 );
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.y;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.y;
        if( !eval_station_building( ti, cost ) )
          return 0;
        station_shape.push_back( OutlineDesc( ti, mult.get_tile(ix,0) ) );
      }
    }
    break;
  case MTIDXstation_up_nosw:
    {
      for( int iy=2; iy>=0; --iy )
      {
        TileKOO ti( t.x, t.y-iy );
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.x;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.x;
        if( !eval_station_building( ti, cost ) )
          return 0;
        station_shape.push_back( OutlineDesc( ti, mult.get_tile(0,2-iy) ) );
      }
    }
    break;
  case MTIDXstation_dn_nosw:
    {
      for( int iy=0; iy<3; ++iy )
      {
        TileKOO ti( t.x, t.y-iy );
        if( !eval_station_building( ti, cost ) )
          return 0;
        station_shape.push_back( OutlineDesc( ti, mult.get_tile(0,2-iy) ) );
        ++ti.x;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.x;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
      }
    }
    break;
  case MTIDXstation_dn_nwso:
    {
      for( int ix=0; ix<3; ++ix )
      {
        TileKOO ti( t.x+ix, t.y-2 );
        if( !eval_station_building( ti, cost ) )
          return 0;
        station_shape.push_back( OutlineDesc( ti, mult.get_tile(ix,0) ) );
        ++ti.y;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
        ++ti.y;
        station_shape.push_back( OutlineDesc( ti, p_empty_tile ) );
      }
    }
    break;
  }
  if( cash_.player_total_ < cost )
    return 0;
  return cost;
}

//----------------------------------------------------------------------------

void WorldMap::build_station_building_tile( const TileKOO&ti, bool player_is_owner, const Tile*p_tile )
{
  MapTile&mt = map_tiles_[ index(ti) ];
  assertCondition( mt.player_can_buy_ || mt.player_is_owner_ );
  if( !mt.player_is_owner_ && player_is_owner)
    buy_land(ti,true);
  else
  {
    mt.player_can_buy_ = false;
    mt.player_is_owner_ = player_is_owner;
  }
  mt.assign_tile( p_tile, (TileClassIDX)-1 );
  mt.can_hold_boxes_ = false;
  mt.is_building_ = true;
  mt.n_boxes_  = 0;
  mt.n_jobs_ = 0;
  mt.n_persons_ = 0;
  mt.bas_value_ = def_values_.bas_value_empty;
}

/*
.   0
.  y x

up_nwso       dn_nwso       up_nosw       dn_nosw
.   o         .   x c       .   o         . c x
.  o o        .  o x        .  o o        .  x o
. r o o       . r o x       . r o x       . r o o
.  x o        .  o o        .  o x        .  o o
. c x         .   o         .   x c       .   o

*/

TileKOO WorldMap::center_tile_of_station( const TileKOO&ref_tile, MultiTileIDX mt_idx ) const
{
  TileKOO center_t(ref_tile);
  switch( mt_idx )
  {
  case MTIDXstation_up_nwso:
    center_t.x += 1;
    center_t.y += 2;
    break;
  case MTIDXstation_dn_nwso:
    center_t.x += 1;
    center_t.y -= 4;
    break;
  case MTIDXstation_up_nosw:
    center_t.x += 4;
    center_t.y -= 1;
    break;
  case MTIDXstation_dn_nosw:
    center_t.x -= 2;
    center_t.y -= 1;
    break;
  }
  return center_t;
}

void WorldMap::build_station( const TileKOO&t, MultiTileIDX mt_idx, bool player_is_owner )
{
  const MultiTile&mult = tile_mgr_.multi_tile_by_idx( mt_idx );
  const int station_id = find_empty_slot( v_stations_ );

  const TileKOO center_t = center_tile_of_station( t, mt_idx );
  const Station new_station( t, center_t, mt_idx, station_id );
  switch(mt_idx)
  {
  case MTIDXstation_up_nwso:
    {
      for( int ix=0; ix<3; ++ix )
      {
        const TileKOO ti( t.x+ix, t.y );
        build_station_building_tile( ti, player_is_owner, mult.get_tile(ix,0) );
      }
    }
    break;
  case MTIDXstation_dn_nwso:
    {
      for( int ix=0; ix<3; ++ix )
      {
        const TileKOO ti( t.x+ix, t.y-2 );
        build_station_building_tile( ti, player_is_owner, mult.get_tile(ix,0) );
      }
    }
    break;
  case MTIDXstation_up_nosw:
    {
      for( int iy=0; iy<3; ++iy )
      {
        const TileKOO ti( t.x+2, t.y-iy );
        build_station_building_tile( ti, player_is_owner, mult.get_tile(0,2-iy) );
      }
    }
    break;
  case MTIDXstation_dn_nosw:
    {
      for( int iy=0; iy<3; ++iy )
      {
        const TileKOO ti( t.x, t.y-iy );
        build_station_building_tile( ti, player_is_owner, mult.get_tile(0,2-iy) );
      }
    }
    break;
  default:
    assertCondition(false);
    break;
  }
  {
    for( int iy=0; iy<3; ++iy )
      for( int ix=0; ix<3; ++ix )
      {
        MapTile&mt = map_tiles_[ index(TileKOO(t.x+ix,t.y-iy)) ];
        mt.station_id_ = station_id;
      }
  }
  if( player_is_owner )
  {
    cash_.trains_re_.push_cost( def_values_.cost_build_station );
    cash_.total_.push_cost( def_values_.cost_build_station );
    cash_.player_total_ -= def_values_.cost_build_station;
  }
  if( station_id >= v_stations_.size() )
    v_stations_.push_back( new_station );
  else
    v_stations_[station_id] = new_station;
}

//----------------------------------------------------------------------------

void WorldMap::delete_station( int station_id )
{
  assertCondition( station_id > 0 );
  Station&station = v_stations_[station_id];
  assertCondition( station_id == station.station_id_ );
  {
    const TileKOO&t = station.ref_tile_;
    for( int iy=0; iy<3; ++iy )
      for( int ix=0; ix<3; ++ix )
      {
        const TileKOO tij( t.x+ix, t.y-iy );
        MapTile&mt = map_tiles_[ index(tij) ];
        mt.station_id_ = -1;
        if( mt.is_building_ )
        {
          assertCondition( mt.player_is_owner_ );
          set_box_tile( tij, true, 0 );
        }
      }
  }
  cash_.trains_re_.push_cost( def_values_.cost_delete_station );
  cash_.total_.push_cost( def_values_.cost_delete_station );
  cash_.player_total_ -= def_values_.cost_delete_station;
  station.clear();
}

//----------------------------------------------------------------------------

void WorldMap::remap_all_stations_buildings()
{
  TileKOO t;
  for( t.x=0; t.x<map_x_ext_; ++t.x )
  {
    for( t.y=0; t.y<map_y_ext_; ++t.y )
    {
      MapTile&mt = map_tiles_[ index(t) ];
      mt.nearest_station_id_ = -1;
    }
  }
  // reset stations
  for( size_t is=1; is<v_stations_.size(); ++is )
  {
    Station&station = v_stations_[is];
    station.n_apartment_env_ = 0;
    station.n_office_env_ = 0;
    const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_station_count );
    for( size_t it=0; it<nb_size; ++it )
    {
      const TileKOO&dt = neighbourhood_.get_nb_le( def_values_.radius_station_count, it );
      const TileKOO tkk = station.center_tile_ + dt;
      if( !is_on_map(tkk) )
        continue;
      MapTile&mt = map_tiles_[ index(tkk) ];
      if( mt.nearest_station_id_ < 0 )
      {
        mt.nearest_station_id_ = station.station_id_;
      }
      else if( mt.nearest_station_id_ > 0 )
      {
        Station&old_station = v_stations_[ mt.nearest_station_id_ ];
        const double d_old_app = distance( tkk, old_station.center_tile_ );
        const double d_new_app = distance( tkk, station.center_tile_ );
        if( d_new_app < d_old_app )
        {
          mt.nearest_station_id_ = station.station_id_;
        }
      }
    }
  }
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    const Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    const MapTile&mt = map_tiles_[ index(app.center_tile_) ];
    if( mt.nearest_station_id_ < 0 )
      continue;
    Station&station = v_stations_[mt.nearest_station_id_];
    ++station.n_apartment_env_;
  }
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    const Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    const MapTile&mt = map_tiles_[ index(off.center_tile_) ];
    if( mt.nearest_station_id_ < 0 )
      continue;
    Station&station = v_stations_[mt.nearest_station_id_];
    ++station.n_office_env_;
  }
}

//----------------------------------------------------------------------------

bool WorldMap::does_train_stop_at_station( int train_id, int station_id ) const
{
  const TrainStationPair key(train_id,station_id);
  const TrainStationSet::const_iterator it = train_station_set_.find(key);
  if( it==train_station_set_.end() )
    return true;
  return false;
}

void WorldMap::toggle_stop_train_at_station( int train_id, int station_id )
{
  const TrainStationPair key(train_id,station_id);
  const TrainStationSet::const_iterator it = train_station_set_.find(key);
  if( it==train_station_set_.end() )
  {
    train_station_set_.insert(key);
  }
  else
  {
    train_station_set_.erase(key);
  }
}

//----------------------------------------------------------------------------
