#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_VIEWMAP
#include "ViewMap.h"
#endif
#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

#include <strstream>
#include <iomanip>

//----------------------------------------------------------------------------

std::ostream& operator<< ( std::ostream&os, const TileKOO&t )
{
  return os << "[" << t.x << "," << t.y << "]";
}

//----------------------------------------------------------------------------

MainView::MainView( const ScreenRect&asr, WorldMap&amap )
: ViewPortBase( asr )
, map(amap)
, offset(0,0)
, frst_x_(-1), frst_y_(-1), last_x_(-1), last_y_(-1)
, cursor_move_step_width(0)
, mouse_scroll_sensitivity(1)
, start_tile_selected()
, tile_clicked()
, mouse_clicked(false)
, station_idx(MTIDXstation_up_nwso)
, sub_idx(0)
, train_reversed(false)
, report_to_show(0)
, do_show_debug_details(false)
{
}

void MainView::init( const char*ini_fn )
{
  assertCondition( ini_fn );

  push_config_state();
  set_config_file(ini_fn);
  cursor_move_step_width = get_config_float( "scrolling", "cursor_scroll_width", 0.5 );
  mouse_scroll_sensitivity = get_config_float( "scrolling", "mouse_sensitivity", 1 );
  pop_config_state();
}

//----------------------------------------------------------------------------

ScreenKOO MainView::convert_tile2screen( const TileKOO&t ) const
{
  return ScreenKOO( area.x + offset.x + map.tile_width()  * (t.x-t.y)
                  , area.y + offset.y + map.tile_height() * (t.x+t.y)
                  );
}

void MainView::draw_one_tile( BITMAP*const db, ScreenKOO s, const Tile&tile ) const
{
  s.y += map.tile_height()*2;
  s.y -= tile.bmp_->h;
  draw_sprite( db, tile.bmp_, s.x, s.y );
}

void MainView::draw_one_tile_lit( BITMAP*const db, ScreenKOO s, const Tile&tile, int r, int g, int b, int a ) const
{
  s.y += map.tile_height()*2;
  s.y -= tile.bmp_->h;
  set_trans_blender( r, g, b, a );
  draw_lit_sprite( db, tile.bmp_, s.x, s.y, a );
}

void MainView::draw_one_tile_trans( BITMAP*const db, ScreenKOO s, const Tile&tile, int r, int g, int b, int a ) const
{
  s.y += map.tile_height()*2;
  s.y -= tile.bmp_->h;
  set_trans_blender( r, g, b, a );
  draw_trans_sprite( db, tile.bmp_, s.x, s.y );
}

void MainView::draw_tile_outline( BITMAP*const db, const TileKOO&t, bool border, int col_outline ) const
{
  const ScreenKOO s = convert_tile2screen(t);
  const int tw = map.tile_width();
  const int th = map.tile_height();
  int points[8];
  points[0] = s.x+tw-1;   points[1] = s.y;
  points[2] = s.x;        points[3] = s.y+th-1;
  points[4] = points[0];  points[5] = s.y+th*2-1;
  points[6] = s.x+tw*2-1; points[7] = points[3];
  set_trans_blender(0,0,0,128);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  polygon( db, 4, points, col_outline );
  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,0);
  if( border )
  {
    line( db, points[0], points[1], points[2], points[3], col_outline );
    line( db, points[2], points[3], points[4], points[5], col_outline );
    line( db, points[4], points[5], points[6], points[7], col_outline );
    line( db, points[6], points[7], points[0], points[1], col_outline );
  }
}

//----------------------------------------------------------------------------

const MapTileViewInf& MainView::get_map_tile_view( const TileKOO&t ) const
{
  return map_tiles_view_inf_[ map.index(t) ];
}

TileKOO MainView::get_visible_center() const
{
  return TileKOO( (last_x_+frst_x_)/2, (last_y_+frst_y_)/2 );
}

//----------------------------------------------------------------------------

void MainView::display( BITMAP*const db )
{
  set_clip_rect( db, area.x, area.y, area.x+area.w-1, area.y+area.h-1 );
  TileKOO mouse_hover(-1,-1);
  const ScreenKOO mouse_pos( mouse_x, mouse_y );
  int count_all_drawn_tiles=0;
  const Tile&dark_tile = map.get_tile( IDXdark );
  const int mwidth = map.map_width();
  const int mheight = map.map_height();
  const int twidth = 2*map.tile_width();
  const int theight = 2*map.tile_height();
  if( map_tiles_view_inf_.size() != mwidth * mheight )
    map_tiles_view_inf_.resize( mwidth * mheight );

  // nach scrolling: berechne neu frst_x_, frst_y_, last_x_ und last_y_ sowie is_tile_visible fr die Lage des sichtbaren Ausschnitts
  if( frst_y_ < 0 )
  {
    const int x0 = area.x - twidth;
    const int y0 = area.y - theight;
    const int x1 = area.x + area.w;
    const int y1 = area.y + area.h;
    for( int iy=0; iy<mheight; ++iy )
    {
      for( int ix=0; ix<mwidth; ++ix )
      {
        const TileKOO txy(ix,iy);
        const int mt_index = map.index(txy);
        const ScreenKOO s = convert_tile2screen(txy);
        if( s.x < x0 || s.y < y0 || s.x > x1 || s.y > y1 )
        {
          map_tiles_view_inf_[ mt_index ].is_tile_visible_ = false;
        }
        else
        {
          map_tiles_view_inf_[ mt_index ].is_tile_visible_ = true;
          if( frst_x_ < 0 || ix<frst_x_ )
            frst_x_ = ix;
          if( frst_y_ < 0 || iy<frst_y_ )
            frst_y_ = iy;
          if( ix>last_x_ )
            last_x_ = ix;
          if( iy>last_y_ )
            last_y_ = iy;
        }
      }
    }
  }

  for( int iy=MAX(0,frst_y_); iy<mheight; ++iy )
  {
    int count_drawn_tiles = 0;
    for( int ix=MAX(0,frst_x_); ix<mwidth; ++ix )
    {
      const TileKOO txy(ix,iy);
      const int mt_index = map.index(txy);
      if( !map_tiles_view_inf_[ mt_index ].is_tile_visible_ )
        continue;
      const MapTile&mt = map.get_map_tile(txy);
      if( !mt.tile_ )
        continue;
      const Tile&tile = *mt.tile_;
      const ScreenKOO s = convert_tile2screen(txy);
      if( s.x < -tile.bmp_->w || s.y < -tile.bmp_->h )
        continue;
      if(  mouse_pos.x >= s.x && mouse_pos.x <= s.x+twidth
        && mouse_pos.y >= s.y && mouse_pos.y <= s.y+theight
        )
      {
        const int old_col = getpixel( db, mouse_pos.x, mouse_pos.y );
        draw_sprite( db, dark_tile.bmp_, s.x, s.y );
        if( mouse_hover.x<0 && mouse_hover.y<0 && is_inside_bitmap( db, mouse_pos.x, mouse_pos.y, true ) && old_col!=getpixel( db, mouse_pos.x, mouse_pos.y ) )
        {
          mouse_hover.x = ix;
          mouse_hover.y = iy;
        }
      }
      if( s.x > area.x+area.w || s.y > area.y+area.h )
        break;
      // overlay map depending on current selection
      if( map.is_inside_map(txy) )
      {
        if(  map.selected_apartment_ >= 0 && mt.apartment_id_ == map.selected_apartment_
          || map.selected_office_ >= 0 && mt.office_id_ == map.selected_office_
          || map.selected_station_ >= 0 && mt.station_id_ == map.selected_station_
          )
          draw_one_tile_lit( db, s, tile, 0, 0, 255, 128 );
        else if( map.selected_station_ >= 0 && mt.nearest_station_id_ == map.selected_station_ )
          draw_one_tile_lit( db, s, tile, 0, 255, 255, 48 );
        else if( map.selected_train_ >= 0 && mt.station_id_ > 0 )
        {
          if( map.does_train_stop_at_station( map.selected_train_, mt.station_id_ ) )
            draw_one_tile_lit( db, s, tile, 0, 255, 0, 180 );
          else
            draw_one_tile_lit( db, s, tile, 255, 0, 0, 180 );
        }
        else
          draw_one_tile( db, s, tile );
      }
      else
        draw_one_tile( db, s, tile );
      // overlay train tiles
      if( mt.train_show_ != NoTrain )
      {
        const Tile&train_tile = map.get_train_overlay( mt.tcl_idx_, mt.train_show_ );
        if( mt.train_id_ >= 0 && map.selected_train_ == mt.train_id_ )
          draw_one_tile_lit( db, s, train_tile, 0, 0, 255, 180 );
        else
          draw_one_tile( db, s, train_tile );
      }
      ++count_all_drawn_tiles;
      ++count_drawn_tiles;
    }
    if( count_all_drawn_tiles > 0 && count_drawn_tiles==0 )
      break;
  }
  // handle mouse clicking on the map
  if( mouse_hover.x >= 0 && mouse_hover.y >= 0 )
  {
    bool mouse_click_complete = false;
    if( !mouse_clicked && (mouse_b&1) )
    {
      mouse_clicked = true;
      tile_clicked = mouse_hover;
    }
    if( mouse_clicked && (mouse_b&2) )
    {
      mouse_clicked = false;
      mouse_click_complete = false;
      tile_clicked.clear();
      start_tile_selected.clear();
    }
    if( mouse_clicked && !(mouse_b&3) )
    {
      mouse_clicked = false;
      mouse_click_complete = ( tile_clicked == mouse_hover );
      if( mouse_click_complete )
      {
        if( map.get_game_state()==LayTrack )
          if( !start_tile_selected.is_valid() )
            start_tile_selected = tile_clicked;
      }
      else
        tile_clicked.clear();
    }
    handle_mouse_clicked( db, mouse_hover, mouse_click_complete );
  }
  switch( report_to_show )
  {
  case 1:
    show_report_1( db );
    break;
  case 2:
    show_report_2( db );
    break;
  case 3:
    show_report_3( db );
    break;
  case 4:
    show_p_train_list( db );
    break;
  }
  if( do_show_debug_details && area.is_inside_rect(mouse_x,mouse_y) )
  {
    show_details_at_mouse( db, mouse_hover, count_all_drawn_tiles );
  }
  map.bot_info_string_ = "";
  if( mouse_hover.is_valid() )
  {
    std::strstream str;
    const MapTile&mt = map.get_map_tile( mouse_hover );
    str << "Value: " << (mt.bas_value_+mt.add_value_) << "$ ";
    if( mt.station_id_ > 0 )
    {
      const Station&station = map.get_stations()[mt.station_id_];
      str << "Station " << mt.station_id_ << ": "
        << station.n_good_passengers_ << " Passengers";
      if( station.n_miss_passengers_ > 0 )
        str << " " << station.n_miss_passengers_ << " missed!";
    }
    if( mt.apartment_id_ >= 0 )
    {
      const Apartment&app = map.get_apartments()[mt.apartment_id_];
      if( app.player_is_owner_ )
        str << "My ";
      str << "Apartment " << mt.apartment_id_ << ": " << mt.n_persons_ << " Pers";
    }
    str << '\0';
    map.bot_info_string_ = str.str();
  }
  set_clip_rect( db, 0, 0, db->w, db->h );
}

//----------------------------------------------------------------------------

void MainView::handle_mouse_clicked( BITMAP*const db, const TileKOO&mouse_hover, bool mouse_click_complete )
{
  const MapTile&mt = map.get_map_tile( mouse_hover );
  switch( map.get_game_state() )
  {
  case BuyLand:
    start_tile_selected.clear();
    if( mt.player_can_buy_ && !mt.is_building_ )
    {
      const int cost = map.buy_land_find_costs( mouse_hover );
      if( cost > 0 )
      {
        draw_tile_outline( db, mouse_hover, true, makecol(0,255,0) );
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
          map.buy_land( tile_clicked, true );
      }
    }
    break;
  case SellLand:
    start_tile_selected.clear();
    if( mt.player_is_owner_ && !mt.is_building_ && !mt.is_rail() )
    {
      const int revenue = map.sell_land_find_revenue( mouse_hover );
      if( revenue > 0 )
      {
        draw_tile_outline( db, mouse_hover, true, makecol(255,0,0) );
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(0,128,0), -1, "%d$", revenue );
        if( mouse_click_complete )
          map.sell_land( tile_clicked );
      }
    }
    break;
  case LayTrack:
    if( !mt.is_building_ && ( mt.player_can_buy_ || mt.player_is_owner_ ) )
    {
      if( start_tile_selected.is_valid() )
      {
        if( mouse_clicked && start_tile_selected == tile_clicked )
        {
          start_tile_selected.clear();
          tile_clicked.clear();
          break;
        }
        std::vector<RailPathDesc> rail_path;
        const int cost = map.lay_rail_find_path( start_tile_selected, mouse_hover, rail_path );
        if( cost > 0 )
        {
          draw_tile_outline( db, start_tile_selected, true, makecol(0,255,0) );
          for( size_t idx=0; idx<rail_path.size(); ++idx )
          {
            const Tile&tile = map.get_tile( rail_path[idx].tcl_ );
            const ScreenKOO s = convert_tile2screen( rail_path[idx].t_ );
            draw_one_tile_trans( db, s, tile, 0, 128, 0, 150 );
            //draw_tile_outline( db, rail_path[idx].t, true, makecol(0,255,0) );
          }
          if( mouse_click_complete )
          {
            map.lay_rail( rail_path );
            start_tile_selected.clear();
          }
          textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        }
        else
        {
          draw_tile_outline( db, start_tile_selected, true, makecol(160,128,0) );
          draw_tile_outline( db, mouse_hover, true, makecol(160,128,0) );
        }
      }
      else
        draw_tile_outline( db, mouse_hover, true, makecol(0,255,0) );
    }
    break;
  case DeleteTrack:
    start_tile_selected.clear();
    if( mt.player_is_owner_ && mt.is_rail() )
    {
      draw_tile_outline( db, mouse_hover, true, makecol(255,0,0) );
      const int cost = map.delete_rail_find_cost( mouse_hover );
      if( cost > 0 )
      {
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
          map.delete_rail( tile_clicked );
      }
    }
    break;
  case BuildStation:
    start_tile_selected.clear();
    //if( ( mt.player_can_buy_ || mt.player_is_owner_ ) && !mt.is_building_ )
    {
      draw_tile_outline( db, mouse_hover, true, makecol(0,255,0) );
      Outline station_shape;
      const int cost = map.build_station_find_pos( mouse_hover, station_idx, station_shape );
      if( cost > 0 )
      {
        for( size_t i=0; i<station_shape.size(); ++i )
        {
          const OutlineDesc&od = station_shape[i];
          const ScreenKOO s = convert_tile2screen( od.t_ );
          draw_one_tile_trans( db, s, *od.p_tile_, 0, 128, 0, 210 );
        }
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
        {
          map.build_station( tile_clicked, station_idx, true );
          map.change_game_state( NormalSimulation );
        }
      }
    }
    break;
  case PlaceFTrain:
  case PlacePTrain:
    start_tile_selected.clear();
    {
      const bool is_freigt_train = map.get_game_state() == PlaceFTrain;
      draw_tile_outline( db, mouse_hover, true, makecol(0,255,0) );
      Outline train_shape;
      const int cost = map.place_train_find_pos( mouse_hover, train_reversed, is_freigt_train, train_shape );
      if( cost > 0 )
      {
        for( size_t i=0; i<train_shape.size(); ++i )
        {
          const OutlineDesc&od = train_shape[i];
          const ScreenKOO s = convert_tile2screen( od.t_ );
          draw_one_tile_trans( db, s, *od.p_tile_, 0, 255, 0, 210 );
        }
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
        {
          if( is_freigt_train )
            map.place_freight_train( train_shape );
          else
            map.place_pers_train( train_shape );
          map.change_game_state( NormalSimulation );
        }
      }
    }
    break;
  case BuildApartm:
    start_tile_selected.clear();
    if( mt.can_build_apartment() )
    {
      Outline apartm_shape;
      const int cost = map.build_apa_find_cost( mouse_hover, sub_idx, apartm_shape );
      if( cost > 0 )
      {
        for( size_t i=0; i<apartm_shape.size(); ++i )
        {
          const OutlineDesc&od = apartm_shape[i];
          const ScreenKOO s = convert_tile2screen( od.t_ );
          draw_one_tile_trans( db, s, *od.p_tile_, 0, 128, 0, 210 );
        }
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
        {
          map.build_apa( tile_clicked, sub_idx );
          map.change_game_state( NormalSimulation );
        }
      }
      else
      {
        draw_tile_outline( db, mouse_hover, true, makecol(255,0,0) );
      }
    }
    break;
  case BuildOffice:
    start_tile_selected.clear();
    if( mt.can_build_office() )
    {
      Outline office_shape;
      const int cost = map.build_off_find_cost( mouse_hover, sub_idx, office_shape );
      if( cost > 0 )
      {
        for( size_t i=0; i<office_shape.size(); ++i )
        {
          const OutlineDesc&od = office_shape[i];
          const ScreenKOO s = convert_tile2screen( od.t_ );
          draw_one_tile_trans( db, s, *od.p_tile_, 0, 128, 0, 210 );
        }
        textprintf_centre_ex( db, font, mouse_x, mouse_y-2*text_height(font), makecol(128,0,0), -1, "%d$", cost );
        if( mouse_click_complete )
        {
          map.build_off( tile_clicked, sub_idx );
          map.change_game_state( NormalSimulation );
        }
      }
      else
      {
        draw_tile_outline( db, mouse_hover, true, makecol(255,0,0) );
      }
    }
    break;
  default:
    start_tile_selected.clear();
    if( mouse_click_complete )
    {
      if( map.selected_train_ >= 0 && mt.station_id_ > 0 )
      {
        map.toggle_stop_train_at_station( map.selected_train_, mt.station_id_ );
      }
      else if( mt.apartment_id_ >= 0 )
      {
        map.reset_selection();
        map.selected_apartment_ = mt.apartment_id_;
      }
      else if( mt.office_id_ >= 0 )
      {
        map.reset_selection();
        map.selected_office_ = mt.office_id_;
      }
      else if( mt.station_id_ > 0 && mt.is_station_building() )
      {
        map.reset_selection();
        map.selected_station_ = mt.station_id_;
      }
      else if( mt.is_rail() && mt.train_show_!=NoTrain )
      {
        map.reset_selection();
        map.selected_train_ = mt.train_id_;
      }
      else
      {
        map.reset_selection();
      }
      mouse_clicked = false;
      tile_clicked.clear();
    }
    draw_tile_outline( db, mouse_hover, true, makecol(64,64,255) );
    break;
  }
}

//----------------------------------------------------------------------------

void MainView::show_report_1( BITMAP*const db ) const
{
  const int fg_red = makecol(255,150,150);
  const int fg_grn = makecol(150,255,150);
  const int fg_wht = makecol(255,255,255);
  const int dy = text_height( font ) + 3;
  const int n_lines = 18;
  const int x0 = area.x + 8;
  const int y0 = area.y+area.h - n_lines * dy;
  set_trans_blender(0,0,0,170);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  rectfill( db, area.x, y0-2, area.x+area.w, area.y+area.h, 0 );
  const Cash&cash = map.get_player_cash();
  const Assets&assets = map.get_player_assets();
  char buf[1024];

  int x = x0;
  int y = y0;

  uszprintf( buf, 1024, "Total Cash: %9d$    P/L: ", cash.player_total_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, "%9d$", cash.total_.cur_profit_loss() );
  textprintf_ex( db, font, x, y, cash.total_.cur_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  if( map.cash_.taxes_to_pay_next_april_ > 0 )
  {
    x += text_length( font, buf );
    uszprintf( buf, 1024, "  Taxes next 1.4.: %9d$", map.cash_.taxes_to_pay_next_april_ );
    textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  }

  x = x0;
  y += dy;
  uszprintf( buf, 512, "Apartments: %d, Offices: %d, Stations: %d, Trains: %d, Raillength: %d"
    , assets.my_apartments_, assets.my_offices_, map.get_stations().size(), map.get_trains().size(), assets.my_rail_tiles_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );

  x = x0;
  y += dy;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Railway       Cost    Revenue        P/L" );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(today): %9d$ %9d$", cash.trains_op_.cur_cst_, cash.trains_op_.cur_rev_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.trains_op_.cur_profit_loss() );
  textprintf_ex( db, font, x, y, cash.trains_op_.cur_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(month): %9d$ %9d$", cash.trains_op_.cst_mtd_, cash.trains_op_.rev_mtd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.trains_op_.mtd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.trains_op_.mtd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, " (year): %9d$ %9d$", cash.trains_op_.cst_ytd_, cash.trains_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.trains_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.trains_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );

  x = x0;
  y += dy;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Apartments    Cost    Revenue        P/L" );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(today): %9d$ %9d$", cash.apartm_op_.cur_cst_, cash.apartm_op_.cur_rev_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.apartm_op_.cur_profit_loss() );
  textprintf_ex( db, font, x, y, cash.apartm_op_.cur_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(month): %9d$ %9d$", cash.apartm_op_.cst_mtd_, cash.apartm_op_.rev_mtd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.apartm_op_.mtd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.apartm_op_.mtd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, " (year): %9d$ %9d$", cash.apartm_op_.cst_ytd_, cash.apartm_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.apartm_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.apartm_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );

  x = x0;
  y += dy;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Offices       Cost    Revenue        P/L" );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(today): %9d$ %9d$", cash.office_op_.cur_cst_, cash.office_op_.cur_rev_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.office_op_.cur_profit_loss() );
  textprintf_ex( db, font, x, y, cash.office_op_.cur_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "(month): %9d$ %9d$", cash.office_op_.cst_mtd_, cash.office_op_.rev_mtd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.office_op_.mtd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.office_op_.mtd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, " (year): %9d$ %9d$", cash.office_op_.cst_ytd_, cash.office_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.office_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.office_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );

  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,255);
}

void MainView::show_report_2( BITMAP*const db ) const
{
  const int fg_red = makecol(255,150,150);
  const int fg_grn = makecol(150,255,150);
  const int fg_yel = makecol(255,255,  0);
  const int fg_wht = makecol(255,255,255);
  const int dy = text_height( font ) + 3;
  const int n_lines = 21;
  const int x0 = area.x + 8;
  const int y0 = area.y+area.h - n_lines * dy;
  set_trans_blender(0,0,0,170);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  rectfill( db, area.x, y0-2, area.x+area.w, area.y+area.h, 0 );
  const Cash&cash = map.get_player_cash();
  const Assets&assets = map.get_player_assets();
  char buf[1024];

  const int tax_asset = assets.taxes_estimate( map.def_values_.tax_rate_assets_ );
  const int tax_cash = cash.taxes_estimate_ytd( map.def_values_.tax_rate_profit_ );

  int x = x0;
  int y = y0;

  textprintf_ex( db, font, x, y, fg_wht, -1, "Railroad Assets:      %9d$", assets.sum_train_value_ );
  x = x0;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Apartments: %4d      %9d$", assets.my_apartments_, assets.sum_apartments_value_ );
  x = x0;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Offices:    %4d      %9d$", assets.my_offices_, assets.sum_office_value_ );
  x = x0;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Land:       %4d      %9d$", assets.my_land_tiles_, assets.sum_land_value_ );
  x = x0;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "Total:                %9d$  Property Tax (est): %9d$", assets.total(), tax_asset );

  x = x0;
  y += dy;
  y += dy;
  textprintf_ex( db, font, x, y, fg_wht, -1, "                    Cost    Revenue        P/L" );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Railroad Op.:  %9d$ %9d$", cash.trains_op_.cst_ytd_, cash.trains_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.trains_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.trains_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Railr. Sales:  %9d$ %9d$", cash.trains_re_.cst_ytd_, cash.trains_re_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.trains_re_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.trains_re_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Apartm. Rent:  %9d$ %9d$", cash.apartm_op_.cst_ytd_, cash.apartm_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.apartm_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.apartm_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Apartm. Sales: %9d$ %9d$", cash.apartm_re_.cst_ytd_, cash.apartm_re_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.apartm_re_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.apartm_re_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Office Rent:   %9d$ %9d$", cash.office_op_.cst_ytd_, cash.office_op_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.office_op_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.office_op_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Office Sales:  %9d$ %9d$", cash.office_re_.cst_ytd_, cash.office_re_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.office_re_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.office_re_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Land Sales:    %9d$ %9d$", cash.land_.cst_ytd_, cash.land_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.land_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.land_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Interest:      %9d$ %9d$", cash.interest_.cst_ytd_, cash.interest_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.interest_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.interest_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Total:         %9d$ %9d$", cash.total_.cst_ytd_, cash.total_.rev_ytd_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x += text_length( font, buf );
  uszprintf( buf, 1024, " %9d$", cash.total_.ytd_profit_loss() );
  textprintf_ex( db, font, x, y, cash.total_.ytd_profit_loss() >= 0 ? fg_grn : fg_red, -1, buf );

  x = x0;
  y += dy;
  y += dy;
  uszprintf( buf, 1024, "Profit/Loss:     %9d$       Income Tax (est):   %9d$", cash.total_.ytd_profit_loss(), tax_cash );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );

  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Cash:            %9d$       Total Tax (est):    %9d$", cash.player_total_, tax_cash+tax_asset );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  x = x0;
  y += dy;
  uszprintf( buf, 1024, "Total Value:     %9d$", cash.player_total_ + assets.total() );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  if( map.cash_.taxes_to_pay_next_april_ > 0 )
  {
    x += text_length( font, buf );
    uszprintf( buf, 1024, "       Taxes next 1.4.:    %9d$", map.cash_.taxes_to_pay_next_april_ );
    textprintf_ex( db, font, x, y, fg_yel, -1, buf );
  }

  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,255);
}

void MainView::show_report_3( BITMAP*const db ) const
{
  const int fg_wht = makecol(255,255,255);
  const int dy = text_height( font ) + 3;
  const int n_lines = 5;
  const int x0 = area.x + 8;
  const int y0 = area.y+area.h - n_lines * dy;
  set_trans_blender(0,0,0,170);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  rectfill( db, area.x, y0-2, area.x+area.w, area.y+area.h, 0 );
  const Cash&cash = map.get_player_cash();
  const Assets&assets = map.get_player_assets();
  char buf[512];

  int x = x0;
  int y = y0;

  uszprintf( buf, 512, "%d Apartments, %d Offices, %u Trains, %d Rail Tiles, %d Land Tiles"
    , assets.my_apartments_, assets.my_offices_, map.get_trains().size(), assets.my_rail_tiles_, assets.my_land_tiles_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );

  x = x0;
  y += dy;
  uszprintf( buf, 512, "%d Inhabitants", assets.all_inhabitants_ );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );

  x = x0;
  y += dy;
  const int sum_all_tiles = assets.all_land_tiles_ + assets.all_office_tiles_ + assets.all_person_tiles_ + assets.all_rail_tiles_;
  if( sum_all_tiles > 0 )
  {
    uszprintf( buf, 512, "Land usage: : %3d%%%% Land, %3d%%%% Rail, %3d%%%% Office, %3d%%%% Living"
      , (assets.all_land_tiles_*100)/sum_all_tiles
      , (assets.all_rail_tiles_*100)/sum_all_tiles
      , (assets.all_office_tiles_*100)/sum_all_tiles
      , (assets.all_person_tiles_*100)/sum_all_tiles
      );
    textprintf_ex( db, font, x, y, fg_wht, -1, buf );
  }

  x = x0;
  y += dy;
  const int sum_city_tiles = assets.all_office_tiles_ + assets.all_person_tiles_ + assets.all_rail_tiles_;
  const int map_size = map.map_width() * map.map_height();
  uszprintf( buf, 512, "City uses %d%%%% of the whole map", (sum_city_tiles*100)/map_size );
  textprintf_ex( db, font, x, y, fg_wht, -1, buf );

  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,255);
}

//----------------------------------------------------------------------------

void MainView::show_p_train_list( BITMAP*const db ) const
{
  const int fg_wht = makecol(255,255,255);
  const int dy = text_height( font ) + 2;
  const int x = area.x + area.w - 130;
  int y = area.y;
  set_trans_blender(0,0,0,150);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  rectfill( db, x, area.y, area.x+area.w, area.y+area.h, 0 );

  const std::vector<Train>&trains = map.get_trains();
  for( int it=0; it<trains.size(); ++it )
  {
    const Train&pt = trains[it];
    if( !pt.is_valid() )
      continue;
    if( pt.is_freight_train() )
      continue;
    textprintf_ex( db, font, x, y, fg_wht, -1, "Train %2d: %3dP", it, pt.n_passengers_ );
    y += dy;
  }

  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,255);
}

//----------------------------------------------------------------------------

void MainView::show_details_at_mouse( BITMAP*const db, const TileKOO&mouse_hover, int count_all_drawn_tiles ) const
{
  const int text_color = makecol(180,180,180);
  std::strstream line[5];
  // create text lines
  line[0] << mouse_x << "," << mouse_y << " -  " << mouse_hover << "    " << frst_x_ << "," << frst_y_ << "-" << last_x_ << "," << last_y_ << ":" << count_all_drawn_tiles << '\0';
  if( mouse_hover.x >= 0 && mouse_hover.y >= 0 )
  {
    const MapTile&mt = map.get_map_tile( mouse_hover );
    {
      std::strstream&str = line[1];
      str << "value=(" << mt.bas_value_ << "," << mt.add_value_ << ")";
      if( mt.n_persons_ > 0 )
        str << " pers=" << mt.n_persons_;
      if( mt.n_jobs_ > 0 )
        str << " jobs=" << mt.n_jobs_;
      if( mt.player_can_buy_ )
        str << " can_buy";
      if( mt.player_is_owner_ )
        str << " owner";
      if( mt.is_building_ )
        str << " building";
      if( mt.station_id_ >= 0 )
        str << " sta=" << mt.station_id_;
      if( mt.apartment_id_ >= 0 )
        str << " app=" << mt.apartment_id_;
      if( mt.office_id_ >= 0 )
        str << " off=" << mt.office_id_;
      str << " show=" << mt.train_show_;
      if( mt.train_id_ >= 0 )
      {
        if( mt.is_train_f_train_ )
          str << " f-train=" << mt.train_id_;
        else
          str << " p-train=" << mt.train_id_;
      }
      if( mt.can_hold_boxes_ )
        str << " " << mt.n_boxes_ << "bx";
      if( mt.nearest_station_id_ > 0 )
        str << " near_st=" << mt.nearest_station_id_;
      str << '\0';
    }
    {
      std::strstream&str = line[2];
      str << "cl=" << mt.tile_->class_idx_;
      str << " tl=" << mt.tile_->tile_idx_;
      if( mt.tile_->is_part_ )
        str << " part";
      str << " tcl=" << (int)mt.tcl_idx_;
      str << " tile=" << (int)mt.tile_idx_;
      str << " " << map.tile_mgr().class_of_tile( *mt.tile_ );
      str << '\0';
    }
    if( mt.station_id_ > 0 )
    {
      const Station&station = map.get_stations()[ mt.station_id_ ];
      {
        std::strstream&str = line[3];
        str << "Station " << mt.station_id_;
        str << " " << station.station_id_
          << " jobs=" << station.n_jobs_env_
          << " persons=" << station.n_persons_env_
          << " apps=" << station.n_apartment_env_
          << " offs=" << station.n_office_env_
          << " center=" << station.center_tile_
          << '\0';
      }
      {
        std::strstream&str = line[4];
        str << "pass(avg)=" << station.n_daily_passengers_counter_.average()
          << " good=" << station.n_good_passengers_
          << " miss=" << station.n_miss_passengers_
          << " q=(";
        for( int i=0; i<station.passenger_queue_.size() && i<station.passenger_queue_.count_down(); ++i )
          str << "," << std::setw(3) << station.passenger_queue_[i];
        str << ") " << station.passenger_queue_.peek_sum() << '\0';
      }
    }
    if( mt.apartment_id_ >= 0 )
    {
      std::strstream&str = line[3];
      str << "Apartment " << mt.apartment_id_;
      const Apartment&app = map.get_apartments()[ mt.apartment_id_ ];
      str << " " << app.apa_id_ << " val=" << app.value_ << " cur p/l=" << app.cost_rev_.cur_profit_loss() << '\0';
    }
    if( mt.office_id_ >= 0 )
    {
      std::strstream&str = line[3];
      str << "Office " << mt.office_id_;
      const Office&off = map.get_offices()[ mt.office_id_ ];
      str << " " << off.off_id_ << " val=" << off.value_ << " cur p/l=" << off.cost_rev_.cur_profit_loss() << '\0';
    }
  }

  // display text lines
  int th = 0;
  int tw = 0;
  for( int i=0; i<5; ++i )
  {
    const char*pc_line = line[i].str();
    if( !pc_line || strlen(pc_line)==0 )
      continue;
    const int twi = text_length( font, pc_line );
    if( tw < twi )
      tw = twi;
    if( twi > 0 )
      ++th;
  }
  const int thf = text_height( font ) + 2;
  th = th * thf + 2;

  int x = mouse_x - 8 - tw;
  int y = mouse_y - 8 - th - 2*text_height(font);
  if( x < area.x )
    x = area.x;
  if( y < area.y )
    y = area.y;

  set_trans_blender(0,0,0,150);
  drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );
  rectfill( db, x, y, x+tw, y+th, 0 );
  for( int i=0; i<5; ++i )
  {
    const char*pc_line = line[i].str();
    if( !pc_line || strlen(pc_line)==0 )
      continue;
    textprintf_ex( db, font, x, y+2+i*thf, text_color, -1, "%s", pc_line );
  }
  drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
  set_trans_blender(0,0,0,255);
}

//----------------------------------------------------------------------------

void MainView::notify_chg_size()
{
  frst_x_ = -1;
  frst_y_ = -1;
  last_x_ = -1;
  last_y_ = -1;
}

//----------------------------------------------------------------------------

void MainView::handle_kbd( int scan_code, int shift_flags )
{
  switch( scan_code )
  {
  case KEY_H:
    do_show_debug_details = !do_show_debug_details;
    break;
  case KEY_R:
    switch( map.get_game_state() )
    {
    case BuildStation:
      station_idx = (MultiTileIDX)((station_idx+1)%4);
      break;
    case PlaceFTrain:
    case PlacePTrain:
      train_reversed = !train_reversed;
      break;
    case BuildApartm:
    case BuildOffice:
      ++sub_idx;
      break;
    }
    break;
  case KEY_M:
    map.change_menu_state( map.menu_show_details_, !map.menu_show_minimap_ );
    notify_chg_size();
    break;
  case KEY_0:
    report_to_show = 0;
    break;
  case KEY_1:
    if( report_to_show != 1 )
      report_to_show = 1;
    else
      report_to_show = 0;
    break;
  case KEY_2:
    if( report_to_show != 2 )
      report_to_show = 2;
    else
      report_to_show = 0;
    break;
  case KEY_3:
    if( report_to_show != 3 )
      report_to_show = 3;
    else
      report_to_show = 0;
    break;
  case KEY_4:
    if( report_to_show != 4 )
      report_to_show = 4;
    else
      report_to_show = 0;
    break;
  }
}

bool MainView::handle_scroll()
{
  bool ret = false;
  if( key[KEY_RIGHT] )
  {
    offset.x -= map.tile_width()*cursor_move_step_width;
    ret = true;
  }
  if( key[KEY_LEFT] )
  {
    offset.x += map.tile_width()*cursor_move_step_width;
    ret = true;
  }
  if( key[KEY_UP] )
  {
    offset.y += map.tile_height()*cursor_move_step_width;
    ret = true;
  }
  if( key[KEY_DOWN] )
  {
    offset.y -= map.tile_height()*cursor_move_step_width;
    ret = true;
  }
  static ScreenKOO mouse_pos_drag_start(-1,-1);
  static bool drag_started = false;
  static ScreenKOO last_mouse_rb(-1,-1);
  const ScreenKOO mouse_pos( mouse_x, mouse_y );
  if(  (mouse_b & 2)
    && area.is_inside_rect( mouse_pos.x, mouse_pos.y )
    )
  {
    if( !drag_started )
    {
      if( mouse_pos_drag_start.x < 0 )
        mouse_pos_drag_start = mouse_pos;
      if(  ABS(mouse_pos_drag_start.x - mouse_pos.x ) > MOUSE_DRAG_THRESHOLD
        || ABS(mouse_pos_drag_start.y - mouse_pos.y ) > MOUSE_DRAG_THRESHOLD
        )
      {
        drag_started = true;
      }
    }
    if( drag_started )
    {
      if( last_mouse_rb.x >= 0 )
      {
        offset.x += (mouse_pos.x - last_mouse_rb.x)*mouse_scroll_sensitivity;
        offset.y += (mouse_pos.y - last_mouse_rb.y)*mouse_scroll_sensitivity;
      }
      last_mouse_rb.x = mouse_pos.x;
      last_mouse_rb.y = mouse_pos.y;
      ret = true;
    }
  }
  else
  {
    last_mouse_rb.x = -1;
  }
  if( !(mouse_b & 2) )
  {
    mouse_pos_drag_start.x = -1;
    drag_started = false;
  }
  if( ret )
  {
    notify_chg_size();
    const int y_max_limit = 0;
    if( offset.y > y_max_limit )
      offset.y = y_max_limit;
    const int y_min_limit = area.h - map.tile_height()*(map.map_height()+map.map_width());
    if( offset.y < y_min_limit )
      offset.y = y_min_limit;
    const int x_min_limit = area.w - map.tile_width()*(map.map_width()+1);
    if( offset.x < x_min_limit )
      offset.x = x_min_limit;
    const int x_max_limit = map.tile_width()*(map.map_height()-1);
    if( offset.x > x_max_limit )
      offset.x = x_max_limit;
  }
  return ret;
}

//----------------------------------------------------------------------------

