/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_TOOLS
#define INC_TOOLS
#endif

#ifndef INC_TYPES
#include "Types.h"
#endif

#ifndef INC_GAMELOOPIFC
#include "../gfwk/src/GameLoopIfc.h"
#endif

#include <vector>

//----------------------------------------------------------------------------

struct CostRevenueCollector
{
  CostRevenueCollector()
    : cur_cst_(0)
    , cst_last_day_(0)
    , cst_mtd_(0)
    , cst_last_month_(0)
    , cst_ytd_(0)
    , cst_last_year_(0)
    , cur_rev_(0)
    , rev_last_day_(0)
    , rev_mtd_(0)
    , rev_last_month_(0)
    , rev_ytd_(0)
    , rev_last_year_(0)
  {}

  void push_cost( int cst )
  {
    cur_cst_ += cst;
    cst_mtd_ += cst;
    cst_ytd_ += cst;
  }
  void push_revenue( int rev )
  {
    cur_rev_ += rev;
    rev_mtd_ += rev;
    rev_ytd_ += rev;
  }
  void update_day();
  void update_month();
  void update_year();

  void clear();
  void operator += ( const CostRevenueCollector&rhs );

  int cur_profit_loss() const { return cur_rev_ - cur_cst_; }
  int mtd_profit_loss() const { return rev_mtd_ - cst_mtd_; }
  int ytd_profit_loss() const { return rev_ytd_ - cst_ytd_; }

  int cur_cst_;
  int cst_last_day_;
  int cst_mtd_;
  int cst_last_month_;
  int cst_ytd_;
  int cst_last_year_;

  int cur_rev_;
  int rev_last_day_;
  int rev_mtd_;
  int rev_last_month_;
  int rev_ytd_;
  int rev_last_year_;
};

//----------------------------------------------------------------------------

template<int n>
struct AverageAggregator
{
  AverageAggregator()
    : cur_pos_(0)
  {
    for( int i=0; i<n; ++i )
      v_[i] = 0;
  }
  int average() const
  {
    int sum = 0;
    for( int i=0; i<n; ++i )
      sum += v_[i];
    return sum / n;
  }
  void push_value( int t )
  {
    v_[cur_pos_] = t;
    ++cur_pos_;
    if( cur_pos_ == n )
      cur_pos_ = 0;
  }
private:
  int v_[n];
  int cur_pos_;
};

//----------------------------------------------------------------------------

template<int n>
struct FixQueue
{
  FixQueue() : count_down_(0)
  {
    for( int i=0; i<n; ++i )
      v_[i] = 0;
  }
  int size() const { return n; }
  int operator[] ( int i ) const { return v_[i]; }
  int count_down() const { return count_down_; }
  int peek_sum() const
  {
    int rc = 0;
    for( int i=0; i<n; ++i )
      rc += v_[i];
    return rc;
  }

  int push_value( int t, int decay_percent )
  {
    if( count_down_>0 )
      --count_down_;
    const int rc = v_[0];
    for( int i=1; i<n; ++i )
    {
      v_[i-1] = ( v_[i] * ( 100 - decay_percent ) ) / 100;
    }
    v_[n-1] = t;
    return count_down_>0 ? rc : 0;
  }
  int pop_value_distributed( int t )
  {
    int rc = 0;
    int sum = 0;
    for( int i=0; i<n; ++i )
    {
      const int x = MIN( t/2, v_[i] );
      v_[i] -= x;
      sum += v_[i];
      t -= x;
      rc += x;
    }
    count_down_ = sum>0 ? 99*n : n;
    return rc;
  }
private:
  int v_[n];
  int count_down_;
};

//----------------------------------------------------------------------------

class Neighbourhood
{
public:
  Neighbourhood( int max_radius );
  size_t get_nb_le_size( size_t radius ) const
  {
    assertCondition( radius < nb_sizes_.size() );
    return nb_sizes_[radius];
  }
  size_t get_nb_eq_size( size_t radius ) const
  {
    assertCondition( radius < nb_eq_.size() );
    return nb_eq_[radius].size();
  }
  const TileKOO&get_nb_le( size_t radius, size_t i ) const
  {
    assertCondition( radius < nb_sizes_.size() );
    assertCondition( i < nb_sizes_[radius] );
    return nb_le_[i];
  }
  const TileKOO&get_nb_eq( size_t radius, size_t i ) const
  {
    assertCondition( radius < nb_sizes_.size() );
    assertCondition( i < nb_eq_[radius].size() );
    return nb_eq_[radius][i];
  }
private:
  std::vector<size_t> nb_sizes_;
  std::vector<TileKOO> nb_le_;
  std::vector< std::vector<TileKOO> > nb_eq_;
};

//----------------------------------------------------------------------------

template<class T>
int find_empty_slot( const std::vector<T>&v )
{
  {
    for( size_t i=0; i<v.size(); ++i )
      if( !v[i].is_valid() )
      {
        return i;
      }
  }
  return v.size();
}

//----------------------------------------------------------------------------
