#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "TileMgr.h"
#endif

#include <string>

//----------------------------------------------------------------------------

namespace
{

inline is_even( int a ) { return (a&1) == 0; }
inline is_odd( int a )  { return (a&1) == 1; }

} // namespace

//----------------------------------------------------------------------------

void MultiTile::set_tile( size_t ix, size_t iy, const Tile*pt )
{
  const size_t i = is_h_flipped_ ? index(iy,ix) : index(ix,iy);
  const size_t sz = tiles_.size();
  assertCondition( i < tiles_.size() );
  tiles_[i] = pt;
}

const Tile* MultiTile::get_tile( size_t ix, size_t iy ) const
{
  const size_t i = is_h_flipped_ ? index(iy,ix) : index(ix,iy);
  const size_t sz = tiles_.size();
  assertCondition( i < tiles_.size() );
  return tiles_[i];
}

//----------------------------------------------------------------------------

TileMgr::TileMgr()
: tile_w_(0), tile_h_(0)
{
}

TileMgr::~TileMgr()
{
  clear();
}

void TileMgr::clear()
{
  for( size_t i=0; i<tiles_.size(); ++i )
  {
    destroy_bitmap( tiles_[i]->bmp_ );
    delete tiles_[i];
  }
  tiles_.clear();
  destroy_bitmap( tileset_ );
  tileset_ = 0;
  class_names_.clear();
  tiles_by_class_id_.clear();
}

//----------------------------------------------------------------------------

void TileMgr::init( const char*ini_fn )
{
  assertCondition( ini_fn );

  push_config_state();
  set_config_file(ini_fn);

  std::string tileset_fn = get_config_string( "system", "tileset", 0 );

  pop_config_state();

  push_config_state();
  set_config_file( tileset_fn.c_str() );

  {
    int count = 0;
    const char*const *const data = get_config_argv("tileset", "dim", &count);
    if( count != 2 )
    {
      allegro_message( "Found %i parameters in tileset.dim instead of the 2 expected.\n", count );
      fatalError("no dim for tiles");
    }
    else
    {
      tile_w_ = atoi(data[0]);
      tile_h_ = atoi(data[1]);
      if( !is_even(tile_w_) )
        fatalError( "tile_w_ must be even" );
      if( !is_even(tile_h_) )
        fatalError( "tile_h_ must be even" );
      tile_w_ = tile_w_/2;
      tile_h_ = tile_h_/2;
      if( tile_w_==0 || tile_h_==0 )
        fatalError( "tw or th is 0" );
    }
  }

  const char*const fname = get_config_string( "tileset", "tileset", 0 );
  if( !fname )
    fatalError( "no tileset given" );

  tileset_ = load_bitmap( fname, NULL );
  if( !tileset_ )
    fatalError("Unable to load tileset!");

  const int n_tiles = get_config_int( "tiles", "anzahl", 0 );
  if( n_tiles<=0 )
    fatalError( "no tiles" );

  tiles_.reserve( 1+n_tiles );
  tiles_.push_back( new Tile() );

  for( int tcnt=1; tcnt<=n_tiles; ++tcnt )
  {
    int count = 0;
    char tilename[12]="";
    sprintf( tilename, "tile%d", tcnt );
    const char*const *const data = get_config_argv( "tiles", tilename, &count );
    if( count==0 )
      continue;
    if( count != 6 && count != 8 )
    {
      allegro_message( "Found %i parameters in tiles.%s instead of the 2 expected.\n", count, tilename );
      fatalError("tile spec incomplete");
    }
    const int ax = atoi(data[0]);
    const int ay = atoi(data[1]);
    const int aw = atoi(data[2]);
    const int ah = atoi(data[3]);
    const int ax_ext = count==8 ? atoi(data[4]) : 0;
    const int ay_ext = count==8 ? atoi(data[5]) : 0;
    const char af = data[count-2][0];
    const bool is_v_flipped = af=='v' || af=='b';
    const bool is_h_flipped = af=='h' || af=='b';
    int class_idx = id_of_class( data[count-1] );
    if( class_idx < 0 )
    {
      class_names_.push_back( data[count-1] );
      tiles_by_class_id_.push_back( std::vector<int>() );
      class_idx = class_names_.size()-1;
    }
    if( count==8 ) // tile consists of multiple parts
    {
      assertCondition( !is_v_flipped );
      assertCondition( ax_ext + ay_ext == aw );
      assertCondition( ax_ext + ay_ext <= ah );
      const size_t mt_idx = mtiles_.size();
      MultiTile mt( -1, mt_idx, is_h_flipped, ax_ext, ay_ext );
      for( int ix=0; ix<ax_ext; ++ix )
      {
        for( int iy=0; iy<ay_ext; ++iy )
        {
          const int tx = ay_ext-1+ix-iy;
          const int ty = (ix==0 || iy==0) ? 0 : (ah-aw+ix+iy);
          const int tw = 2;
          const int th = (ix==0 || iy==0) ? (ah-aw+2+ix+iy) : 2;
          BITMAP*const sub_bmp = create_sub_bitmap( tileset_, (ax+tx)*tile_w_, (ay+ty)*tile_h_, tw*tile_w_, th*tile_h_ );
          BITMAP*const sys_bmp = create_system_bitmap( sub_bmp->w, sub_bmp->h );
          clear_to_color( sys_bmp, makecol(255,0,255) );
          if( is_h_flipped )
            draw_sprite_h_flip( sys_bmp, sub_bmp, 0, 0 );
          else
            draw_sprite( sys_bmp, sub_bmp, 0, 0 );
          destroy_bitmap( sub_bmp );
          const Tile*pt = new Tile( sys_bmp, class_idx, tiles_.size(), true );
          if( mt.is_h_flipped_ )
            mt.set_tile( iy, ix, pt );
          else
            mt.set_tile( ix, iy, pt );
          tiles_.push_back( pt );
        }
      }
      mtiles_.push_back(mt);
    }
    else
    {
      BITMAP*const sub_bmp = create_sub_bitmap( tileset_, ax*tile_w_, ay*tile_h_, aw*tile_w_, ah*tile_h_ );
      BITMAP*const sys_bmp = create_system_bitmap( sub_bmp->w, sub_bmp->h );
      clear_to_color( sys_bmp, makecol(255,0,255) );
      if( is_h_flipped )
        if( is_v_flipped )
          draw_sprite_vh_flip( sys_bmp, sub_bmp, 0, -1 );
        else
          draw_sprite_h_flip( sys_bmp, sub_bmp, 0, 0 );
      else
        if( is_v_flipped )
          draw_sprite_v_flip( sys_bmp, sub_bmp, 0, -1 );
        else
          draw_sprite( sys_bmp, sub_bmp, 0, 0 );
      destroy_bitmap( sub_bmp );
      const int tile_idx = tiles_.size();
      tiles_.push_back( new Tile( sys_bmp, class_idx, tile_idx, false ) );
      tiles_by_class_id_[class_idx].push_back(tile_idx);
    }
  }

  pop_config_state();
}

int TileMgr::id_of_class( const char*class_name ) const
{
  for( size_t i=0; i<class_names_.size(); ++i )
    if( class_names_[i] == class_name )
      return i;
  return -1;
}

const char* TileMgr::class_of_tile( const Tile&tile ) const
{
  if( tile.class_idx_ >= class_names_.size() )
    return "unkown";
  assertCondition( tile.class_idx_ < class_names_.size() );
  return class_names_[ tile.class_idx_ ].c_str();
}

//----------------------------------------------------------------------------

