#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

//----------------------------------------------------------------------------

void WorldMap::set_box_tile( const TileKOO&t, bool player_is_owner, int n_boxes )
{
  MapTile&mt = set_general_tile( t, IDXempty, SUB_IDX_RANDOM, false, player_is_owner );
  mt.can_hold_boxes_ = true;
  mt.is_building_ = false;
  mt.bas_value_ = def_values_.bas_value_empty;
  chg_n_boxes( mt, n_boxes );
}

void WorldMap::chg_n_boxes( MapTile&mt, int diff_n_boxes )
{
  assertCondition( mt.can_hold_boxes_ );
  mt.n_boxes_ += diff_n_boxes;
  assertCondition(mt.n_boxes_>=0);
  assertCondition(mt.n_boxes_<=8);
  if( 0==mt.n_boxes_ )
  {
    const int tile_idx = tile_idxs_[IDXempty];
    mt.assign_tile( &tile_mgr_.tile_by_tile_idx( tile_idx ), IDXempty );
  }
  else
  {
    const int tile_idx = tile_idxs_[IDXboxes1+mt.n_boxes_-1];
    mt.assign_tile( &tile_mgr_.tile_by_tile_idx( tile_idx ), (TileClassIDX)(IDXboxes1+mt.n_boxes_-1) );
  }
}

void WorldMap::load_boxes_to_tile( MapTile&mt, Train&train )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_freight_train() );
  assertCondition( mt.n_boxes_ < 8 );
  assertCondition( train.n_boxes_ > 0 );
  const int nbox = MIN( 8-mt.n_boxes_, train.n_boxes_ );
  assertCondition( nbox > 0 && nbox <= 4 );
  chg_n_boxes( mt, nbox );
  train.n_boxes_ -= nbox;
}

void WorldMap::unload_boxes_from_tile( MapTile&mt, Train&train )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_freight_train() );
  assertCondition( mt.n_boxes_ > 0 );
  assertCondition( train.n_boxes_ < 4 );
  const int nbox = MIN( mt.n_boxes_, 4-train.n_boxes_ );
  assertCondition( nbox > 0 && nbox <= 4 );
  chg_n_boxes( mt, -nbox );
  train.n_boxes_ += nbox;
}

int WorldMap::unload_boxes_from_tile_for_construction( MapTile&mt, int n_boxes )
{
  assertCondition( mt.n_boxes_ > 0 );
  const int nbox = MIN( mt.n_boxes_, n_boxes );
  chg_n_boxes( mt, -nbox );
  return nbox;
}

void WorldMap::load_all_boxes_to_train( const TileKOO&tile, Train&train )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_freight_train() );
  const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_box_load_unload_station );
  for( size_t it=0; it<nb_size; ++it )
  {
    const TileKOO&dt = neighbourhood_.get_nb_le( def_values_.radius_box_load_unload_station, it );
    const TileKOO t = tile + dt;
    if( !is_on_map(t) )
      continue;
    MapTile&mt = map_tiles_[ index(t) ];
    if( !mt.can_hold_boxes_ )
      continue;
    if( mt.n_boxes_ == 0 )
      continue;
    unload_boxes_from_tile( mt, train );
    if( train.n_boxes_ == 4 )
      return;
  }
}

void WorldMap::unload_all_boxes_from_train( const TileKOO&tile, Train&train )
{
  assertCondition( train.is_valid() );
  assertCondition( train.is_freight_train() );
  const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_box_load_unload_station );
  for( size_t it=0; it<nb_size; ++it )
  {
    const TileKOO&dt = neighbourhood_.get_nb_le( def_values_.radius_box_load_unload_station, it );
    const TileKOO t = tile + dt;
    if( !is_on_map(t) )
      continue;
    MapTile&mt = map_tiles_[ index(t) ];
    if( !mt.can_hold_boxes_ )
      continue;
    if( mt.n_boxes_ == 8 )
      continue;
    load_boxes_to_tile( mt, train );
    if( train.n_boxes_ == 0 )
      return;
  }
}

//----------------------------------------------------------------------------

bool WorldMap::find_boxes_near( int n_boxes, const TileKOO&tile ) const
{
  int count_boxes = 0;
  const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_box_use_for_construction );
  for( size_t it=0; it<nb_size; ++it )
  {
    const TileKOO&dt = neighbourhood_.get_nb_le( def_values_.radius_box_use_for_construction, it );
    const TileKOO t = tile + dt;
    if( !is_on_map(t) )
      continue;
    const MapTile&mt = map_tiles_[ index(t) ];
    if( mt.n_boxes_ > 0 )
      count_boxes += mt.n_boxes_;
    if( count_boxes >= n_boxes )
      return true;
  }
  return false;
}

void WorldMap::unload_boxes_for_construction( int n_boxes, const TileKOO&tile )
{
  const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_box_use_for_construction );
  for( size_t it=0; it<nb_size && n_boxes>0; ++it )
  {
    const TileKOO&dt = neighbourhood_.get_nb_le( def_values_.radius_box_use_for_construction, it );
    const TileKOO t = tile + dt;
    if( !is_on_map(t) )
      continue;
    MapTile&mt = map_tiles_[ index(t) ];
    if( mt.n_boxes_ > 0 )
    {
      const n_boxes_unloaded = unload_boxes_from_tile_for_construction( mt, n_boxes );
      assertCondition( n_boxes_unloaded <= n_boxes );
      n_boxes -= n_boxes_unloaded;
    }
  }
  assertCondition( n_boxes == 0 );
}

//----------------------------------------------------------------------------
