#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

//----------------------------------------------------------------------------

Apartment::Apartment( const TileKOO&at, int aapa_id, int avalue, bool is_owner )
: center_tile_(at)
, apa_id_(aapa_id)
, player_is_owner_(is_owner)
, value_(avalue)
{
}

//----------------------------------------------------------------------------

int WorldMap::build_apa_find_cost( const TileKOO&t, int&sub_idx, Outline&apartm_shape ) const
{
  apartm_shape.clear();
  if( !is_inside_map(t) )
    return 0;
  const MapTile&mt = map_tiles_[ index(t) ];
  if( !mt.player_can_buy_ && !mt.player_is_owner_ )
    return 0;
  if( mt.is_rail() || mt.is_building_ )
    return 0;
  if( !find_boxes_near( def_values_.boxes_used_for_apartm_, t ) )
    return 0;
  int cost = 0;
  if( mt.player_can_buy_ && !mt.player_is_owner_ )
    cost += mt.value();
  cost += def_values_.cost_build_apartment + def_values_.cost_buy_apartment;
  if( cost <= cash_.player_total_ )
  {
    const int class_idx = class_idxs_[IDXapartmentPlayer1];
    const int n_alt_tiles = tile_mgr_.num_tiles_of_class_idx(class_idx);
    if( sub_idx < 0 || sub_idx >= n_alt_tiles )
      sub_idx = 0;
    const size_t tile_idx = tile_mgr().tile_idx_by_class_idx( class_idx, sub_idx );
    const Tile*const p_tile = &tile_mgr().tile_by_tile_idx( tile_idx );
    apartm_shape.push_back( OutlineDesc( t, p_tile ) );
    return cost;
  }
  return 0;
}

bool WorldMap::build_apa( const TileKOO&t, int sub_idx )
{
  assertCondition( is_on_map(t) );
  MapTile&mt = map_tiles_[ index(t) ];
  assertCondition( mt.player_can_buy_ || mt.player_is_owner_ );
  assertCondition( !mt.is_rail() );
  assertCondition( !mt.is_building_ );
  if( !find_boxes_near( def_values_.boxes_used_for_apartm_, t ) )
    return false;
  if( mt.player_can_buy_ && !mt.player_is_owner_ )
  {
    if( !buy_land(t,true) )
      return false;
  }
  assertCondition( mt.player_is_owner_ );
  const int cost = def_values_.cost_build_apartment + def_values_.cost_buy_apartment;
  cash_.apartm_re_.push_cost( cost );
  cash_.total_.push_cost( cost );
  cash_.player_total_ -= cost;

  const int app_id = find_empty_slot( v_apartments_ );

  const Apartment new_app( t, app_id, def_values_.cost_build_apartment, true );
  if( app_id >= v_apartments_.size() )
    v_apartments_.push_back( new_app );
  else
    v_apartments_[app_id] = new_app;

  unload_boxes_for_construction( def_values_.boxes_used_for_apartm_, t );
  set_general_tile( t, IDXapartmentPlayer1, sub_idx, mt.player_can_buy_, mt.player_is_owner_ );
  mt.apartment_id_ = app_id;
  mt.is_building_ = true;
  mt.can_hold_boxes_ = false;
  mt.n_persons_ = def_values_.min_persons_apartment_;
  mt.n_jobs_ = def_values_.n_jobs_apartment_;
  return true;
}

//----------------------------------------------------------------------------

int WorldMap::buy_apa_find_cost( const TileKOO&t ) const
{
  if( !is_inside_map(t) )
    return 0;
  const MapTile&mt = map_tiles_[ index(t) ];
  if( !mt.can_buy_apartment() )
    return 0;
  const Apartment&app = v_apartments_[ mt.apartment_id_ ];
  const int cost = app.value_ + mt.value() + def_values_.cost_buy_apartment;
  if( cost <= cash_.player_total_ )
    return cost;
  return 0;
}

bool WorldMap::buy_apa( const TileKOO&t )
{
  assertCondition( is_on_map(t) );
  MapTile&mt = map_tiles_[ index(t) ];
  assertCondition( mt.can_buy_apartment() );
  chg_maptile_only_tile( mt, IDXapartmentPlayer1, SUB_IDX_KEEPORIG );
  Apartment&app = v_apartments_[ mt.apartment_id_ ];
  app.player_is_owner_ = true;
  const int cost = app.value_ + def_values_.cost_buy_apartment;
  cash_.apartm_re_.push_cost( cost );
  cash_.total_.push_cost( cost );
  cash_.player_total_ -= cost;
  buy_land( t, false );
  return true;
}

//----------------------------------------------------------------------------

int WorldMap::sell_apa_find_revenue( const TileKOO&t ) const
{
  if( !is_inside_map(t) )
    return 0;
  const MapTile&mt = map_tiles_[ index(t) ];
  if( !mt.can_sell_apartment() )
    return 0;
  const Apartment&app = v_apartments_[ mt.apartment_id_ ];
  const int revenue = app.value_ + mt.value();
  return revenue;
}

bool WorldMap::sell_apa( const TileKOO&t )
{
  assertCondition( is_on_map(t) );
  MapTile&mt = map_tiles_[ index(t) ];
  assertCondition( mt.can_sell_apartment() );
  chg_maptile_only_tile( mt, IDXapartment1, SUB_IDX_KEEPORIG );
  Apartment&app = v_apartments_[ mt.apartment_id_ ];
  app.player_is_owner_ = false;
  const int revenue = app.value_;
  cash_.apartm_re_.push_revenue( revenue );
  cash_.total_.push_revenue( revenue );
  cash_.player_total_ += revenue;
  sell_land( t );
  return true;
}

//----------------------------------------------------------------------------

bool WorldMap::delete_apa( const TileKOO&t )
{
  assertCondition( is_on_map(t) );
  MapTile&mt = map_tiles_[ index(t) ];
  if( mt.player_is_owner_ )
  {
    assertCondition( mt.can_sell_apartment() );
  }
  Apartment&app = v_apartments_[ mt.apartment_id_ ];
  set_general_tile( t, IDXempty, SUB_IDX_RANDOM, mt.player_can_buy_, mt.player_is_owner_ );
  app.clear();
  if( mt.player_is_owner_ )
  {
    const int cost = def_values_.cost_delete_apartment;
    cash_.apartm_re_.push_cost( cost );
    cash_.total_.push_cost( cost );
    cash_.player_total_ -= cost;
  }
  return true;
}

//----------------------------------------------------------------------------

void WorldMap::update_apartments()
{
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    MapTile&mt = map_tiles_[ index(app.center_tile_) ];
    // change number of persons
    if( mt.add_value_ > mt.bas_value_ )
      mt.n_persons_ += mt.add_value_ / mt.bas_value_;
    if( mt.n_persons_ < def_values_.max_persons_apartment_ / 3 )
      mt.n_persons_ += 1;
    if( mt.nearest_station_id_ >= 0 )
    {
      const Station&station = v_stations_[mt.nearest_station_id_];
      if( station.n_apartment_env_ > 0 )
      {
        const int d_persons = ( station.n_daily_passengers_counter_.average() + station.n_jobs_env_ ) / station.n_apartment_env_;
        if( d_persons < 40 )
          mt.n_persons_ -= ( 50 - d_persons ) / 5;
        else
          mt.n_persons_ += MIN( ( 30 + d_persons ) / 50, def_values_.max_persons_apartment_/20 );
      }
    }
    // limit number of persons
    if( mt.n_persons_ < def_values_.min_persons_apartment_ )
      mt.n_persons_ = def_values_.min_persons_apartment_;
    else if( mt.n_persons_ > def_values_.max_persons_apartment_ )
      mt.n_persons_ = def_values_.max_persons_apartment_;
  }
}

//----------------------------------------------------------------------------


