/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#define INC_WORLDMAP
#endif

#ifndef INC_TILEMGR
#include "TileMgr.h"
#endif
#ifndef INC_GAMECLOCK
#include "../gfwk/src/GameClock.h"
#endif
#ifndef INC_STRINGMGR
#include "StringMgr.h"
#endif
#ifndef INC_TYPES
#include "Types.h"
#endif
#ifndef INC_TOOLS
#include "Tools.h"
#endif

#include <vector>
#include <set>

#define MOUSE_DRAG_THRESHOLD 3

//----------------------------------------------------------------------------

enum TileClassIDX
{
  IDXrail_nnss
, IDXrail_wwoo
, IDXrail_nwso
, IDXrail_nosw
, IDXrail_straight = IDXrail_nosw

, IDXrail_wwso
, IDXrail_swoo
, IDXrail_noww
, IDXrail_oonw
, IDXrail_sonn
, IDXrail_nnsw
, IDXrail_ssno
, IDXrail_nwss
, IDXraillast = IDXrail_nwss

, IDXlok_nnss
, IDXlok_wwoo
, IDXlok_nwso
, IDXlok_nosw

, IDXpwag_nnss
, IDXpwag_wwoo
, IDXpwag_nwso
, IDXpwag_nosw

, IDXfwag0_nnss
, IDXfwag0_wwoo
, IDXfwag0_nwso
, IDXfwag0_nosw

, IDXfwag1_nnss
, IDXfwag1_wwoo
, IDXfwag1_nwso
, IDXfwag1_nosw

, IDXfwag2_nnss
, IDXfwag2_wwoo
, IDXfwag2_nwso
, IDXfwag2_nosw

, IDXempty
, IDXgras
, IDXsmall_house
, IDXlarge_house
, IDXwater
, IDXdark

, IDXboxes1
, IDXboxes2
, IDXboxes3
, IDXboxes4
, IDXboxes5
, IDXboxes6
, IDXboxes7
, IDXboxes8

, IDXapartment1
, IDXapartmentPlayer1

, IDXoffice1
, IDXofficePlayer1
};

//----------------------------------------------------------------------------

#define SUB_IDX_RANDOM -1
#define SUB_IDX_KEEPORIG -2

//----------------------------------------------------------------------------

enum Direction
{
  DIR_nn, DIR_no, DIR_oo, DIR_so, DIR_ss, DIR_sw, DIR_ww, DIR_nw, DIR_none
};

inline Direction opposite_dir( Direction dir )
{
  return (Direction)( (dir+4) % 8 );
}

//----------------------------------------------------------------------------

enum TrainShow
{
  NoTrain, PersLok, PersWaggon, FreightLok, Freight0, Freight1, Freight2, TrainShow_size
};

//----------------------------------------------------------------------------

enum GameState
{
  NormalSimulation
, BuyLand
, SellLand
, LayTrack
, DeleteTrack
, BuildStation
, PlaceFTrain
, PlacePTrain
, BuildApartm
, BuildOffice
};

//----------------------------------------------------------------------------

struct MapTile
{
  MapTile() : tile_(0), add_value_(0) {}
  MapTile( const Tile* tile, TileClassIDX tcl_idx, int bas_value, int n_persons );
  void assign_tile( const Tile*p_tile, TileClassIDX tcl_idx )
  {
    tcl_idx_ = tcl_idx;
    tile_idx_ = p_tile->tile_idx_;
    tile_ = p_tile;
  }
  bool is_rail() const { return tcl_idx_<=IDXraillast; }
  bool is_station_rail() const      { return station_id_ >= 0 && !is_building_; }
  bool is_station_building() const  { return station_id_ >= 0 && is_building_; }
  bool can_build_rail() const       { return ( player_can_buy_ || player_is_owner_ ) && !is_rail() && !is_building_; }
  bool can_delete_station() const   { return player_is_owner_ && is_station_building(); }
  bool can_build_apartment() const  { return ( player_can_buy_ || player_is_owner_ ) && !is_rail() && !is_building_; }
  bool can_buy_apartment() const    { return player_can_buy_ && apartment_id_ >= 0; }
  bool can_sell_apartment() const   { return player_is_owner_ && apartment_id_ >= 0; }
  bool can_build_office() const     { return ( player_can_buy_ || player_is_owner_ ) && !is_rail() && !is_building_; }
  bool can_buy_office() const       { return player_can_buy_ && office_id_ >= 0; }
  bool can_sell_office() const      { return player_is_owner_ && office_id_ >= 0; }
  int value() const { return bas_value_ + add_value_; }

  const Tile* tile_;
  TileClassIDX tcl_idx_;
  int tile_idx_;
  TrainShow train_show_;
  bool player_can_buy_;
  bool player_is_owner_;
  bool can_hold_boxes_;
  bool is_building_;
  bool is_train_f_train_;
  int nearest_station_id_;
  int n_boxes_;
  int n_persons_;
  int n_jobs_;
  int train_id_;
  int station_id_;
  int office_id_;
  int apartment_id_;
  int bas_value_;
  int add_value_;
  int tmp_value_;
};

//----------------------------------------------------------------------------

class Train
{
public:
  Train() {}
  Train( int train_id, bool is_f_train, int max_speed_10th_tiles_per_hour, int station_time_10th_of_hour, int cur_game_time );

  typedef TileKOO Positions[3];

  bool is_valid() const { return max_speed_10th_tiles_per_hour_ > 0; }
  bool is_freight_train() const { return is_freight_train_; }
  bool is_passenger_train() const { return !is_freight_train_; }
  TrainShow get_train_show( int i ) const;
  void clear()
  {
    cur_speed_10th_tiles_per_hour_ = 0;
    max_speed_10th_tiles_per_hour_ = 0;
    train_id_ = -1;
  }
  bool needs_update( int cur_game_time ) const;
  void update_time( int cur_game_time );
  void reduce_speed_entering_station( int parts_of_train_in_station );
  void increase_speed_leaving_station();
  void reset_game_time();
  void reverse();

  Positions pos_;
  CostRevenueCollector cost_rev_;
  int train_id_;
  bool is_stopped_at_station_;
  // freight train only
  int n_boxes_;
  // passenger train only
  int n_passengers_;
  int last_visited_station_id_;
protected:
  bool is_freight_train_;
public:
  int cur_speed_10th_tiles_per_hour_;
  int max_speed_10th_tiles_per_hour_;
protected:
  int station_time_10th_of_hour_;
  int last_moved_;
};

class ExtTrain : public Train
{
public:
  ExtTrain( int p_max_speed, int p_station_time, int f_max_speed, int f_station_time );
  void switch_train();
  void reset_game_time();

  bool is_on_map_;
  bool is_importing_;
private:
  const int p_max_speed_10th_tiles_per_hour_;
  const int f_max_speed_10th_tiles_per_hour_;
  const int p_station_time_10th_of_hour_;
  const int f_station_time_10th_of_hour_;
};

//----------------------------------------------------------------------------

class Station
{
public:
  Station() : station_id_(-1) {}
  Station( const TileKOO&ref_tile, const TileKOO&center_tile, MultiTileIDX mt_idx, int station_id );
  bool is_valid() const { return station_id_ >= 0; }
  void clear() { station_id_ = -1; ref_tile_.clear(); center_tile_.clear(); }

  int station_id_;
  MultiTileIDX mt_idx_;
  TileKOO ref_tile_;
  TileKOO center_tile_;
  int n_persons_env_;
  int n_jobs_env_;
  int n_good_passengers_;
  int n_miss_passengers_;
  int n_apartment_env_;
  int n_office_env_;
  FixQueue<8> passenger_queue_;
  AverageAggregator<7> n_daily_passengers_counter_;
};

//----------------------------------------------------------------------------

class Apartment
{
public:
  Apartment() : apa_id_(-1) {}
  Apartment( const TileKOO&at, int aapp_id, int avalue, bool is_owner );
  bool is_valid() const { return apa_id_ >= 0; }
  void clear() { apa_id_ = -1; center_tile_.clear(); }

  int apa_id_;
  TileKOO center_tile_;
  bool player_is_owner_;
  int value_;
  CostRevenueCollector cost_rev_;
};

//----------------------------------------------------------------------------

class Office
{
public:
  Office() : off_id_(-1) {}
  Office( const TileKOO&at, int aoff_id, int avalue, bool is_owner );
  bool is_valid() const { return off_id_ >= 0; }
  void clear() { off_id_ = -1; center_tile_.clear(); }

  int off_id_;
  TileKOO center_tile_;
  bool player_is_owner_;
  int value_;
  CostRevenueCollector cost_rev_;
};

//----------------------------------------------------------------------------

struct RailPathDesc
{
  RailPathDesc( const TileKOO&at, TileClassIDX atcl ) : t_(at), tcl_(atcl) {}
  TileKOO t_;
  TileClassIDX tcl_;
};

typedef std::vector<RailPathDesc> RailPath;

struct OutlineDesc
{
  OutlineDesc( const TileKOO&at, const Tile*ap_tile ) : t_(at), p_tile_(ap_tile) {}
  TileKOO t_;
  const Tile*p_tile_;
};

typedef std::vector<OutlineDesc> Outline;

//----------------------------------------------------------------------------

struct Cash
{
  Cash( int player_initial );

  CostRevenueCollector trains_op_;
  CostRevenueCollector trains_re_;
  CostRevenueCollector land_;
  CostRevenueCollector apartm_op_;
  CostRevenueCollector apartm_re_;
  CostRevenueCollector office_op_;
  CostRevenueCollector office_re_;
  CostRevenueCollector interest_;
  CostRevenueCollector total_;
  int player_total_;
  int taxes_to_pay_next_april_;

  int taxes_estimate_ytd( int percent ) const;
  void pay_taxes();
  void update_day();
  void update_month();
  void update_year( int tax_for_last_year );
};

//----------------------------------------------------------------------------

struct Assets
{
  Assets();
  int sum_land_value_;
  int sum_train_value_;
  int sum_apartments_value_;
  int sum_office_value_;
  int all_inhabitants_;
  int my_apartments_;
  int my_offices_;
  int my_land_tiles_;
  int my_rail_tiles_;
  int all_land_tiles_;
  int all_person_tiles_;
  int all_office_tiles_;
  int all_rail_tiles_;

  int total() const;
  int taxes_estimate( int percent ) const;
  void clear();
};

//----------------------------------------------------------------------------

class WorldMap : public GameClockIfc, public SysMenuIfc
{
  struct Defaults
  {
    Defaults();
    int menu_small_width_;
    int menu_large_width_;
    int initial_total_cash_;
    int f_speed_tenth_tiles_per_hour;
    int p_speed_tenth_tiles_per_hour;
    int f_station_time_tenth_of_hour;
    int p_station_time_tenth_of_hour;
    int bas_value_empty;
    int bas_value_gras;
    int bas_value_water;
    int bas_value_small_house;
    int bas_value_large_house;
    int cost_build_station;
    int cost_delete_station;
    int cost_lay_track;
    int cost_delete_track;
    int cost_place_train;
    int revenue_remove_train;
    int cost_build_apartment;
    int cost_buy_apartment;
    int cost_delete_apartment;
    int cost_build_office;
    int cost_buy_office;
    int cost_delete_office;
    int radius_box_load_unload_station;
    int radius_box_use_for_construction;
    int radius_station_count;
    int radius_add_value;
    int n_persons_small_house;
    int n_persons_large_house;
    int n_jobs_station;
    int boxes_used_for_small_house;
    int boxes_used_for_large_house;
    int revenue_train_fix_per_10passengers_;
    int revenue_train_var_per_10passengers_per_10tiles_;
    int cost_station_per_hour_less_500_;
    int cost_station_per_hour_less_2500_;
    int cost_station_per_hour_less_10000_;
    int cost_station_per_hour_more_10000_;
    int cost_train_per_hour_;
    int max_passengers_train_;
    int n_jobs_apartment_;
    int max_persons_apartment_;
    int min_persons_apartment_;
    int max_jobs_office_;
    int min_jobs_office_;
    int boxes_used_for_apartm_;
    int boxes_used_for_office_;
    int cost_apartm_per_day_;
    int revenue_apartm_per_10persons_per_day_;
    int cost_office_per_day_;
    int revenue_office_per_10persons_per_day_;
    int station_passenger_queue_decay_;
    int tax_rate_profit_;
    int tax_rate_assets_;
    int interest_rate_per_million_;
  };
public:
  WorldMap( GameClock&, const TileMgr&, const StringMgr& );
  ~WorldMap();
  void attach_viewports( ViewPortBase&mm, ViewPortBase&mv )
  {
    p_main_menu_ = &mm;
    p_main_view_ = &mv;
  }
  void init( const char*ini_fn );
  // GameClockIfc
  virtual void advance_game_time();
  // SysMenuIfc
  virtual void new_game();
  virtual void save_game();
  virtual void load_game();
  virtual bool can_continue() const;
  virtual void save_autosave();
  virtual void load_autosave();
  virtual bool exists_autosave();

  int map_width() const    { return map_x_ext_; }
  int map_height() const   { return map_y_ext_; }
  int index( const TileKOO&t ) const { return t.x+map_x_ext_*t.y; }
  int tile_width() const   { return tile_mgr_.tile_width(); }
  int tile_height() const  { return tile_mgr_.tile_height(); }
  bool is_on_map( const TileKOO&t ) const { return t.x>=0 && t.x<map_x_ext_ && t.y>=0 && t.y<map_y_ext_; }
  bool is_inside_map( const TileKOO&t ) const { return t.x>=3 && t.x<map_x_ext_-3 && t.y>=3 && t.y<map_y_ext_-3; }
  const TileMgr& tile_mgr() const { return tile_mgr_; }

  void generate_random_map( unsigned int seed, size_t aw, size_t ah );
  const Tile& get_tile( TileClassIDX ) const;
  const Tile& get_train_overlay( TileClassIDX, TrainShow ) const;
  const MapTile& get_map_tile( const TileKOO& ) const;

  const std::vector<Train>& get_trains() const { return v_trains_; }
  const std::vector<Station>& get_stations() const { return v_stations_; }
  const std::vector<Apartment>& get_apartments() const { return v_apartments_; }
  const std::vector<Office>& get_offices() const { return v_offices_; }

  void change_game_state( GameState new_game_state );
  GameState get_game_state() const { return game_state_; }
  void change_menu_state( bool show_details, bool show_minimap );
  bool stop_if_build() const { return stop_if_build_; }

  // handle land
  int buy_land_find_costs( const TileKOO&t ) const;
  bool buy_land( const TileKOO&t, bool make_box_tile );
  int sell_land_find_revenue( const TileKOO&t ) const;
  bool sell_land( const TileKOO&t );

  // handling tracks
  int lay_rail_find_path( const TileKOO&t_start, const TileKOO&t_end, RailPath&rail_path ) const;
  void lay_rail( const RailPath&rail_path );
  int delete_rail_find_cost( const TileKOO&t ) const;
  void delete_rail( const TileKOO&t );

  // handling stations
  int build_station_find_pos( const TileKOO&t, MultiTileIDX mt_idx, Outline&station_shape ) const;
  void build_station( const TileKOO&t, MultiTileIDX mt_idx, bool player_is_owner );
  TileKOO center_tile_of_station( const TileKOO&ref_tile, MultiTileIDX mt_idx ) const;
  void delete_station( int station_id );

  // handle trains
  int place_train_find_pos( const TileKOO&t, bool train_reversed, bool is_freight_train, Outline&train_shape ) const;
  void place_freight_train( const Outline&train_shape );
  void place_pers_train( const Outline&train_shape );
  void remove_train( size_t train_id );
  void reverse_train( bool is_freight_train, int train_id );

  // handle train - station pair
  bool does_train_stop_at_station( int train_id, int station_id ) const;
  void toggle_stop_train_at_station( int train_id, int station_id );

  // handle apartments
  int build_apa_find_cost( const TileKOO&t, int&sub_idx, Outline&apartm_shape ) const;
  bool build_apa( const TileKOO&t, int sub_idx );
  int buy_apa_find_cost( const TileKOO&t ) const;
  bool buy_apa( const TileKOO&t );
  int sell_apa_find_revenue( const TileKOO&t ) const;
  bool sell_apa( const TileKOO&t );
  bool delete_apa( const TileKOO&t );

  // handle offices
  int build_off_find_cost( const TileKOO&t, int&sub_idx, Outline&office_shape ) const;
  bool build_off( const TileKOO&t, int sub_idx );
  int buy_off_find_cost( const TileKOO&t ) const;
  bool buy_off( const TileKOO&t );
  int sell_off_find_revenue( const TileKOO&t ) const;
  bool sell_off( const TileKOO&t );
  bool delete_off( const TileKOO&t );

  // handle money
  const Cash& get_player_cash() const { return cash_; }
  const Assets& get_player_assets() const { return assets_; }

  // handle selection
  void reset_selection();
  bool is_anything_selected() const;

private:
  void chg_maptile_only_tile( MapTile&mt, TileClassIDX tcl_idx, int sub_idx );
  MapTile& set_general_tile( const TileKOO&t, TileClassIDX tcl_idx, int sub_idx, bool player_can_buy, bool player_is_owner );
  MapTile& set_part_of_multi_tile( const TileKOO&t, MultiTileIDX mt_idx, const TileKOO&part, bool player_can_buy, bool player_is_owner );

  // handling train simulation
  bool is_train_complete_off_map( const Train& ) const;
  bool is_train_complete_on_map( const Train& ) const;
  int is_train_complete_in_station( const Train& ) const;
  int count_train_parts_in_station( const Train& ) const;
  void show_train( const TileKOO&t, TrainShow, int train_id, bool is_f_train );
  void advance_ext_train_to_next_tile( ExtTrain&train );
  void advance_train_to_next_tile( Train&train, bool allow_reverse );
  int advance_ext_train( int cur_game_time );
  int advance_f_train( Train&train, int cur_game_time );
  int advance_p_train( Train&train, int cur_game_time );

  // handling boxes
  void set_box_tile( const TileKOO&t, bool player_is_owner, int n_boxes );
  void chg_n_boxes( MapTile&mt, int diff_n_boxes );
  void load_all_boxes_to_train( const TileKOO&tile, Train&train );
  void unload_all_boxes_from_train( const TileKOO&tile, Train&train );
  void load_boxes_to_tile( MapTile&mt, Train&train );
  void unload_boxes_from_tile( MapTile&mt, Train&train );
  bool find_boxes_near( int n_boxes, const TileKOO&tile ) const;
  int unload_boxes_from_tile_for_construction( MapTile&mt, int n_boxes );
  void unload_boxes_for_construction( int n_boxes, const TileKOO&tile );

  // handling tracks
  const TileKOO find_next_rail_tile( Train::Positions& ) const;
  bool is_rail_valid_here( const TileKOO&t, TileClassIDX tcl ) const;
  bool is_rail_compatible( TileClassIDX t, TileClassIDX u, Direction dir ) const;
  int count_connecting_rail_neighbours( const TileKOO&t ) const;
  int find_connecting_rail_neighbours( const TileKOO&t, std::vector<Direction>&neighbour_directions ) const;
  int find_connecting_rail_neighbours( const TileKOO&t, std::vector<TileKOO>&neighbour_tiles ) const;
  Direction lay_rail_find_straight_line( const TileKOO&t_start, const TileKOO&t_end, RailPath&rail_path, bool include_first ) const;

  // handling stations
  bool eval_station_building( const TileKOO&ti, int&cost ) const;
  void build_station_building_tile( const TileKOO&ti, bool player_is_owner, const Tile*p_tile );

  void remap_all_stations_buildings();

  // building houses
  bool build_small_house( const TileKOO&t );
  bool build_large_house( const TileKOO&t );

  // handling regular updates
  void update_hour();
  void update_day();
  void update_month();
  void update_year();
  void update_value_and_assets();
  void update_build_decision();
  void update_apartments();
  void update_offices();

  // other functions
  bool force_sale();

  // save game and load game
  void save_to_file( PACKFILE* ) const;
  int load_from_file( PACKFILE* );

public:
  GameClock& game_clock_;
  const Neighbourhood neighbourhood_;
  const StringMgr&str_mgr_;
  Defaults def_values_;
  // money
  Cash cash_;
  Assets assets_;
  // infos to display in other viewports
  std::string bot_state_string_;
  std::string bot_info_string_;
  // selection
  int selected_train_;
  int selected_station_;
  int selected_apartment_;
  int selected_office_;
  // menu state
  bool menu_show_details_;
  bool menu_show_minimap_;
private:
  const TileMgr& tile_mgr_;
  bool stop_if_build_;

  std::vector<int> class_idxs_;
  std::vector<int> tile_idxs_;
  std::vector<MapTile> map_tiles_;
  int map_x_ext_;
  int map_y_ext_;
  TileKOO ext_entry_;
  ExtTrain ext_train_;
  std::vector<Station> v_stations_;
  std::vector<Train> v_trains_;
  std::vector<Apartment> v_apartments_;
  std::vector<Office> v_offices_;

  typedef std::pair<int,int> TrainStationPair;
  typedef std::set< TrainStationPair > TrainStationSet;
  TrainStationSet train_station_set_;

  GameState game_state_;
  ViewPortBase* p_main_menu_;
  ViewPortBase* p_main_view_;

  int current_hour_since0_;
  int current_day_since0_;
  int current_month_;
  int current_year_since0_;
  int last_value_update_time_;
  int last_build_decision_time_;
  int last_update_buildings_time_;
};

//----------------------------------------------------------------------------
