#include "StdAfx.h"

/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_WORLDMAP
#include "WorldMap.h"
#endif

#ifndef INC_MTRAND
#include "../gfwk/src/mtrand.h"
#endif

//----------------------------------------------------------------------------

WorldMap::Defaults::Defaults()
: menu_small_width_(72)
, menu_large_width_(184)
, initial_total_cash_(5000000)
, f_speed_tenth_tiles_per_hour(30)
, p_speed_tenth_tiles_per_hour(60)
, f_station_time_tenth_of_hour(12)
, p_station_time_tenth_of_hour(8)
, bas_value_empty( 1000 )
, bas_value_gras( 1200 )
, bas_value_water( 1300 )
, bas_value_small_house( 2000 )
, bas_value_large_house( 5000 )
, cost_build_station( 120000 )
, cost_delete_station( 12000 )
, cost_lay_track( 300 )
, cost_delete_track( 200 )
, cost_place_train( 80000 )
, revenue_remove_train( 40000 )
, cost_build_apartment( 340000 )
, cost_buy_apartment( 17000 )
, cost_delete_apartment( 50000 )
, cost_build_office( 860000 )
, cost_buy_office( 43000 )
, cost_delete_office( 75000 )
, radius_box_load_unload_station( 7 )
, radius_box_use_for_construction( 10 )
, radius_station_count( 8 )
, radius_add_value( 8 )
, n_persons_small_house( 5 )
, n_persons_large_house( 20 )
, n_jobs_station( 30 )
, boxes_used_for_small_house(2)
, boxes_used_for_large_house(5)
, revenue_train_fix_per_10passengers_(15)
, revenue_train_var_per_10passengers_per_10tiles_(12)
, cost_train_per_hour_(120)
, cost_station_per_hour_less_500_(10)
, cost_station_per_hour_less_2500_(40)
, cost_station_per_hour_less_10000_(150)
, cost_station_per_hour_more_10000_(500)
, max_passengers_train_(800)
, n_jobs_apartment_(10)
, max_persons_apartment_(400)
, min_persons_apartment_(40)
, max_jobs_office_(600)
, min_jobs_office_(30)
, boxes_used_for_apartm_(8)
, boxes_used_for_office_(12)
, cost_apartm_per_day_(186)
, revenue_apartm_per_10persons_per_day_(6)
, cost_office_per_day_(471)
, revenue_office_per_10persons_per_day_(10)
, station_passenger_queue_decay_(10)
, tax_rate_profit_(50)
, tax_rate_assets_(5)
, interest_rate_per_million_(500)
{
}

//----------------------------------------------------------------------------

MapTile::MapTile( const Tile* p_tile, TileClassIDX tcl_idx, int bas_value, int n_persons )
: tile_(p_tile)
, tcl_idx_(tcl_idx)
, tile_idx_( p_tile ? p_tile->tile_idx_ : -1 )
, train_show_(NoTrain)
, player_can_buy_(true), player_is_owner_(false), can_hold_boxes_(false), is_building_(false)
, nearest_station_id_(-1)
, n_boxes_(0), n_persons_(n_persons), n_jobs_(0)
, train_id_(-1), is_train_f_train_(false), station_id_(-1), office_id_(-1), apartment_id_(-1)
, bas_value_(bas_value), add_value_(0), tmp_value_(0)
{
}

//----------------------------------------------------------------------------

Cash::Cash( int player_initial )
: player_total_(player_initial)
, taxes_to_pay_next_april_(0)
{
}

int Cash::taxes_estimate_ytd( int percent ) const
{
  int tax = ( total_.rev_ytd_ - total_.cst_ytd_ ) * ( percent / 100.0 );
  if( tax < 0 )
    tax = 0;
  return tax;
}

void Cash::pay_taxes()
{
  player_total_ -= taxes_to_pay_next_april_;
  taxes_to_pay_next_april_ = 0;
}

void Cash::update_day()
{
  trains_op_.update_day();
  trains_re_.update_day();
  land_.update_day();
  apartm_op_.update_day();
  apartm_re_.update_day();
  office_op_.update_day();
  office_re_.update_day();
  interest_.update_day();
  total_.update_day();
}

void Cash::update_month()
{
  trains_op_.update_month();
  trains_re_.update_month();
  land_.update_month();
  apartm_op_.update_month();
  apartm_re_.update_month();
  office_op_.update_month();
  office_re_.update_month();
  interest_.update_month();
  total_.update_month();
}

void Cash::update_year( int tax_for_last_year )
{
  trains_op_.update_year();
  trains_re_.update_year();
  land_.update_year();
  apartm_op_.update_year();
  apartm_re_.update_year();
  office_op_.update_year();
  office_re_.update_year();
  interest_.update_year();
  total_.update_year();
  taxes_to_pay_next_april_ = tax_for_last_year;
}

//----------------------------------------------------------------------------

Assets::Assets()
: sum_land_value_(0)
, sum_train_value_(0)
, sum_apartments_value_(0)
, sum_office_value_(0)
, all_inhabitants_(0)
, my_apartments_(0)
, my_offices_(0)
, my_land_tiles_(0)
, my_rail_tiles_(0)
, all_land_tiles_(0)
, all_person_tiles_(0)
, all_office_tiles_(0)
, all_rail_tiles_(0)
{
}

int Assets::total() const
{
  return sum_land_value_ + sum_train_value_ + sum_apartments_value_ + sum_office_value_;
}

int Assets::taxes_estimate( int percent ) const
{
  return total() * ( percent / 100.0 );
}

void Assets::clear()
{
  sum_land_value_ = 0;
  sum_train_value_ = 0;
  sum_apartments_value_ = 0;
  sum_office_value_ = 0;
  all_inhabitants_ = 0;
  my_apartments_ = 0;
  my_offices_ = 0;
  my_land_tiles_ = 0;
  my_rail_tiles_ = 0;
  all_land_tiles_ = 0;
  all_person_tiles_ = 0;
  all_office_tiles_ = 0;
  all_rail_tiles_ = 0;
}

//----------------------------------------------------------------------------

#define TNT_CFG( name, val ) val = get_config_int( "tnt", name, val )

void WorldMap::init( const char*ini_fn )
{
  assertCondition( ini_fn );

  push_config_state();
  set_config_file( ini_fn );

  if( get_config_int( "options", "stop_if_build", 0 ) != 0 )
    stop_if_build_ = true;
  else
    stop_if_build_ = false;

  TNT_CFG( "menu_small_width", def_values_.menu_small_width_ );
  TNT_CFG( "menu_large_width", def_values_.menu_large_width_ );
  TNT_CFG( "initial_total_cash", def_values_.initial_total_cash_ );
  TNT_CFG( "f_speed_tenth_tiles_per_hour", def_values_.f_speed_tenth_tiles_per_hour );
  TNT_CFG( "p_speed_tenth_tiles_per_hour", def_values_.p_speed_tenth_tiles_per_hour );
  TNT_CFG( "f_station_time_tenth_of_hour", def_values_.f_station_time_tenth_of_hour );
  TNT_CFG( "p_station_time_tenth_of_hour", def_values_.p_station_time_tenth_of_hour );
  TNT_CFG( "bas_value_empty", def_values_.bas_value_empty );
  TNT_CFG( "bas_value_gras", def_values_.bas_value_gras );
  TNT_CFG( "bas_value_water", def_values_.bas_value_water );
  TNT_CFG( "bas_value_small_house", def_values_.bas_value_small_house );
  TNT_CFG( "bas_value_large_house", def_values_.bas_value_large_house );
  TNT_CFG( "cost_build_station", def_values_.cost_build_station );
  TNT_CFG( "cost_delete_station", def_values_.cost_delete_station );
  TNT_CFG( "cost_lay_track", def_values_.cost_lay_track );
  TNT_CFG( "cost_delete_track", def_values_.cost_delete_track );
  TNT_CFG( "cost_place_train", def_values_.cost_place_train );
  TNT_CFG( "revenue_remove_train", def_values_.revenue_remove_train );
  TNT_CFG( "cost_build_apartment", def_values_.cost_build_apartment );
  TNT_CFG( "cost_buy_apartment", def_values_.cost_buy_apartment );
  TNT_CFG( "cost_delete_apartment", def_values_.cost_delete_apartment );
  TNT_CFG( "cost_build_office", def_values_.cost_build_office );
  TNT_CFG( "cost_buy_office", def_values_.cost_buy_office );
  TNT_CFG( "cost_delete_office", def_values_.cost_delete_office );
  TNT_CFG( "radius_box_load_unload_station", def_values_.radius_box_load_unload_station );
  TNT_CFG( "radius_box_use_for_construction", def_values_.radius_box_use_for_construction );
  TNT_CFG( "radius_station_count", def_values_.radius_station_count );
  TNT_CFG( "radius_add_value", def_values_.radius_add_value );
  TNT_CFG( "n_persons_small_house", def_values_.n_persons_small_house );
  TNT_CFG( "n_persons_large_house", def_values_.n_persons_large_house );
  TNT_CFG( "n_jobs_station", def_values_.n_jobs_station );
  TNT_CFG( "boxes_used_for_small_house", def_values_.boxes_used_for_small_house );
  TNT_CFG( "boxes_used_for_large_house", def_values_.boxes_used_for_large_house );
  TNT_CFG( "revenue_train_fix_per_10passengers", def_values_.revenue_train_fix_per_10passengers_ );
  TNT_CFG( "revenue_train_var_per_10passengers_per_10tiles", def_values_.revenue_train_var_per_10passengers_per_10tiles_ );
  TNT_CFG( "cost_train_per_hour", def_values_.cost_train_per_hour_ );
  TNT_CFG( "cost_station_per_hour_less_500", def_values_.cost_station_per_hour_less_500_ );
  TNT_CFG( "cost_station_per_hour_less_2500", def_values_.cost_station_per_hour_less_2500_ );
  TNT_CFG( "cost_station_per_hour_less_10000", def_values_.cost_station_per_hour_less_10000_ );
  TNT_CFG( "cost_station_per_hour_more_10000", def_values_.cost_station_per_hour_more_10000_ );
  TNT_CFG( "max_passengers_train", def_values_.max_passengers_train_ );
  TNT_CFG( "n_jobs_apartment", def_values_.n_jobs_apartment_ );
  TNT_CFG( "max_persons_apartment", def_values_.max_persons_apartment_ );
  TNT_CFG( "min_persons_apartment", def_values_.min_persons_apartment_ );
  TNT_CFG( "max_jobs_office", def_values_.max_jobs_office_ );
  TNT_CFG( "min_jobs_office", def_values_.min_jobs_office_ );
  TNT_CFG( "boxes_used_for_apartm", def_values_.boxes_used_for_apartm_ );
  TNT_CFG( "boxes_used_for_office", def_values_.boxes_used_for_office_ );
  TNT_CFG( "cost_apartm_per_day", def_values_.cost_apartm_per_day_ );
  TNT_CFG( "revenue_apartm_per_10persons_per_day", def_values_.revenue_apartm_per_10persons_per_day_ );
  TNT_CFG( "cost_office_per_day", def_values_.cost_office_per_day_ );
  TNT_CFG( "revenue_office_per_10persons_per_day", def_values_.revenue_office_per_10persons_per_day_ );
  TNT_CFG( "station_passenger_queue_decay", def_values_.station_passenger_queue_decay_ );
  TNT_CFG( "tax_rate_profit", def_values_.tax_rate_profit_ );
  TNT_CFG( "tax_rate_assets", def_values_.tax_rate_assets_ );
  TNT_CFG( "interest_rate_per_million", def_values_.interest_rate_per_million_ );

  pop_config_state();
}

//----------------------------------------------------------------------------

bool WorldMap::can_continue() const
{
  return map_x_ext_>0 && map_y_ext_>0;
}

//----------------------------------------------------------------------------

void WorldMap::change_game_state( GameState new_game_state )
{
  if( game_state_ == new_game_state )
    return;
  else if( game_state_ != NormalSimulation && new_game_state != NormalSimulation )
  {
    change_game_state( NormalSimulation );
    change_game_state( new_game_state );
  }
  else if( game_state_ == NormalSimulation )
  {
    change_menu_state( false, false );
    // enter new state
    game_state_ = new_game_state;
    if(  game_state_ == PlaceFTrain || game_state_ == PlacePTrain
      || stop_if_build()
      )
      game_clock_.stop(true);
  }
  else
  {
    // cancel current state
    game_state_ = NormalSimulation;
    game_clock_.stop(false);
  }
  switch( game_state_ )
  {
  case NormalSimulation:
    bot_state_string_ = "";
    break;
  case BuyLand:
    bot_state_string_ = str_mgr_.get_string("main_buy_land");
    break;
  case SellLand:
    bot_state_string_ = str_mgr_.get_string("main_sell_land");
    break;
  case LayTrack:
    bot_state_string_ = str_mgr_.get_string("main_lay_tracks");
    break;
  case DeleteTrack:
    bot_state_string_ = str_mgr_.get_string("main_delete_tracks");
    break;
  case BuildStation:
    bot_state_string_ = str_mgr_.get_string("main_build_station");
    break;
  case PlaceFTrain:
    bot_state_string_ = str_mgr_.get_string("main_place_f_train");
    break;
  case PlacePTrain:
    bot_state_string_ = str_mgr_.get_string("main_place_p_train");
    break;
  case BuildApartm:
    bot_state_string_ = str_mgr_.get_string("main_build_apa");
    break;
  case BuildOffice:
    bot_state_string_ = str_mgr_.get_string("main_build_off");
    break;
  }
}

void WorldMap::change_menu_state( bool show_details, bool show_minimap )
{
  if( (show_details == menu_show_details_) && (show_minimap == menu_show_minimap_) )
    return;

  menu_show_details_ = show_details;
  menu_show_minimap_ = show_minimap;

  assertCondition( p_main_menu_ );
  assertCondition( p_main_view_ );
  if( show_details || show_minimap )
  {
    p_main_menu_->chg_width_left( def_values_.menu_large_width_ );
    p_main_view_->chg_width_right( SCREEN_W - def_values_.menu_large_width_ );
  }
  else
  {
    p_main_menu_->chg_width_left( def_values_.menu_small_width_ );
    p_main_view_->chg_width_right( SCREEN_W - def_values_.menu_small_width_ );
  }

}

//----------------------------------------------------------------------------

static int passengers_per_hour( int persons, int jobs )
{
  const int min = MIN( persons, jobs );
  const int dif = ABS( persons - jobs );
  return (5*min + 11*dif + 80)/100;
}

void WorldMap::update_hour()
{
  // update stations costs and passenger queue
  for( size_t is=1; is<v_stations_.size(); ++is )
  {
    Station&station = v_stations_[is];
    if( !station.is_valid() )
      continue;
    station.n_miss_passengers_ += station.passenger_queue_.push_value( passengers_per_hour( station.n_persons_env_, station.n_jobs_env_ ), def_values_.station_passenger_queue_decay_ );
    const int avg_passengers = station.n_daily_passengers_counter_.average();
    const int cost
      = avg_passengers <   500 ? def_values_.cost_station_per_hour_less_500_
      : avg_passengers <  2500 ? def_values_.cost_station_per_hour_less_2500_
      : avg_passengers < 10000 ? def_values_.cost_station_per_hour_less_10000_
      : def_values_.cost_station_per_hour_more_10000_;
    cash_.trains_op_.push_cost( cost );
    cash_.total_.push_cost( cost );
    cash_.player_total_ -= cost;
  }
  // update trains
  for( size_t it=0; it<v_trains_.size(); ++it )
  {
    Train&train = v_trains_[it];
    const int cost = def_values_.cost_train_per_hour_;
    train.cost_rev_.push_cost( cost );
    cash_.trains_op_.push_cost( cost );
    cash_.total_.push_cost( cost );
    cash_.player_total_ -= cost;
  }
  // update apartmens
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    const MapTile&mt = map_tiles_[ index(app.center_tile_) ];
    const int cost = def_values_.cost_apartm_per_day_ / 24;
    app.cost_rev_.push_cost( cost );
    const double lvc1 = 1 + ( mt.add_value_ / ( 2 * (double)mt.bas_value_ ) );
    const double lvc = MIN( 2, lvc1 );
    const int rev = ( def_values_.revenue_apartm_per_10persons_per_day_ * mt.n_persons_ * lvc ) / 240;
    app.cost_rev_.push_revenue( rev );
    if( app.player_is_owner_ )
    {
      cash_.apartm_op_.push_cost( cost );
      cash_.total_.push_cost( cost );
      cash_.player_total_ -= cost;
      cash_.apartm_op_.push_revenue( rev );
      cash_.total_.push_revenue( rev );
      cash_.player_total_ += rev;
    }
  }
  // update offices
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    const MapTile&mt = map_tiles_[ index(off.center_tile_) ];
    const int cost = def_values_.cost_office_per_day_ / 24;
    off.cost_rev_.push_cost(cost);
    const double lvc1 = 1 + ( mt.add_value_ / ( 2 * (double)mt.bas_value_ ) );
    const double lvc = MIN( 2, lvc1 );
    const int rev = ( def_values_.revenue_office_per_10persons_per_day_ * mt.n_jobs_ * lvc ) / 240;
    off.cost_rev_.push_revenue(rev);
    if( off.player_is_owner_ )
    {
      cash_.office_op_.push_cost( cost );
      cash_.total_.push_cost( cost );
      cash_.player_total_ -= cost;
      cash_.office_op_.push_revenue( rev );
      cash_.total_.push_revenue( rev );
      cash_.player_total_ += rev;
    }
  }
  // update interests and trigger forced sale
  {
    if( cash_.player_total_ < 0 )
    {
      const int interest = ( - cash_.player_total_ * (double)def_values_.interest_rate_per_million_ ) / 1000000;
      cash_.interest_.push_cost( interest );
      cash_.total_.push_cost( interest );
      cash_.player_total_ -= interest;
    }
    const int total_asset_value = assets_.total();
    const int debt_limit = total_asset_value / 2;
    if( assets_.all_inhabitants_ > 0 && cash_.player_total_ < -debt_limit )
    {
      const bool rc = force_sale();
      if( !rc )
      {
        allegro_message( "You have lost the game because you're in debt and do not have any more assets to sell..." );
        simulate_keypress( KEY_F10 << 8 );
      }
      else
      {
        allegro_message( "Some of your assets have been sold by your bank because you're in debt" );
      }
    }
  }
}

void WorldMap::update_day()
{
  cash_.update_day();
  // update stations counts
  for( size_t is=0; is<v_stations_.size(); ++is )
  {
    Station&station = v_stations_[is];
    if( !station.is_valid() )
      continue;
    const int passengers_count = MAX( 0, station.n_good_passengers_ - station.n_miss_passengers_ * 2 );
    station.n_daily_passengers_counter_.push_value( passengers_count );
    station.n_good_passengers_ = 0;
    station.n_miss_passengers_ = 0;
  }
  // update trains
  for( size_t it=0; it<v_trains_.size(); ++it )
  {
    Train&train = v_trains_[it];
    train.cost_rev_.update_day();
  }
  // update apartmens
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    // change value
    app.value_ += app.cost_rev_.cur_profit_loss() / 2;
    // limit value
    if( app.value_ < def_values_.cost_build_apartment )
      app.value_ = def_values_.cost_build_apartment;
    else if( app.value_ > 2 * def_values_.cost_build_apartment )
      app.value_ = 2 * def_values_.cost_build_apartment;
    // update account
    app.cost_rev_.update_day();
  }
  // update offices
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    // change value
    off.value_ += off.cost_rev_.cur_profit_loss() / 3;
    // limit value
    if( off.value_ < def_values_.cost_build_office )
      off.value_ = def_values_.cost_build_office;
    else if( off.value_ > 2 * def_values_.cost_build_office )
      off.value_ = 2 * def_values_.cost_build_office;
    // update account
    off.cost_rev_.update_day();
  }
}

void WorldMap::update_month()
{
  cash_.update_month();
  // update trains
  for( size_t it=0; it<v_trains_.size(); ++it )
  {
    Train&train = v_trains_[it];
    train.cost_rev_.update_month();
  }
  // update apartmens
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    app.cost_rev_.update_month();
  }
  // update offices
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    off.cost_rev_.update_month();
  }
}

void WorldMap::update_year()
{
  const int tax_asset = assets_.taxes_estimate( def_values_.tax_rate_assets_ );
  const int tax_cash  = cash_.taxes_estimate_ytd( def_values_.tax_rate_profit_ );
  cash_.update_year( tax_asset + tax_cash );
  // update trains
  for( size_t it=0; it<v_trains_.size(); ++it )
  {
    Train&train = v_trains_[it];
    train.cost_rev_.update_year();
  }
  // update apartmens
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    app.cost_rev_.update_year();
  }
  // update offices
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    off.cost_rev_.update_year();
  }
}

//----------------------------------------------------------------------------

void WorldMap::update_build_decision()
{
  if( v_stations_.size() > 1 )
  {
    for( int i=0; i<10; ++i )
    {
      const size_t rand_station_id = 1 + rng.IRand( v_stations_.size()-1 );
      assertCondition( rand_station_id < v_stations_.size() );
      assertCondition( rand_station_id > 0 );
      const Station&station = v_stations_[rand_station_id];
      if( !station.is_valid() )
        continue;
      const int daily_pass_average = station.n_daily_passengers_counter_.average();
      if( daily_pass_average < 10 )
        continue;
      const size_t nb_size = neighbourhood_.get_nb_le_size( def_values_.radius_station_count );
      const size_t rand_tile_id = rng.IRand(nb_size);
      const TileKOO th = station.center_tile_ + neighbourhood_.get_nb_le( def_values_.radius_station_count, rand_tile_id );
      if( !is_on_map(th) )
        break;
      const MapTile&mt = map_tiles_[ index(th) ];
      if( mt.player_is_owner_ || mt.is_rail() || mt.station_id_>=0 )
        continue;
      const int decicion_rnd = rng.IRand(4000);
      if( decicion_rnd > 200 + daily_pass_average )
      {
        // delete or shrink some house
        if( mt.apartment_id_ >= 0 )
        {
          if( mt.n_persons_==def_values_.min_persons_apartment_ )
            delete_apa(th);
        }
        else if( mt.office_id_ >= 0 )
        {
          const Office&off = v_offices_[ mt.office_id_ ];
          if( mt.n_persons_==def_values_.min_jobs_office_ )
            delete_off(th);
        }
        else if( mt.n_persons_ > 0 || mt.n_jobs_ > 0 )
          set_general_tile( th, IDXgras, SUB_IDX_RANDOM, mt.player_can_buy_, mt.player_is_owner_ );
        else if( mt.can_hold_boxes_ && mt.n_boxes_==0 )
          set_general_tile( th, IDXgras, SUB_IDX_RANDOM, mt.player_can_buy_, mt.player_is_owner_ );
      }
      else if( decicion_rnd < daily_pass_average )
      {
        build_large_house(th);
      }
      else
      {
        build_small_house(th);
      }
      break;
    }
  }
}

void WorldMap::update_value_and_assets()
{
  assets_.clear();
  // count persons and jobs in environment of station
  if( def_values_.radius_station_count > 0 )
  {
    for( size_t is=0; is<v_stations_.size(); ++is )
    {
      Station&station = v_stations_[is];
      if( !station.is_valid() )
        continue;
      station.n_persons_env_ = 0;
      station.n_jobs_env_ = def_values_.n_jobs_station;
      for( size_t ir=0; ir<def_values_.radius_station_count; ++ir )
      {
        const size_t nb_size = neighbourhood_.get_nb_eq_size( ir );
        for( size_t it=0; it<nb_size; ++it )
        {
          const TileKOO&dt = neighbourhood_.get_nb_eq( ir, it );
          const TileKOO tnb = station.center_tile_ + dt;
          if( !is_on_map(tnb) )
            continue;
          const MapTile&mt = map_tiles_[ index(tnb) ];
          if( mt.station_id_ > 0 )
            continue;
          if( mt.nearest_station_id_ > 0 && mt.nearest_station_id_ != station.station_id_ )
            continue;
          station.n_persons_env_ += mt.n_persons_;
          station.n_jobs_env_ += mt.n_jobs_;
        }
      }
    }
  }
  // set statistic values to 0 and count my buildings
  {
    for( size_t iat=0; iat<v_apartments_.size(); ++iat )
    {
      const Apartment&app = v_apartments_[iat];
      if( !app.is_valid() )
        continue;
      if( app.player_is_owner_ )
      {
        ++assets_.my_apartments_;
        assets_.sum_apartments_value_ += app.value_;
      }
    }
    for( size_t iot=0; iot<v_offices_.size(); ++iot )
    {
      const Office&off = v_offices_[iot];
      if( !off.is_valid() )
        continue;
      if( off.player_is_owner_ )
      {
        ++assets_.my_offices_;
        assets_.sum_office_value_ += off.value_;
      }
    }
  }
  // set add_value to 0 and count on map
  {
    TileKOO t;
    for( t.x=0; t.x<map_x_ext_; ++t.x )
    {
      for( t.y=0; t.y<map_y_ext_; ++t.y )
      {
        MapTile&mt = map_tiles_[ index(t) ];
        mt.tmp_value_ = mt.add_value_;
        mt.add_value_ = 0;
        if( mt.player_is_owner_ )
          ++assets_.my_land_tiles_;
        if( mt.player_is_owner_ && mt.is_rail() )
          ++assets_.my_rail_tiles_;
        assets_.all_inhabitants_ += mt.n_persons_;
        if( mt.nearest_station_id_ < 0 )
          continue;
        if( mt.apartment_id_ >= 0 || ( !mt.is_building_ && mt.n_persons_ > 0 ) )
          ++assets_.all_person_tiles_;
        else if( mt.office_id_ >= 0 )
          ++assets_.all_office_tiles_;
        else if( mt.is_rail() || mt.station_id_ >= 0 )
          ++assets_.all_rail_tiles_;
        else
          ++assets_.all_land_tiles_;
      }
    }
  }
  // compute new add_value
  if( def_values_.radius_add_value > 0 )
  {
    for( size_t is=0; is<v_stations_.size(); ++is )
    {
      const Station&station = v_stations_[is];
      if( !station.is_valid() )
        continue;
      if( is > 0 )
        assets_.sum_train_value_ += def_values_.cost_build_station;
      const int station_value = station.n_daily_passengers_counter_.average()*3;
      for( size_t ir=0; ir<def_values_.radius_add_value; ++ir )
      {
        const size_t nb_size = neighbourhood_.get_nb_eq_size( ir );
        for( size_t it=0; it<nb_size; ++it )
        {
          const TileKOO&dt = neighbourhood_.get_nb_eq( ir, it );
          const TileKOO tnb = station.center_tile_ + dt;
          if( !is_on_map(tnb) )
            continue;
          MapTile&mt = map_tiles_[ index(tnb) ];
          mt.add_value_ += station_value / (double)(5+ir);
        }
      }
    }
  }
  // adjust old value to new value, count and sum all player assets
  {
    TileKOO t;
    for( t.x=0; t.x<map_x_ext_; ++t.x )
    {
      for( t.y=0; t.y<map_y_ext_; ++t.y )
      {
        MapTile&mtxy = map_tiles_[ index(t) ];
        mtxy.add_value_ =  mtxy.add_value_ * 0.02 + mtxy.tmp_value_ * 0.98;
        // assets
        if( !mtxy.player_is_owner_ )
          continue;
        assets_.sum_land_value_ += mtxy.value();
        if( mtxy.is_rail() && !mtxy.is_station_building() )
          assets_.sum_train_value_ += def_values_.cost_lay_track;
      }
    }
  }
  // add value for stations
  for( size_t is=1; is<v_stations_.size(); ++is )
  {
    const Station&station = v_stations_[is];
    if( !station.is_valid() )
      continue;
    assets_.sum_train_value_ += def_values_.cost_build_station;
  }
  // add value for trains
  for( size_t iat=0; iat<v_trains_.size(); ++iat )
  {
    const Train&train = v_trains_[iat];
    if( !train.is_valid() )
      continue;
    assets_.sum_train_value_ += def_values_.cost_place_train;
  }
}

//----------------------------------------------------------------------------

void WorldMap::advance_game_time()
{
  if( !can_continue() )
    return;

  const int cur_game_time = game_clock_.get_time_since0();
  const int cur_hour = game_clock_.get_hour_since0();
  const int cur_day = game_clock_.get_days_since0();
  const int cur_mon = game_clock_.get_month_of_year();
  const int cur_year = game_clock_.get_years_since0();
  if( current_day_since0_ < 0 )
  {
    current_hour_since0_ = cur_hour;
    current_day_since0_ = cur_day;
    current_month_ = cur_mon;
    current_year_since0_ = cur_year;
  }

  while(true)
  {

    int count_updates = 0;

    // new hour: update hourly
    if( cur_hour != current_hour_since0_ )
    {
      update_hour();
      current_hour_since0_ = cur_hour;
      ++count_updates;
    }
    // new day: compute dayly balance sheet and update dayly counts
    if( cur_day != current_day_since0_ )
    {
      update_day();
      current_day_since0_ = cur_day;
      ++count_updates;
    }
    // new month: compute monthly balance sheet and update monthly counts
    if( cur_mon != current_month_ )
    {
      update_month();
      current_month_ = cur_mon;
      ++count_updates;
    }
    // new year: compute complete balance and taxes
    if( cur_year != current_year_since0_ )
    {
      update_year();
      current_year_since0_ = cur_year;
      ++count_updates;
    }
    // 1st of april: pay taxes
    if( cash_.taxes_to_pay_next_april_ > 0 && cur_mon == 4 )
    {
      cash_.pay_taxes();
      ++ count_updates;
    }
    // move trains
    count_updates += advance_ext_train( cur_game_time );
    {
      for( size_t it=0; it<v_trains_.size(); ++it )
      {
        Train&train = v_trains_[it];
        if( !train.is_valid() )
          continue;
        if( train.is_freight_train() )
          count_updates += advance_f_train( train, cur_game_time );
        else
          count_updates += advance_p_train( train, cur_game_time );
      }
    }
    // decide to build houses
    const int time_diff_build_decision = 32 * GameClock::ticks_per_game_hour;
    if( cur_game_time - time_diff_build_decision > last_build_decision_time_ )
    {
      update_build_decision();
      last_build_decision_time_ += time_diff_build_decision;
      ++count_updates;
    }
    // update persons in apartments and jobs in offices
    const int time_diff_update_buildings = 19 * GameClock::ticks_per_game_hour;
    if( cur_game_time - time_diff_update_buildings > last_update_buildings_time_ )
    {
      update_apartments();
      update_offices();
      last_update_buildings_time_ += time_diff_update_buildings;
      ++count_updates;
    }
    // update map value
    const int time_diff_update_value = 2 * GameClock::ticks_per_game_hour;
    if( cur_game_time - time_diff_update_value > last_value_update_time_ )
    {
      remap_all_stations_buildings();
      update_value_and_assets();
      last_value_update_time_ += time_diff_update_value;
      ++count_updates;
    }

    if( count_updates == 0 )
      break;

  } // while(true)

}

//----------------------------------------------------------------------------

bool WorldMap::force_sale()
{
  std::set<int> apa_in_debt_;
  std::set<int> all_apa_;
  std::set<int> off_in_debt_;
  std::set<int> all_off_;
  std::set<int> trains_in_debt_;
  std::set<int> all_trains_;
  int count_objects_in_debt_ = 0;
  int count_all_objects_ = 0;
  // collect apartmens
  for( size_t iat=0; iat<v_apartments_.size(); ++iat )
  {
    const Apartment&app = v_apartments_[iat];
    if( !app.is_valid() )
      continue;
    if( !app.player_is_owner_ )
      continue;
    if( app.cost_rev_.mtd_profit_loss() < 0 )
    {
      apa_in_debt_.insert( iat );
      ++count_objects_in_debt_;
    }
    all_apa_.insert( iat );
    ++count_all_objects_;
  }
  // collect offices
  for( size_t iot=0; iot<v_offices_.size(); ++iot )
  {
    const Office&off = v_offices_[iot];
    if( !off.is_valid() )
      continue;
    if( !off.player_is_owner_ )
      continue;
    if( off.cost_rev_.mtd_profit_loss() < 0 )
    {
      off_in_debt_.insert( iot );
      ++count_objects_in_debt_;
    }
    all_off_.insert( iot );
    ++count_all_objects_;
  }
  // collect trains
  for( size_t it=0; it<v_trains_.size(); ++it )
  {
    const Train&train = v_trains_[it];
    if( !train.is_valid() )
      continue;
    if( train.cost_rev_.mtd_profit_loss() < 0 )
    {
      trains_in_debt_.insert( it );
      ++count_objects_in_debt_;
    }
    all_trains_.insert( it );
    ++count_all_objects_;
  }
  if( count_objects_in_debt_ > 0 )
  {
    if( off_in_debt_.size() > 0 )
    {
      const int iot = *off_in_debt_.begin();
      const Office&off = v_offices_[ iot ];
      if( sell_off( off.center_tile_ ) )
        return true;
    }
    if( apa_in_debt_.size() > 0 )
    {
      const int iat = *apa_in_debt_.begin();
      const Apartment&apa = v_apartments_[ iat ];
      if( sell_apa( apa.center_tile_ ) )
        return true;
    }
    if( trains_in_debt_.size() > 0 )
    {
      const int itt = *trains_in_debt_.begin();
      remove_train( itt );
      return true;
    }
  }
  return false;
}

//----------------------------------------------------------------------------
