/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_GAMELOOP
#include "GameLoop.h"
#endif
#ifndef INC_SYSTEMDIALOG
#include "SystemDialog.h"
#endif

#include <allegro.h>

#include <crtdbg.h>

//----------------------------------------------------------------------------

void fatalError( const char *str )
{
  allegro_message(str);
  _ASSERT(false);
  exit(0);
}

void fatalError( const char*msg, const char*fname, int line_no )
{
  std::string msg2;
  msg2 = msg;
  msg2 += " ";
  msg2 += fname;
  msg2 += " ";
  char buffer[20];
  msg2 += itoa( line_no, buffer, 10 );
  fatalError( msg2.c_str() );
}

//----------------------------------------------------------------------------

namespace
{

volatile int timer_counter = 0;

void timerCounterUpdater()
{
  ++timer_counter;
}

volatile int fps_counter = 0;

void fpsCounterUpdater()
{
  ++fps_counter;
}

volatile int perf_counter = 0;

void perfCounterUpdater()
{
  ++perf_counter;
}

//----------------------------------------------------------------------------

struct FPSInfo
{
  int logic;
  int displayed;
  int iteration;
  int advance_cputime;
  int display_cputime;
  int show_cputime;
  int perf_counter_;
};


//----------------------------------------------------------------------------

// frame counter
FPSInfo cur        = { 0, 0, 0, 0, 0, 0, 0 };
FPSInfo displayed  = { 0, 0, 0, 0, 0, 0, 0 };

//----------------------------------------------------------------------------

} // namespace

//----------------------------------------------------------------------------

GameLoop::GameLoop( DisplayEngine&ade, SystemDialog&asys )
: display_engine(ade)
, sys_dialog(asys)
, max_skip(2)
, rest_dur_ms(1)
, normal_fps(30)
{
}

void GameLoop::add_view_port( ViewPortIfc&vp )
{
  v_viewports.push_back( &vp );
}

void GameLoop::add_scroll_handler( ScrollingIfc&sc )
{
  v_scroll_hndls.push_back( &sc );
}

void GameLoop::add_kbd_handler( KbdHndlIfc&kh )
{
  v_kbd_hndls.push_back( &kh );
}

void GameLoop::add_gcl_handler( GameClockIfc&gc )
{
  v_game_clock_hndls.push_back( &gc );
}

//----------------------------------------------------------------------------

void GameLoop::config( const char*ini_fn )
{
  // reset timers
  timer_counter = 0;
  fps_counter   = 0;
  perf_counter  = 0;

  assertCondition( ini_fn );

  push_config_state();
  set_config_file(ini_fn);
  // init timer
  max_skip = get_config_int( "timer", "max_skip", max_skip );
  rest_dur_ms = get_config_int( "timer", "rest_dur_ms", rest_dur_ms );
  normal_fps = get_config_int( "timer", "normal_fps", normal_fps );
  if( normal_fps > 0 )
  {
    /* Timer related functions and vars need to be locked */
    LOCK_FUNCTION(timerCounterUpdater);
    LOCK_VARIABLE(timer_counter);
    const int rc = install_int_ex( timerCounterUpdater, BPS_TO_TIMER( normal_fps ) );
    assertCondition(rc==0);
  }
  {
    LOCK_FUNCTION(fpsCounterUpdater);
    LOCK_VARIABLE(fps_counter);
    const int rc = install_int_ex( fpsCounterUpdater, SECS_TO_TIMER(1) );
    assertCondition(rc==0);
  }
  {
    LOCK_FUNCTION(perfCounterUpdater);
    LOCK_VARIABLE(perf_counter);
    const int rc = install_int_ex( perfCounterUpdater, BPS_TO_TIMER(2000) );
    assertCondition(rc==0);
  }
  pop_config_state();
}

//----------------------------------------------------------------------------

void GameLoop::loop()
{
  sys_dialog.initial_dialog();
  while(true)
  {
    bool needsRefresh = false;
    cur_skip = 0;
    while( timer_counter > 0 || normal_fps==0 )
    {
      cur.advance_cputime -= perf_counter;
      // handle user input
      bool have_scrolled = false;
      if( v_scroll_hndls.size() > 0 )
      {
        for( std::vector<ScrollingIfc*>::iterator it = v_scroll_hndls.begin(); it != v_scroll_hndls.end(); ++it )
        {
          have_scrolled = (*it)->handle_scroll() || have_scrolled;
        }
      }
      if( ( !have_scrolled && keypressed() ) )
      {
        const int scan_code = readkey() >> 8;
        if( scan_code == KEY_ESC )
        {
          sys_dialog.quit();
        }
        if( scan_code == KEY_F10 )
          sys_dialog.ingame_dialog();
        const int shift_flags = key_shifts & ( KB_SHIFT_FLAG | KB_CTRL_FLAG | KB_ALT_FLAG );
        for( std::vector<KbdHndlIfc*>::iterator it = v_kbd_hndls.begin(); it != v_kbd_hndls.end(); ++it )
        {
          (*it)->handle_kbd( scan_code, shift_flags );
        }
      }

      // Book keeping
      ++cur.logic;
      if( normal_fps > 0 )
        --timer_counter;
      ++cur_skip;
      {
        for( std::vector<GameClockIfc*>::iterator it = v_game_clock_hndls.begin(); it != v_game_clock_hndls.end(); ++it )
        {
          (*it)->advance_game_time();
        }
      }
      cur.advance_cputime += perf_counter;
      if( cur_skip >= max_skip )
      {
        timer_counter = 0;
        break;
      }
      needsRefresh = true;
    }
    if( needsRefresh )
    {
      cur.display_cputime -= perf_counter;
      // draw backbuffer
      BITMAP*const draw_buffer = display_engine.draw_buffer();
      display_engine.prepare_draw_buffer();
      {
        for( std::vector<ViewPortIfc*>::iterator it = v_viewports.begin(); it != v_viewports.end(); ++it )
        {
          (*it)->display( draw_buffer );
        }
      }
      cur.display_cputime += perf_counter;

      // display backbuffer
      needsRefresh = false;
      cur.show_cputime -= perf_counter;
      display_engine.display_draw_buffer();
      cur.show_cputime += perf_counter;
      cur.displayed++;
    }
    else
    {
      if( rest_dur_ms > 0 )
        rest( rest_dur_ms );
    }
    cur.iteration++;
    if( fps_counter > 0 )
    {
      displayed  = cur;
      displayed.perf_counter_ = perf_counter;
      cur.logic  = cur.iteration = cur.displayed = 0;
      cur.advance_cputime = cur.display_cputime = cur.show_cputime = 0;
      fps_counter = 0;
			perf_counter = 0;
    }
  }
}

//----------------------------------------------------------------------------

GameLoopDebugView::GameLoopDebugView( const ScreenRect&asr, const DisplayEngine&ade, const GameLoop&agl )
: ViewPortBase( asr )
, display_engine(ade)
, game_loop(agl)
, what_to_show_(0)
{
}

void GameLoopDebugView::display( BITMAP*const db )
{
  rectfill( db, area.x, area.y, area.x+area.w-1, area.y+area.h-1, makecol(200,200,0) );
  rect(     db, area.x, area.y, area.x+area.w-1, area.y+area.h-1, makecol(0,0,255) );
  if( what_to_show_ == 0 )
    return;

  if( displayed.perf_counter_ > 0 )
  {
    textprintf_ex( db, font, area.x+3, area.y+3, makecol(0,0,0), -1
      , "FPS: %d-%d-%d / CPU: %d%%-%d%%-%d%% %d%% %d"
      , displayed.logic
      , displayed.displayed
      , displayed.iteration
      , (displayed.advance_cputime*100)/displayed.perf_counter_
      , (displayed.display_cputime*100)/displayed.perf_counter_
      , (displayed.show_cputime*100)/displayed.perf_counter_
      , ((displayed.advance_cputime+displayed.display_cputime+displayed.show_cputime)*100)/displayed.perf_counter_
      , display_engine.getCurrentUpdateMethod()
      );
  }
}

void GameLoopDebugView::handle_kbd( int scan_code, int shift_flags )
{
  switch( scan_code )
  {
  case KEY_G:
    what_to_show_ = what_to_show_ > 0 ? 0 : 1;
    break;
  }
}

//----------------------------------------------------------------------------

